/*

    Copyright (C) 2002,2003  John Darrington 

    This program is free software; you can redistibute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
const static char RCSID[]="$Id: holiday.c,v 1.18 2003/06/04 00:26:13 john Exp $";

#define _XOPEN_SOURCE
#define _XOPEN_SOURCE_EXTENDED

#include "config.h"

#include "event.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <assert.h>
#include "common.h"
#include "pom.h"
#include "holiday.h"
#include <limits.h>

int match(struct dt date, union event *e, int *);

struct dt prev( const struct dt *date);

time_t maketime(struct tm *timeptr) ;


const char *
isholiday(const struct dt *date, union event *holidays[], int we)
{

  const static char unknown_holiday[]="unspecified public holiday";

  int h;
  int i=0;
  union event *ev = 0 ;
 

  while ( (ev = holidays[i++]) ) {
    if ( ev->eh.flag ) { 
      /* if we is not set, then skip week-ends */
      if ( !we && ( ev->eh.flag == 2 ) ) 
	continue;
     h=match(*date, ev,0);
     if ( h ) 
       break;
    }
  }

  if ( h ) {
    if ( ev->eh.name ) 
      return ev->eh.name;
    else
      return unknown_holiday;
  }
  else
    return 0;

}


/* return 1 if date matches *e.  0 otherwise */
/* if c is non NULL, then after this function returns, it will point to 
   an integer containing the number of days between constraint and 
   constrainee */
int
match(struct dt date, union event *e, int *distance)
{
  switch (e->eh.ty) { 
  case E_CALENDAR:
    {
    struct calendarEvent  *ce = (struct calendarEvent *) e ;

    /* Check that each field matches, or is wild */

    if ( ce->dom != date.dt_mday && ( ce->dom != -1 ) ) 
      return 0;

    if ( ce->month != date.dt_mon && ce->month != -1 ) 
      return 0;

    if ( ce->year != date.dt_year && ( ce->year != 0 ) ) 
      return 0;

    return 1;
    }
    break;
  case E_DOW:
    {
    struct dayEvent  *de = (struct dayEvent *) e ;

    if ( de->dow & get_day_of_week(date) ) 
      return 1;
    else 
      return 0;
    }
    break;
  case E_COMPOUND:
    {
      struct compoundEvent  *comp = (struct compoundEvent *) e ;

      if ( distance ) *distance = 0;
      if ( comp->cstr->ordinal == 0 ) {
	switch (comp->cstr->dir){
	case NOT:
	  {
	    if ( ! match ( date, (union event *) (comp->constrainee),0 ))
	     return 0;
	    if ( match ( date, (union event *) (comp->constrainant),0 ))
	      return 0;
	    else
	      return 1;
	  } 
	  break;
	case NEAREST:
	  {
	    int d1, d2;
	    int m1, m2;
	    struct constraint fooa;
	    struct constraint foob;
	    struct compoundEvent ce;

	    ce.eh.ty = E_COMPOUND;
	    ce.eh.name = 0;
	    ce.eh.flag = 0;
	    ce.constrainant = comp->constrainant; 
	    ce.constrainee  = comp->constrainee;

	    fooa.dir=ON_OR_AFTER;
	    fooa.ordinal=1;
	    foob.dir=ON_OR_BEFORE;
	    foob.ordinal=1;

	    if ( ! match ( date, (union event *) (comp->constrainant),distance )) 
	      return 0;


	    /* find first on or after */
	    ce.cstr=&fooa;
	    m1 = match ( date, (union event *) &ce ,&d1 );

	    /* find first on or before */
	    ce.cstr=&foob;
	    m2 = match ( date, (union event *) &ce ,&d2 );

	    /* pick the closest */
	    if ( m1 )  { 
	      d2 = d2 - d1;
	    }
	    if ( m2 ) {
	      d1 = d1 - d2;
	    }

	    if ( m1  && d1 <= d2  ) 
	      return 1;

	    if ( m2  && d2 <= d1  ) 
	      return 1;


	    return 0;

	  }
	  break;
	default:
	  { 
	    if ( ! match ( date, (union event *) (comp->constrainee),0 ))
	     return 0;

	    if ( ! match ( date, (union event *) (comp->constrainant),0 )) 
	      return 0;
	    else {
	      return 1;
	    }
	  }
	  break;
	}
      }   
      else {
	struct dt temp_date = date;
	int count=0;
	if ( ! match ( date, (union event *) (comp->constrainant),0 ) )
	{ 
	  return 0;
	}
	else {
	  ++count;
	}
	if (  ( comp->cstr->dir == ON_OR_AFTER || 
	      comp->cstr->dir == ON_OR_BEFORE ) && 
	      ( comp->cstr->ordinal == count) ) { 
	  if ( match ( date, (union event *) (comp->constrainee) ,0 )) {
	    return 1;
	  }
	}
	while ( count <= comp->cstr->ordinal ) { 
	  struct dt prev_date ;
	  if (    comp->cstr->dir == AFTER 
		  || comp->cstr->dir == ON_OR_AFTER) { 
	    prev_date = prev(&temp_date);
	    if ( distance ) ++(*distance);
	  }
	  else if (    comp->cstr->dir == BEFORE 
		       || comp->cstr->dir == ON_OR_BEFORE) {
	    prev_date = next(&temp_date);
	    if ( distance ) ++(*distance);
	  }
	  else { 
	    assert(0);
	  }
	  if ( comp->cstr->dir == ON_OR_BEFORE
	       || comp->cstr->dir == ON_OR_AFTER ) {
	   if ( match ( prev_date, (union event *) (comp->constrainant),0 )) 
	     ++count;	
	  }
	  if ( match ( prev_date, (union event *) (comp->constrainee), 0 )) 
	    break;
	  if ( comp->cstr->dir == BEFORE
	       || comp->cstr->dir == AFTER ) {
	    if ( match ( prev_date, (union event *) (comp->constrainant), 0 )) 
	      ++count;	
	  }
	  temp_date = prev_date ;
	}
	if ( count == comp->cstr->ordinal ) {
	  return 1;
	}
	else 
	  return 0;
      }
    }
    break;
  case E_LUNAR:
    {

      struct tm pTime ;
      time_t tmt;
      double per;
      struct lunarEvent  *le = (struct lunarEvent *) e ;
      
      pTime.tm_mday = date.dt_mday;
      pTime.tm_mon  = date.dt_mon;
      pTime.tm_year = date.dt_year - 1900 ;
      pTime.tm_sec = 0;
      pTime.tm_min = 0;  
      pTime.tm_hour = 12;
      pTime.tm_isdst = 0;

      tmt = maketime(&pTime);
      assert ( tmt != -1);
      

      per = potm(tmt);

      if ( (int) per == le->percent_full) { 
	return 1;
      }
    }
    break;
  default:
    assert(0);
    break;
  }

  return 0;
}



int 
get_day_of_week( struct dt date)
{
  struct tm pTime ;
  pTime.tm_mday = date.dt_mday;
  pTime.tm_mon  = date.dt_mon;
  pTime.tm_year = date.dt_year - 1900 ;
  pTime.tm_sec = 0;
  pTime.tm_min = 0;  
  pTime.tm_hour = 0;
  pTime.tm_isdst = 0;
  pTime.tm_wday = 0;
  pTime.tm_yday = 0;

  if ( (time_t) -1 == maketime (&pTime) ) {
    fprintf(stderr,"Date out of range\n");
    exit (1);
  }

  switch ( pTime.tm_wday ) { 
  case 0: return SUN  ;  break;
  case 1: return MON  ;  break;
  case 2: return TUE  ;  break;
  case 3: return WED  ;  break;
  case 4: return THU  ;  break;
  case 5: return FRI  ;  break;
  case 6: return SAT  ;  break;
  default: assert (0) ;  break;
  }

  return -1;
}



static int days_in_month[] = 
{
/*J  F  M  A  M  J  J  A  S  O  N  D  */
  31,28,31,30,31,30,31,31,30,31,30,31
};

int
isLeapYear(int year)
{
  assert  ( year > 0 ) ;

  if (!( year % 400 ) )
    return 1;

  if ( !(year % 4) && ( year % 100 ))
    return 1;

  return 0;
  
}

struct tm
noon(const struct dt *date)
{
  struct tm result ;
  result.tm_year = date->dt_year - 1900;
  result.tm_mon  = date->dt_mon;
  result.tm_mday = date->dt_mday;
  result.tm_sec  = 0 ;
  result.tm_min  = 0 ;
  result.tm_hour = 12;

  assert( -1 != maketime(&result));

  return result;
}


void
print_date(FILE* fp, const struct dt *date, const char *fmt)
{

  char string[100];
  struct tm midday ;

  midday = noon(date);

  if ( 0 < strftime(string, 100, fmt, &midday) ) 
    fputs(string,fp);
}




struct dt 
next( const struct dt *date)
{
  int days_this_month;
  struct dt result = *date;

  result.dt_mday++;
  
  days_this_month = days_in_month[result.dt_mon] ;
  if ( date->dt_mon == 1 && isLeapYear(date->dt_year)) {
    days_this_month++;
  }

  if ( result.dt_mday  > days_this_month ) { 
    result.dt_mon++;
    if ( result.dt_mon > 11 ) { 
      result.dt_mon = 0;
      result.dt_year++;
    }
    result.dt_mday=1;
  }
    

  return result;
}





struct dt 
prev( const struct dt *date)
{
  int days_last_month;
  struct dt result = *date;

  --result.dt_mday;

  if ( result.dt_mday <=0 ) { 
    --result.dt_mon;
    if ( result.dt_mon < 0 ) {
      result.dt_mon = 11;    
      --result.dt_year ;
    }

    days_last_month = days_in_month[result.dt_mon] ;
    if ( result.dt_mon == 1 && isLeapYear(date->dt_year)) {
      days_last_month++;
    }
    result.dt_mday = days_last_month;
  }
    

  return result;
}




struct dt
date_now()
{
  struct dt result;
  time_t nowt = time(0);
  struct tm *now = localtime(&nowt);

  result.dt_year = now->tm_year + 1900;
  result.dt_mon = now->tm_mon ;
  result.dt_mday = now->tm_mday;
  

  assert ( date_is_valid( result));

  return result;
}


int 
year_now()
{
  time_t nowt = time(0);
  struct tm *now = localtime(&nowt);

  return ( now->tm_year + 1900 );
}


int
year_from_string(const char *s)
{
  int year=0;
  
  year = atoi(s);
  if ( 0 >= year ) {
    fprintf(stderr,_("Invalid year: %s\n"), s);
    exit (1);
  }

  return year;
}

/* parse date in the form [YYYY]mmdd or let getdate do it*/
struct dt
date_from_string(const char *s)
{
  struct dt result;
  char *datemsk = 0;

  char shortdate[9];
  char temp[9];
  int len = strlen(s);
  
  result.dt_year=0;

#ifdef HAVE_GETDATE
  /* If DATEMSK is set, try using getdate */
  datemsk = getenv("DATEMSK");
  if ( datemsk && 0!=strcmp("",datemsk) ) {
    struct tm *t = 0;
    t = getdate(s);
    if (t) { 
      result.dt_year = t->tm_year + 1900;
      result.dt_mon =  t->tm_mon ;
      result.dt_mday = t->tm_mday;
      
      if ( date_is_valid(result))
	return result;
    }
    /* If that fails then try the normal method ... */
  }
#endif

  switch (len) {
  case 8: /* eight letters long. Get Year, Month, Date */
    {
      strcpy(temp,s);
      temp[4]='\0';
      result.dt_year = atoi(temp);
      strcpy(shortdate,s + 4);
    }
    break;
  case 4: /* four letters long.  Get the month and date. Assume this year */
    {
      time_t nowt = time(0);
      struct tm *now = localtime(&nowt);
      result.dt_year = now->tm_year + 1900 ;
      strcpy(shortdate,s);
    }

    break;
  default:
    fprintf(stderr,_("Invalid date format: %s\n"), s);
    exit (1);
  }



  result.dt_mday = atoi(shortdate+2);
  strcpy(temp,shortdate);
  temp[2]='\0';
  result.dt_mon = atoi(temp)-1;



  if ( ! date_is_valid(result)) {
    fprintf(stderr,_("Invalid date format: %s\n"),s);
    exit(1);
  }
  
  return result;
}



/* return 1 if date is a valid date. 0 otherwise */
int 
date_is_valid(struct dt date)
{

  int days_this_month;

  struct tm pTime ;
  pTime.tm_mday = date.dt_mday;
  pTime.tm_mon  = date.dt_mon;
  pTime.tm_year = date.dt_year - 1900 ;
  pTime.tm_sec = 0;
  pTime.tm_min = 0;  
  pTime.tm_hour = 0;

  if ( (time_t) -1 == maketime (&pTime) ) {
    return 0;
  }

  if ( date.dt_mon > 11 ) return 0;
  if ( date.dt_mon < 0  ) return 0;
  if ( date.dt_mday < 0  ) return 0;

  days_this_month = days_in_month[date.dt_mon];

  if ( isLeapYear(date.dt_year) && date.dt_mon == 1) {
    ++days_this_month;
  }

  if ( date.dt_mday > days_this_month ) return 0;
   


  return 1;
}




time_t 
maketime(struct tm *timeptr)
{
  time_t result ; 

  result = mktime(timeptr);

  if ( -1 == result ) { 
    const time_t maximum = LONG_MAX;
    const time_t minimum = 0;

    struct dt date;
    struct dt date1;
    struct dt date2;
    struct tm *time;

    time = gmtime(&maximum);

    date.dt_mday = time->tm_mday;
    date.dt_mon = time->tm_mon;
    date.dt_year = time->tm_year +1900 ;

    if ( time->tm_hour < 12 ) 
      date1 = prev(&date);
    else
      date1 = date;

    time = gmtime(&minimum);
    date2.dt_mday = time->tm_mday;
    date2.dt_mon = time->tm_mon;
    date2.dt_year = time->tm_year +1900 ;


    fprintf(stderr,_("Sorry! This date is out range. The largest date that can work with this version of wday is "));
    print_date(stderr,&date1,"%a %B %d %Y");
    fprintf(stderr,_(" and the earliest is "));
    print_date(stderr,&date2,"%a %B %d %Y");
    fprintf(stderr,".\n");
    exit(1);
  }

  return result;
}
