(* Copyright Sharif Oerton 2008 *)

(* This file is part of waudiocd.

   waudiocd is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation, either version 3 of the License, or (at your
   option) any later version.

   waudiocd is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
   Public License for more details.

   You should have received a copy of the GNU General Public License along
   with waudiocd. If not, see <http://www.gnu.org/licenses/>. *)

open Printf

let stripspecial s = 
  let rec rs sb accbuf = 
    let sb_res = try 
      Scanf.bscanf sb "%c" (fun x -> match x with 
      ' ' -> Buffer.add_string accbuf "\\ "; None
      | '(' -> Buffer.add_string accbuf "\\("; None
      | ')' -> Buffer.add_string accbuf "\\)"; None
      | ':' -> Buffer.add_string accbuf "\\:"; None
      | ';' -> Buffer.add_string accbuf "\\:"; None
      | '\'' -> Buffer.add_string accbuf "\\'"; None
      | _ -> Buffer.add_char accbuf x; None)
    with End_of_file -> Some (Buffer.contents accbuf)
    in 
    match sb_res with
      None -> rs sb accbuf
      | Some n -> n
  in
  let scanbuf = Scanf.Scanning.from_string s in
  let buf = Buffer.create (String.length s) in
  rs scanbuf buf;;

let readfile channel = 
  let rec loop rlst = 
    let line, eof = 
      try
        (input_line channel), false
      with
        | End_of_file -> "", true
    in
    if not eof then
      loop (line :: rlst)
    else
      List.rev rlst
  in
  loop [];;

let gettracklist file = 
  let ic = open_in file in
  try
    let tracklist = readfile ic in
    close_in ic;
    tracklist
  with e ->
    close_in_noerr ic;
    raise e;;

let writedisk numtracks = 
  let command = "cdrecord -eject -v -dao -pad -audio" in
  let rec loop c n acc = 
    if acc = n then 
      c
    else (
      let num = sprintf " %s%d.wav" (if acc < 10 then "0" else "") acc in
      loop (String.concat "" (c :: [num])) n (acc + 1)
    )
  in
  let command = loop command numtracks 1 in 
  Sys.command command;
  Sys.command "rm *.wav*";;

let converttrack track number = 
  let pos = sprintf "%s%d" (if number < 10 then "0" else "") number in
  let command = sprintf "mplayer -ao pcm:file=%s.wav %s" pos track in
  Sys.command command;
  ();;

let convandburn tracks = 
  let f x =
    stripspecial x
  in
  let tracks = List.map f tracks in
  let len = List.length tracks in
  let rec loop tracks acc =
    if acc > len then
      writedisk acc
    else (
      converttrack (List.nth tracks (acc - 1)) acc;
      loop tracks (acc + 1)
    )
  in
  loop tracks 1;;

let dodir () = 
  let p x = 
    let f fmt = 
      Filename.check_suffix x fmt
    in
    f "ogg" || f "mp3" || f "flac" || f "aiff" || f "au" || f "wma" || f "ra" || f "mp4"
  in
  let filelist = Array.to_list (Sys.readdir ".") in
  let filelist = List.filter p filelist in
  let filelist = List.sort compare filelist in
  convandburn filelist;;

if Array.length Sys.argv = 1 then (
  dodir ()
) else (
  convandburn (gettracklist Sys.argv.(1))
)
