#!/usr/bin/env python
# -*- coding: utf-8 -*- #

###
#
# WatchVideo is the legal property of Leonardo Gastón De Luca
# leo[at]kde.org.ar Copyright (c) 2009 Leonardo Gastón De Luca
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.

# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
###

import threading
import os
import time
from threading import Timer
from datetime import datetime, timedelta
import watchvideo.constants as c

class Convert(threading.Thread):
    '''Thread to convert to Ogg Vorbis or Theora, or rips the audio from the video'''
    def __init__(self, callback, filepath, audio_only=True, remove_original=True):
        self.callback = callback
        self.filepath = filepath
        self.filename = os.path.split(filepath)[1]
        self.audio_only = audio_only
        self.remove_original = remove_original
        self.type = callback.__name__
        self.successful = None
        
        threading.Thread.__init__(self)
        
    def run(self):
        if self.type == "rip":
            self.successful = self.callback(self.filepath, self.remove_original)
        else:
            self.successful = self.callback(self.filepath, self.audio_only, self.remove_original)
            
        

class Download(threading.Thread):
    """Downloads the video"""
    
    def __init__(self, video, path, title="", tree_item=None):
        self.video = video
        self.path = path
        self.title = title
        self.local_size = 0
        self.remote_size_kb = 0
        self.stop = False #if true means the download is paused
        self.abort = False #if true the download will stop
                

        threading.Thread.__init__(self)
    
    def run(self):
        
        try:
            remote_file = c.OPENER.open(self.video)
            local_file = open(self.path, 'wb')
            local_size = 0.0
            remote_size = int(remote_file.info()['content-length'])
            remote_size_kb = remote_size // 1024
            while local_size < remote_size:
                if self.abort: break
                
                if self.stop is False:
                    self.local_size = local_size//1024
                    self.remote_size_kb = remote_size_kb
                    buffering = remote_file.read(4096)
                    local_file.write(buffering)
                    local_size += len(buffering)
            self.local_size = local_size//1024
            self.remote_size_kb = remote_size_kb
            remote_file.close()
            local_file.close()

        except IOError:
            pass    # The process was killed, 
                    #probably because the user closed the GUI


class DownloadItem(threading.Thread):
    def __init__(self, tree_item, url, dl_url, dest_file, after_complete=0):
        self.tree_item = tree_item
        self.url = url
        self.dest_file = dest_file
        self.dl_url = dl_url
        self.after_complete = after_complete
        self.completed = False
        self.abort = False
        self.stop = False
        self.init_date = datetime.now()
        self.stop_date = None
        self.time_passed = timedelta(seconds=0)
        self.time_left_str = "00:00:00"
        self.time_stopped = timedelta(seconds=0)
        self.downloaded = 0
        self.local_size_ant = 0
        self.timeout = 0
        self.speed = 0.0
        
        #set unit size
        remote_file = c.OPENER.open(dl_url)
        self.size_kib = self.size_mib = int(remote_file.info()['content-length']) / 1024
        self.unit = "KiB"
        if self.size_kib > 1024:
            self.size_mib /= 1024.0
            self.size_mib = round(self.size_mib, 2)
            self.unit = "MiB"
    
        threading.Thread.__init__(self)
        
    def run(self):
        self.threadDl = Download(self.dl_url, self.dest_file)
        self.threadDl.start()

        Timer(1, self.checkDownloadState).start()
        
    def checkDownloadState(self):
        self.timeout += 1
        self.threadDl.stop = self.stop
        self.threadDl.abort = self.abort
        self.time_passed = datetime.now() - self.init_date
        
        if self.stop and self.stop_date is None: #if stopped
            self.stop_date = datetime.now()
        
        if self.stop_date and not self.stop: #if restarted 
            self.time_stopped += datetime.now() - self.stop_date
            self.stop_date = None
            
        self.calcTimeLeft()
            
        if self.timeout == 20 and self.threadDl.local_size == 0:
            self.completed = True

        #if finished
        if self.threadDl.local_size >= self.threadDl.remote_size_kb and self.threadDl.local_size:
            self.completed = True
        #else update 
        elif self.local_size_ant < self.threadDl.local_size:
            self.speed = self.threadDl.local_size - self.local_size_ant
            self.local_size_ant = self.downloaded = self.threadDl.local_size
        
        if not self.completed and not self.abort: Timer(1, self.checkDownloadState).start()

    def calcTimeLeft(self):
        sec = float(str(self.time_passed-self.time_stopped).split(':')[2])
        min = float(str(self.time_passed-self.time_stopped).split(':')[1]) * 60
        hour = float(str(self.time_passed-self.time_stopped).split(':')[0]) * 3600
        
        time_passed_sec = sec + min + hour
        
        try:
            time_left_sec = ((self.size_kib - self.downloaded) * time_passed_sec) / self.downloaded
            self.time_left_str = time.strftime('%H:%M:%S', time.gmtime(time_left_sec))
        except ZeroDivisionError:
            pass
