# -*- coding: utf-8 -*-
# Copyright (C) 2010  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
Functions operating on classes.
"""


def overrides(method_name, subclass, superclass):
    """Check if `subclass` overrides `method_name` from `superclass`.

    It works on both instance and class methods.

    :Parameters:
      `method_name`
        string naming the method checked
      `subclass`
        a class
      `superclass`
        a superclass of `subclass`

    :Returns:
      `True` if method named `method_name` is different in `subclass`
      and `superclass`, otherwise `False`

    :Exceptions:
      - `ValueError`
        if `subclass` is not a subclass of `superclass`
      - `AttributeError`
        if one of the classes does not have a method named
        `method_name`
      - `TypeError`
        if an argument has invalid type
    """
    if not issubclass(subclass, superclass):
        raise ValueError("%s is not a subclass of %s")
    submethod = getattr(subclass, method_name)
    supermethod = getattr(superclass, method_name)
    try:
        if submethod.im_func is not supermethod.im_func:
            return True
    except AttributeError:
        if submethod is not supermethod:
            return True
    return False
