#!/usr/bin/env python
# -*- coding: utf-8 -*- #
'''WatchVideo GUI'''
###
#
# WatchVideo is the legal property of Leonardo Gastón De Luca
# leo[at]kde.org.ar Copyright (c) 2009 Leonardo Gastón De Luca
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.

# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
###


try:
    from PyQt4 import QtCore, QtGui
except ImportError:
    print "\nError!"
    print "-------------\n"
    print "Dependency Problem - The required PyQt4 libraries couldn\"t be found.\n"
    print "If you are on Debian or one of it\"s derivated systems (like Ubuntu)"
    print "type «sudo aptitude -y install python-qt4» in order to install them."
    print "If you are on another distribution, search for «pyqt4» if it has a package"
    print "manager, or download (and install) it from <http://www.riverbankcomputing.co.uk/pyqt/download.php>.\n"
    exit()
    
from threading import Thread
import sys
import os
import subprocess
try:
    import json
except ImportError:
    import simplejson as json


from PyQt4.QtGui import (QMenu, QAction, QIcon, QProgressBar,
                         QPushButton, QSystemTrayIcon, QComboBox, QTreeWidgetItem)

from watchvideo.ui_main import Ui_MainWindow
from watchvideo.threads import DownloadItem, Convert
from watchvideo.preferences import Preferences
from watchvideo.notification import Notification
from watchvideo import main
import watchvideo.constants as c
from watchvideo import gui_utils
from getmediumurl import Matcher
from watchvideo.utils import is_command


class Gui(QtGui.QMainWindow):
    """WatchVideo's main class."""
    
    def __init__(self, parent=None):
        """Initialize the GUI and many variables and load the configuration."""
        
        QtGui.QWidget.__init__(self, None)
        self.ui = Ui_MainWindow()
        self.ui.setupUi(self)
        
        self.parent = parent
        
        # Load settings
        self.settings = Preferences.loadSettings()
        
        self.download_urls = {}
        self.download_folder = c.HOME_FOLDER
        self.threadUrl = self.timerThread = self.timer_check_downloads = self.convert_timer = None
        self.convert_thread = self.convert_timer = None
        self.active_downloads = []
        self.completed_downloads = []
        self.stopped_downloads = []
        self.videos_info = []
        self.match_url = None
        self.qaction_pressed = ""
        
        self.b_cancelSearch = None
        self.ui.tree_downloads.setColumnCount(5)

        #create tray icon
        self.trayicon = QSystemTrayIcon(QIcon(c.ICON_WATCHVIDEO), self.parent)
        self.trayicon.show()
        
        self.notification = Notification(self.trayicon, self.settings)
        

        self.dialogs = {
            'about': None,        # The instance of the "About..." window will be stored here
            'preferences': None,        # The instance of the "Preferences" window will be stored here
            'playlist': None,
            'history': None,
            'add_videos': None
            }

        
        if c.FIREFOX_SESSION is None:
            self.ui.a_searchBrowser.setDisabled(True)
            
        
        #create the right click menu
        self.ui.tree_downloads.setContextMenuPolicy(QtCore.Qt.CustomContextMenu)
        self.menu = QMenu(self)
        self.start = QAction(QIcon(c.PATH_MEDIA + "start.png"), self.tr("Start"), self)
        self.stop = QAction(QIcon(c.PATH_MEDIA + "stop.png"), self.tr("Stop"), self)
        self.remove = QAction(QIcon(c.ICON_REMOVE), self.tr("Remove"), self)
        self.remove_delete = QAction(QIcon(c.ICON_REMOVE), self.tr("Remove and delete file"), self)
        self.opendir = QAction(QIcon(c.ICON_OPEN_FOLDER), self.tr("Open Folder"), self)
        self.copyurl = QAction(QIcon(c.ICON_COPY), self.tr("Copy original URL"), self)
        self.a_play_local = QAction(self.tr("Play local file"), self)
        
        #add actions to the right click menu
        self.menu.addAction(self.start)
        self.menu.addAction(self.stop)
        self.menu.addSeparator()
        self.menu.addAction(self.remove)
        self.menu.addAction(self.remove_delete)
        self.menu.addSeparator()
        self.menu.addAction(self.opendir)
        self.menu.addAction(self.copyurl)
        self.menu.addAction(self.a_play_local)

        
        # Connect signals with slots
        QtCore.QObject.connect(self.ui.a_addVideos, QtCore.SIGNAL("triggered()"), self.showAddVideos)
        QtCore.QObject.connect(self.ui.a_searchBrowser, QtCore.SIGNAL("triggered()"), self.searchBrowser)
        QtCore.QObject.connect(self.ui.a_clearCompleted, QtCore.SIGNAL("triggered()"), self.clearCompleted)
        QtCore.QObject.connect(self.ui.a_preferences, QtCore.SIGNAL("triggered()"), self.showPreferences)
        QtCore.QObject.connect(self.ui.a_about, QtCore.SIGNAL("triggered()"), self.showAbout)
        QtCore.QObject.connect(self.ui.a_quit, QtCore.SIGNAL("triggered()"), self.close)
        QtCore.QObject.connect(self.ui.a_clipboard, QtCore.SIGNAL("triggered()"), self.addFromClipboard)
        QtCore.QObject.connect(self.ui.a_clipboard, QtCore.SIGNAL("hovered()"), self.showStatusMessage)
        QtCore.QObject.connect(self.ui.a_play, QtCore.SIGNAL("triggered()"), self.addFromClipboard)
        QtCore.QObject.connect(self.ui.a_play, QtCore.SIGNAL("hovered()"), self.showStatusMessage)
        QtCore.QObject.connect(self.ui.a_download, QtCore.SIGNAL("triggered()"), self.addFromClipboard)
        QtCore.QObject.connect(self.ui.a_download, QtCore.SIGNAL("hovered()"), self.showStatusMessage)
        QtCore.QObject.connect(self.copyurl, QtCore.SIGNAL("triggered()"), self.copyOriginalUrl)
        QtCore.QObject.connect(self.ui.tree_downloads, QtCore.SIGNAL("customContextMenuRequested(const QPoint&)"), self.showRightClickMenu)
        QtCore.QObject.connect(self.ui.tree_downloads, QtCore.SIGNAL("itemDoubleClicked(QTreeWidgetItem*, int)"), self.onDownloadDoubleClick)
        QtCore.QObject.connect(self.start, QtCore.SIGNAL("triggered()"), self.startDownloads)
        QtCore.QObject.connect(self.stop, QtCore.SIGNAL("triggered()"), self.stopDownloads)
        QtCore.QObject.connect(self.remove, QtCore.SIGNAL("triggered()"), self.removeTriggered)
        QtCore.QObject.connect(self.remove_delete, QtCore.SIGNAL("triggered()"), self.removeAndDelete)
        QtCore.QObject.connect(self.opendir, QtCore.SIGNAL("triggered()"), self.openDir)
        QtCore.QObject.connect(self.trayicon, QtCore.SIGNAL("activated( QSystemTrayIcon::ActivationReason)"), 
                                                        self.openWindow)
        QtCore.QObject.connect(self.a_play_local, QtCore.SIGNAL("triggered()"), self.play)
        #QtCore.QObject.connect(self.close, QtCore.SIGNAL("triggered()"), self.quit)
     
     
    def onDownloadDoubleClick(self, item, pos):
        #if the download is completed we can either open the directory or play
        #the video file, so we'll get the option selected on the preferences
        for download in self.completed_downloads + self.active_downloads + self.stopped_downloads:
            if download.tree_item == item:
                if self.settings.value('DownloadOpt/dc_downloads').toInt()[0] == 0:
                    self.play(download)
                else:
                    self.openDir(os.path.split(download.dest_file)[0])
    
    def play(self, download=None):
        play = False
        
        if download is not None:
            play = True
        else:
            for item in self.ui.tree_downloads.selectedItems():
                for download in self.completed_downloads:
                    if download.tree_item == item:
                        play = True
                        break
                            
        if play:
            if os.path.exists(download.dest_file):
                filename = os.path.split(download.dest_file)[1]
                Thread(target=main.play, args=(download.dest_file, filename,
                self.settings.value('General/mediaPlayer').toString())).start()
                
                
    def startDownloads(self):
        for item in self.ui.tree_downloads.selectedItems():
            for download in self.stopped_downloads:
                if download.tree_item == item:
                    download.stop = False
                    self.stopped_downloads.remove(download)
                    self.active_downloads.append(download)
                    

    def stopDownloads(self):
        for item in self.ui.tree_downloads.selectedItems():
            for download in self.active_downloads:
                if download.tree_item == item and download.stop is False:
                    download.stop = True
                    download.tree_item.setText(3, "-")
                    download.tree_item.setText(4, "-")
                    self.active_downloads.remove(download)
                    self.stopped_downloads.append(download)
                    
    def removeTriggered(self):
        downloads_removed = []
        
        for item in self.ui.tree_downloads.selectedItems(): 
            for download in self.active_downloads + self.completed_downloads + self.stopped_downloads:
                if download.tree_item == item:
                    if not download.completed:
                        download.abort = True
                        
                    i = self.ui.tree_downloads.indexOfTopLevelItem(item)
                    self.ui.tree_downloads.takeTopLevelItem(i)
                    
                    if download.stop:
                        self.stopped_downloads.remove(download)
                    elif download.completed:
                        self.completed_downloads.remove(download)
                    else:
                        self.active_downloads.remove(download)
                        
                    downloads_removed.append(download)
        
        if self.ui.tree_downloads.topLevelItemCount() == 0:
            self.ui.a_clearCompleted.setDisabled(True)
                    
        return downloads_removed
                        
            
    
    def removeAndDelete(self):
        downloads_removed = self.removeTriggered()
        
        for download in downloads_removed:
            os.remove(download.dest_file)

     
    def showRightClickMenu(self, point):
        if self.ui.tree_downloads.selectedItems():
            stopped_found = False
            active_found = False
            completed_found = False
            
            for item in self.ui.tree_downloads.selectedItems():
                #search stopped/paused downloads
                for download in self.stopped_downloads:
                    if download.tree_item == item:
                        stopped_found = True
                        break
                        
                if stopped_found: #if a stopped download is found
                    if not self.start.isEnabled(): self.start.setEnabled(True)
                else:
                    if self.start.isEnabled(): self.start.setEnabled(False)
                
                #search active downloads
                for download in self.active_downloads:
                    if download.tree_item == item:
                        active_found = True
                        break
                        
                if active_found: #if an active download is found
                    if not self.stop.isEnabled(): self.stop.setEnabled(True)
                else:
                    if self.stop.isEnabled(): self.stop.setEnabled(False)
            
                #search completed downloads
                for download in self.completed_downloads:
                    if download.tree_item == item:
                        completed_found = True
                        break
                    
                if completed_found:
                    self.a_play_local.setEnabled(True)
                else:
                    self.a_play_local.setEnabled(False)
                        
            point = QtGui.QCursor.pos() 
            self.menu.exec_(point)
            


        
    def addNewDownloads(self, videos):
        if not videos: 
            del self.videos_info [:]
            return
        
        if videos[0].down: #download
            for video in videos:
                if len(video.name) * 7 > self.ui.tree_downloads.header().sectionSize(0):
                    self.ui.tree_downloads.header().resizeSection(0, len(video.name)*7)
                
                item = QtGui.QTreeWidgetItem([video.name, "-", "", "0", "00:00:00"])
                #insert QProgressBar
                self.ui.tree_downloads.addTopLevelItem(item)
                self.ui.tree_downloads.setItemWidget(item, 2, QProgressBar(self))
                
                download_item = DownloadItem(item, video.url, video.dl_url, video.filepath, video.after_complete)
                download_item.start()
                
                self.active_downloads.append(download_item)
                
                if not self.timer_check_downloads:
                    self.timer_check_downloads = QtCore.QTimer()
                    QtCore.QObject.connect(self.timer_check_downloads, QtCore.SIGNAL("timeout()"), self.updateDownloads)
                    self.timer_check_downloads.start(750)
        else: #Play
            for video in videos:
                Thread(target=main.play, args=(video.dl_url, video.name,
                self.settings.value('General/mediaPlayer').toString())).start()
                
        del self.videos_info [:]

    def afterComplete(self, download_item):
        start_thread = False
        if download_item.after_complete == 1: #play option
            Thread(target=main.play, args=(download_item.dest_file, os.path.split(download_item.dest_file)[1],
            self.settings.value('General/mediaPlayer').toString())).start()
        elif download_item.after_complete == 2: #convert to Ogg Vorbis
            self.convert_thread = Convert(main.convert, download_item.dest_file)
            start_thread = True
        elif download_item.after_complete == 3: #convert to Ogg Theora
            self.convert_thread = Convert(main.convert, download_item.dest_file, audio_only=False)
            start_thread = True
        elif download_item.after_complete == 4: #rip audio option
            self.convert_thread = Convert(main.rip, download_item.dest_file)
            start_thread = True
            
        if start_thread:
            self.convert_thread.start()
            self.convert_timer = QtCore.QTimer()
            QtCore.QObject.connect(self.convert_timer, QtCore.SIGNAL("timeout()"), self.isConversionDone)
            self.convert_timer.start(750)
            
            
    def isConversionDone(self):
        if self.convert_thread.type == "rip":
            message_waiting = self.tr("Ripping audio of") + " " + self.convert_thread.filename
            title_done = self.tr("Audio Ripped")
            title_error = self.tr("Error ripping audio")
        else:
            message_waiting = self.tr("Converting video:") + " " + self.convert_thread.filename
            title_done = self.tr("Video Converted")
            title_error = self.tr("Error converting video")

        if self.convert_thread.successful is not None:
            self.convert_timer.stop()

            if self.convert_thread.successful:
                self.notification.show(title_done, self.convert_thread.filename)
                self.ui.statusbar.showMessage(title_done + ": " + self.convert_thread.filename)
            else:
                self.notification.show(title_error, self.convert_thread.filename)
                self.ui.statusbar.showMessage(title_error + ": " + self.convert_thread.filename)
                if self.settings.value("Notification/notifyError").toBool():
                    self.notification.show(title_error, self.convert_thread.filename)
            
            self.convert_thread = None
        else:
            self.ui.statusbar.showMessage(message_waiting)

    def searchBrowser(self):
        session = json.loads(open(c.FIREFOX_SESSION).read().strip("()"))
        urls = []
        matcher = Matcher()
        regxps = [plugin.match_re for plugin in matcher.plugins.itervalues()]
        
        #search windows and tabs open in firefox
        for window in session["windows"]:
            for tab in window["tabs"]:
                for entry in tab["entries"]:
                    try:
                        url = entry["url"]
                        for regxp in regxps:
                            if regxp and regxp.match(url):
                                urls.append(url)
                    except IndexError:
                        pass
         
        self.getFilesUrl(urls)
                        

    def getFilesUrl(self, urls):
        if self.match_url: return
        
        urls_clean = []
        for url in urls:
            if url[:7] == "http://" and len(url) != len("http://"):
                urls_clean.append(url)
        urls = urls_clean     
                
        if urls:
            #add cancel button
            if not self.b_cancelSearch:
                self.b_cancelSearch = QPushButton("Cancel", self.ui.statusbar)
                QtCore.QObject.connect(self.b_cancelSearch, QtCore.SIGNAL("clicked()"), self.cancelSearch)
            else:
                self.b_cancelSearch.show()
            self.ui.statusbar.addPermanentWidget(self.b_cancelSearch)
    
            self.cleanStatusTip()
            
            self.match_url = main.Match(urls, False, self.getQualityOptions()) #start a check thread       
            Thread(target=self.match_url.match, args=()).start()
            
            self.timerThread = QtCore.QTimer()
            QtCore.QObject.connect(self.timerThread, QtCore.SIGNAL("timeout()"), self.isCheckingDone)
            self.timerThread.start(750)
        else:
            self.ui.statusbar.showMessage("Not a valid URL")
    
    
    
    def isCheckingDone(self):
        """Called in intervals of 750ms to see if the url thread is ready"""
        if self.match_url.done:
            self.timerThread.stop()
            self.timerThread = None
            self.match_url.done = False
            valid_videos = self.match_url.valid_videos
            self.match_url = None
            
            if valid_videos:
                self.resetStatusBar()
                
                if self.qaction_pressed == self.tr("Download"):
                    for video in valid_videos: 
                        video.folder = self.settings.value("DownloadOpt/folder").toString()
                        video.after_complete = self.settings.value("DownloadOpt/afterDl").toInt()[0]
                    self.addNewDownloads(valid_videos)
                elif self.qaction_pressed == self.tr("Play"):
                    for video in valid_videos: video.toPlay()
                    self.addNewDownloads(valid_videos)
                else:
                    self.showAddVideos(valid_videos)
                
            else:
                self.ui.statusbar.showMessage(self.tr("No supported services were found"))
                self.b_cancelSearch.setText(self.tr("Close"))
        elif self.match_url.url: #if the url is valid, show it
            self.ui.statusbar.showMessage(self.tr("Currently checking: ") + self.match_url.url)
    
    def getQualityOptions(self):
        quality_options = {}
        plugins = Matcher().plugins.values()
        for plugin in plugins:
            if plugin.website_name is not None and isinstance(plugin.website_name, basestring):
                plug_name = plugin.website_name
            else:
                plug_name = plugin.__name__
                
            value = self.settings.value("VideoQuality/%s" % plug_name).toInt()[0]
            quality_options[plug_name] = value
            
        return quality_options
    
    
    def clearCompleted(self):
        for i in xrange(self.ui.tree_downloads.topLevelItemCount()-1, -1, -1):
            currentItem = self.ui.tree_downloads.topLevelItem(i)
            for download in self.completed_downloads:
                if download.tree_item == currentItem:
                    self.ui.tree_downloads.takeTopLevelItem(i)
                    self.completed_downloads.remove(download)
        
        self.ui.a_clearCompleted.setDisabled(True)
        

        
    def updateDownloads(self):
        for download in self.active_downloads:
            progressBar = self.ui.tree_downloads.itemWidget(download.tree_item, 2)
            total_kib = download.size_kib
            total_mib = download.size_mib
            unit = download.unit
            downloaded = download.downloaded
            speed = download.speed
            time_left_str = download.time_left_str
            
            if download.completed:
                progressBar.setValue(total_kib)
                download.tree_item.setText(3, "") #Cleans the 'speed' column
                download.tree_item.setText(4, "") #Cleans the 'time left' column
                self.completed_downloads.append(download)
                self.active_downloads.remove(download)
                
                if not self.ui.a_clearCompleted.isEnabled():
                    self.ui.a_clearCompleted.setEnabled(True)
                
                if self.settings.value("Notification/notifyDone").toBool():
                    self.notification.show(self.tr("Download Finished"), 
                    os.path.split(download.dest_file)[1])
                
                self.afterComplete(download)
                
            else:
                download.tree_item.setText(1, str(total_mib) + unit)
                progressBar.setMaximum(total_kib)
                progressBar.setValue(downloaded)
                
                download.tree_item.setText(3, str(speed) + " KiB/s")
                download.tree_item.setText(4, time_left_str)

    def openDir(self, dir=None):
        
        if dir is None:
            directories = []
            for item in self.ui.tree_downloads.selectedItems():
                for download in self.stopped_downloads + self.active_downloads + self.completed_downloads:
                    if download.tree_item == item:
                        directories.append(os.path.split(str(download.dest_file))[0])
            
            directories = [ directory.decode("utf-8") for directory in directories ] 
            directories = set(directories)
            filemanager = None
        else:
            directories = [dir]
        
        for manager in c.FILEMANAGERS:
            if is_command(manager):
                filemanager = manager
                break

        if filemanager:
            for directory in directories:
                subprocess.Popen((filemanager, directory.encode()))
            
            self.ui.statusbar.showMessage(self.tr("Opening download directory ") +
            " ".join(directories) + " " + self.tr("with") + " " + filemanager, 5000)
            
    #dialogs
    def showAddVideos(self, urls=None):
        if self.dialogs["add_videos"] is None:
            from add_videos import AddVideosDialog
            self.dialogs["add_videos"] = AddVideosDialog(self)
       
        self.dialogs["add_videos"].load_settings(self.settings)
        if urls: 
            self.dialogs["add_videos"].load_valid_urls(urls)
            self.dialogs["add_videos"].ui.g_pasteLinks.hide()
            self.dialogs["add_videos"].ui.b_checkLinks.hide()
            self.dialogs["add_videos"].ui.l_highquality.hide()
            self.dialogs["add_videos"].ui.cb_highquality.hide()
        else:
            self.dialogs["add_videos"].ui.g_pasteLinks.show()
            self.dialogs["add_videos"].ui.b_checkLinks.show()
            self.dialogs["add_videos"].ui.l_highquality.show()
            self.dialogs["add_videos"].ui.cb_highquality.show()
            
        self.dialogs["add_videos"].show()
        self.dialogs["add_videos"].activateWindow()
        
        

    def showPreferences(self):
        """Displays the Preferences dialog."""
        self.dialogs['preferences'] = Preferences(self, self.settings)
        self.dialogs['preferences'].show()
        self.dialogs['preferences'].activateWindow()
        self.dialogs['preferences'] = None
        
    def showAbout(self):
        """Displays the About dialog."""
        if self.dialogs['about'] is None:
            from about import AboutDialog
            self.dialogs['about'] = AboutDialog(self)
        
        self.dialogs['about'].show()
        self.dialogs['about'].activateWindow()
        
    def addFromClipboard(self):
        urls = str(self.getClipboardText()).split()
        self.qaction_pressed = str(self.sender().text()).decode("utf-8")  
        self.getFilesUrl(urls)
        
    def getClipboardText(self):
        return str(QtGui.QApplication.clipboard().text())
        
    def copyOriginalUrl(self):
        """Copies the original url to the Clipboard"""
        urls = []
        for item in self.ui.tree_downloads.selectedItems():
            for download in self.active_downloads + self.completed_downloads + self.stopped_downloads:
                if download.tree_item == item:
                    urls.append(download.url)
                    
        QtGui.QApplication.clipboard().setText("\n".join(urls))
        
    def showStatusMessage(self):
        if not self.match_url:
            self.sender().setStatusTip(self.getClipboardText())
            self.sender().showStatusText()
            
    def cleanStatusTip(self):
        self.ui.a_clipboard.setStatusTip("")
        self.ui.a_download.setStatusTip("")
        self.ui.a_play.setStatusTip("")
    
    def cancelSearch(self):
        self.stopChecking()
        self.resetStatusBar()
        
    def stopChecking(self):
        if self.match_url:
            self.match_url.stop = True
            self.timerThread.stop()
            self.match_url = None
            self.timerThread = None
            
    def resetStatusBar(self):
        self.ui.statusbar.showMessage("")
        self.ui.statusbar.removeWidget(self.b_cancelSearch)    
    
    def openWindow(self, activation):
        if activation in (3, 2): #show/hide window
            if self.isVisible():
                self.hide()
            else:
                self.show()
    
    def closeEvent(self, event):
        if self.active_downloads or self.stopped_downloads:
            answer = gui_utils.confirm(self, title="Quit?",
            msg=self.tr("There are still downloads active, do you really want to quit?" + 
            "\n(you won't be able to continue the downloads after quit)"))

            if answer == QtGui.QMessageBox.Yes:
                for download in self.active_downloads + self.stopped_downloads:
                    download.abort = True
            else:
                event.ignore()
        
def run():
    app = QtGui.QApplication(sys.argv)
    
    # Assert it to avoid pyflakes unused import warning -- resource
    # module imports have side effects.
    import translations_rc
    assert translations_rc

    locale = str(QtCore.QLocale.system().name())
    translator = QtCore.QTranslator()
    translator.load(':/po/' + locale.split('_')[0])
    
    app.installTranslator(translator)
    
    mainApp = Gui(app)
    mainApp.show()
    
    try:
        app.exec_()
    except KeyboardInterrupt:
        sys.exit()
        
