/*
 * This file is a part of VyQChat.
 *
 * Copyright (C) 2002-2004 Pawel Stolowski <yogin@linux.bydg.org>
 *
 * VyQChat is free software; you can redestribute it and/or modify it
 * under terms of GNU General Public License by Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY. See GPL for more details.
 */

#include "ulist.h"
#include "msgsndwin.h"
#include "icons.h"
#include "vcprotocol.h"
#include <qpopupmenu.h>

UsersList::UsersList(QWidget *parent): QListBox(parent)/*{{{*/
{
	setMinimumWidth(120);
	setFocusPolicy(QWidget::NoFocus);
	setSelectionMode(QListBox::Single);
	setSizePolicy(QSizePolicy(QSizePolicy::Minimum, QSizePolicy::Expanding));
	setAutoBottomScrollBar(false);

	//
	// context menu when no user selected
	menu = new QPopupMenu(this);
	menu->insertItem(Icons::icon_massmessage, tr("Mass Message"), ID_MASSMESSAGE);
	menu->insertItem(tr("Refresh list"), ID_REFRESH);
	connect(menu, SIGNAL(activated(int)), this, SLOT(menuAction(int)));

	//
	// context menu for user
	usermenu = new QPopupMenu(this);
	usermenu->insertItem(Icons::icon_message, tr("Message"), ID_MESSAGE);
	usermenu->insertItem(Icons::icon_private, tr("Private chat"), ID_PRVCHAT);
	usermenu->insertItem(Icons::icon_info, tr("Request info"), ID_REQINFO);
	usermenu->insertItem(Icons::icon_beep, tr("Beep"), ID_BEEP);
	usermenu->insertItem(tr("Say \"Nick:\""), ID_SAYNICK);
	usermenu->insertSeparator();
	usermenu->insertItem(Icons::icon_massmessage, tr("Mass Message"), ID_MASSMESSAGE);
	usermenu->insertSeparator();
	usermenu->insertItem(tr("Refresh list"), ID_REFRESH);

	connect(usermenu, SIGNAL(activated(int)), this, SLOT(menuAction(int)));
	connect(this, SIGNAL(contextMenuRequested(QListBoxItem *, const QPoint &)), this, SLOT(popupContextMenu(QListBoxItem*, const QPoint &)));
	connect(this, SIGNAL(doubleClicked(QListBoxItem *)), this, SLOT(sendMessage(QListBoxItem *)));
}/*}}}*/

UsersList::~UsersList()/*{{{*/
{
}/*}}}*/

bool UsersList::newNick(const QString &onick, const QString &nnick)/*{{{*/
{
	User *user = getUser(onick);
	if (user)
	{
		user->setNick(nnick);
		umap.remove(onick);
		umap[nnick] = user;
		updateItem(user);
		return true;
	}
	return false;
}/*}}}*/

User *UsersList::addUser(const QString &nick, char gender, char status, const QHostAddress &addr)/*{{{*/
{
	if (exists(nick))
		return NULL;
	
	User *u = new User(this, nick, addr);
	u->setGender(gender);
	u->setStatus(status);
	QListBox::sort();
	return umap[nick] = u;
}/*}}}*/

bool UsersList::delUser(const QString &nick)/*{{{*/
{
	User *user = getUser(nick);
	if (user)
	{
		umap.remove(nick);
		delete user;
		return true;
	}
	return false;
}/*}}}*/

char UsersList::getStatus(const QString &nick)/*{{{*/
{
	User *user = getUser(nick);
	return user!=NULL ? user->getStatus() : STATUS_UNKNOWN; 
}/*}}}*/

bool UsersList::setStatus(const QString &nick, char s)/*{{{*/
{
	User *user = getUser(nick);
	return user!=NULL ? updateItem(user), setStatus(user, s) : false;
}/*}}}*/

void UsersList::decDeadCounters()/*{{{*/
{
	QStringList l;
	//
	// create a temporary list of non-existent users;
	for (UserMap::iterator it = umap.begin(); it!=umap.end(); ++it)
		if (it.data()->decDeadCounter() < 0)
			l.append(it.key());
	//
	// delete all users from list
	if (!l.empty())
		for (QStringList::iterator it=l.begin(); it!=l.end(); ++it)
			delUser(*it);
}/*}}}*/

void UsersList::popupContextMenu(QListBoxItem *it, const QPoint &p)/*{{{*/
{
	if (it)
		usermenu->popup(p);
	else
		menu->popup(p);
}/*}}}*/

void UsersList::dummy()/*{{{*/
{
}/*}}}*/

void UsersList::sendMessage(QListBoxItem *it)/*{{{*/
{
	if (it)
		emit sendMessage(it->text());
}/*}}}*/

void UsersList::menuAction(int option)/*{{{*/
{
//	printf("%d\n", option);

	int i = currentItem();
	const QString nick = text(i);

	if (option == ID_MASSMESSAGE)
	{
/*		QStringList allnicks;
		for (int i=0; i<count(); i++)
			allnicks.append(text(i));*/
		emit sendMassMessage(getAllUsers());
		return;
	}
	if (option == ID_REFRESH)
	{
		emit refresh();
		return ;
	}

	if (i<0)
		return;
	
	switch (option)
	{
		case ID_MESSAGE: emit sendMessage(nick); break;
		case ID_REQINFO: emit requestInfo(nick); break;
		case ID_BEEP:    emit beep(nick); break;
		case ID_PRVCHAT: emit privateChat(nick); break;
		case ID_SAYNICK: emit sayNick(nick); break;
		default: break;
	}
}/*}}}*/

QStringList UsersList::getAllUsers()/*{{{*/
{
	QStringList allnicks;
	for (int i=0; i<count(); i++)
		allnicks.append(text(i));
	return allnicks;
}/*}}}*/
	
