/*
 * This file is a part of VyQChat.
 *
 * Copyright (C) 2002-2004 Pawel Stolowski <yogin@linux.bydg.org>
 *
 * VyQChat is free software; you can redestribute it and/or modify it
 * under terms of GNU General Public License by Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY. See GPL for more details.
 */

#ifndef __SETTINGS_H
#define __SETTINGS_H

#include "sound.h"
#include "vcprotocol.h"
#include "uuid.h"
#include "history.h"
#include <qstring.h>
#include <qcolor.h>
#include <qsettings.h>

#include "vyevent.h"

class SettingsDialog;

class Settings
{
	private:
		UUID uuid;
		QString icontheme;
		QString nick, def_nick;
		QString hostname, def_hostname;
		//QString ip, def_ip;
		QString def_ip;
		QHostAddress ip, broadcast, mcastgrp;
		unsigned int port, def_port;
		QString def_broadcast, def_mcastgrp;
		QString topic, def_topic;
		Sound::SoundType def_sound;
		int status;  //status on exit
		int refresh;
		QColor special_color;
		QColor normal_color;
		QColor own_color;
		int x, y, width, height, listwidth;
		Sound::SoundType sound;
		QString playcmd;
		QString snd[EventDummy];
		bool sndp[EventDummy]; //play-dont play flags
		QString scr[EventDummy];
		bool scrf[EventDummy]; //script enabled-disabled
		QString def_snd[EventDummy];
		QString def_playcmd;
		bool useutf;
		bool confirm_exit;
		bool hide_channels;
		bool regexp_on;
		bool toolbar_on;
		bool userslist_on;
		bool nickcombo_on;
		bool popup;
		bool nopopup_dnd;
		bool keep_status; //if last status should be restored on startup
		bool systray;  //dock into system tray?
		QString encoding;
		QString version; //vyqchat version for detecting older config file

		QSettings cfg;

		void setUUID(const UUID &u) { uuid = u; cfg.writeEntry("/General/UUID", uuid.asString()); }
		const QString getEventKey(VyEvent s) const;
	
	public:
		Settings();
		~Settings();
		bool getDefaults();
		bool load();

		//
		// Get methods
		const UUID& Settings::getUUID()    const { return uuid; }
		const QString& getIconThemeName()  const { return icontheme; }
		const QString& getNick()           const { return nick; }
		const QHostAddress& getBroadcast() const { return broadcast; }
		const QHostAddress& getIP()        const { return ip; }
		const QString& getHostname()       const { return hostname; }
		const QHostAddress& getMulticastGroup() const { return mcastgrp; }
		const unsigned int getPort()       const { return port; }
		bool getUseSystray()               const { return systray; }
		int getStatus()                    const { return status; }
		bool getKeepStatus()               const { return keep_status; }
		bool getRegexpOn()                 const { return regexp_on; }
		bool getTooolbarOn()               const { return toolbar_on; }
		bool getHideChannels()             const { return hide_channels; }
		bool getUsersListOn()              const { return userslist_on; }
		bool getNickComboOn()              const { return nickcombo_on; }
		int getUsersListWidth()            const { return listwidth; }
		int getWidth()                     const { return width; }
		int getHeight()                    const { return height; }
		int getX()                         const { return x; }
		int getY()                         const { return y; }
		int getRefresh()                   const { return refresh; }
		const QString& getVersion()        const { return version; }
		const QString& getEncoding()       const { return encoding; }
		bool getUseUTF()                   const { return useutf; }
		const QString& getTopic()          const { return topic; }
		bool getConfirmExit()              const { return confirm_exit; }
		bool getPopupsOn()                 const { return popup; }
		bool getNoDNDpopups()              const { return nopopup_dnd; }
		bool isSoundEnabled(VyEvent i)  const { return getSampleEnabled(i); }
		const QString& getPlayCommand()    const { return playcmd; }
		Sound::SoundType getSoundType()    const { return sound; }
		bool getSampleEnabled(VyEvent i) const { return sndp[i]; }
		const QString& getSampleFileName(VyEvent i) const { return snd[i]; }
		const QString& getScriptFileName(VyEvent e) const { return scr[e]; }
		bool getScriptEnabled(VyEvent e)  const { return scrf[e]; }
		const History getDNDhistory()     const  { return cfg.readListEntry("/History/dnd"); }
		const History getOfflineHistory() const  { return cfg.readListEntry("/History/offline"); }
		const History getAwayHistory()    const  { return cfg.readListEntry("/History/away"); }
		const History getNicksHistory()   const  { return cfg.readListEntry("/History/nicks"); }
		const History getRegexpHistory()  const  { return cfg.readListEntry("/History/regexp"); }

		//
		// Set methods
		void setVersion(const QString &v) { cfg.writeEntry("/General/Version", this->version = v); }
		void setUseSystray(bool f) { cfg.writeEntry("/Misc/Systray", this->systray = f); }
		void setIconThemeName(const QString &name) { cfg.writeEntry("/Look/IconTheme", this->icontheme = name); }
		void setNick(const QString &nick) { cfg.writeEntry("/Network/Nick", this->nick = nick); }
		void setHostname(const QString &hostname) { cfg.writeEntry("/Network/Hostname", this->hostname = hostname); }
		void setIP(const QString &ip) { cfg.writeEntry("/Network/IP", ip); this->ip.setAddress(ip); }
		void setPort(unsigned int p) { cfg.writeEntry("/Network/Port", static_cast<int>(this->port = p)); }
		void setBroadcast(const QString &ip) { cfg.writeEntry("/Network/Broadcast", ip); this->broadcast.setAddress(ip); }
		void setMulticast(const QString &ip) { cfg.writeEntry("/Network/MulticastGroup", ip); this->mcastgrp.setAddress(ip); }
		void setRefresh(const int t) { cfg.writeEntry("/Network/Refresh", this->refresh = t); }
		void setGeometry(int x, int y, int w, int h)
		{
			cfg.beginGroup("/Window");
			cfg.writeEntry("/x", this->x = x);
			cfg.writeEntry("/y", this->y = y);
			cfg.writeEntry("/Width", this->width = w);
			cfg.writeEntry("/Height", this->height = h);
			cfg.endGroup();
		}
		void setUsersListWidth(int w) { listwidth = w; }
		void setRegexpOn(bool f) { cfg.writeEntry("/General/RegExpOn", regexp_on = f); }
		void setToobarOn(bool f) { cfg.writeEntry("/General/ToolbarOn", toolbar_on = f); }
		void setUsersListOn(bool f) { cfg.writeEntry("/General/UsersListOn", userslist_on = f); }
		void setNickComboOn(bool f) { cfg.writeEntry("/General/NickComboOn", nickcombo_on = f); }
		void setTopic(const QString &t) { cfg.writeEntry("/General/Topic", topic = t); }
		void setConfirmExit(bool f) { cfg.writeEntry("/General/ConfirmExit", confirm_exit = f); }
		void setKeepStatus(bool f) { cfg.writeEntry("/General/KeepStatus", keep_status = f); }
		void setStatus(int s) { cfg.writeEntry("/General/Status", status = s); }
		void setHideChannels(bool f) { cfg.writeEntry("/Misc/HideChannels", hide_channels = f); }
		void setPopups(bool f) { cfg.writeEntry("/Misc/Popup", popup = f); }
		void setNoDNDpopups(bool f) { cfg.writeEntry("/Misc/NoPopupDND", nopopup_dnd = f); }
		void setEncodingName(const QString e) { cfg.writeEntry("/Misc/Encoding", encoding = e); }
		void setUseUTF(bool f) { cfg.writeEntry("/Misc/UTF", useutf = f); }
		void setSoundType(Sound::SoundType t) { cfg.writeEntry("/Sound/sound", sound = t); }
		void setPlayCommand(const QString &cmd) { cfg.writeEntry("/Sound/PlayCommand", playcmd = cmd); }
		void setSampleFileName(VyEvent s, const QString &f);
		void setSampleEnabled(VyEvent s, bool f);
		void setScriptFileName(VyEvent e, const QString &f);
		void setScriptEnabled(VyEvent e, bool f);
		void setDNDhistory(const History &s) { cfg.writeEntry("/History/dnd", s.getAll()); }
		void setAwayHistory(const History &s) { cfg.writeEntry("/History/away", s.getAll()); }
		void setOfflineHistory(const History &s) { cfg.writeEntry("/History/offline", s.getAll()); }
		void setNicksHistory(const History &s) { cfg.writeEntry("/History/nicks", s.getAll()); }
		void setRegexpHistory(const History &s) { cfg.writeEntry("/History/regexp", s.getAll()); }
};

#endif

