/*
 * This file is a part of VyQChat.
 *
 * Copyright (C) 2002-2004 Pawel Stolowski <yogin@linux.bydg.org>
 *
 * VyQChat is free software; you can redestribute it and/or modify it
 * under terms of GNU General Public License by Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY. See GPL for more details.
 */

#include "global.h"
#include "settings.h"
#include <unistd.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <arpa/inet.h>
#include <stdlib.h>

Settings::Settings() /*{{{*/
{
	cfg.beginGroup("/vyqchat");
}/*}}}*/

Settings::~Settings()/*{{{*/
{
	cfg.endGroup();
}/*}}}*/

bool Settings::getDefaults()/*{{{*/
{
	QString datapath(DATADIR);

	bool status = false;
	def_topic = "no topic";
	def_sound = Sound::PCSpeaker;
	def_playcmd = "/usr/bin/play";
	def_snd[EventChatline] = datapath + "chatline.wav";
	def_snd[EventBeep] = datapath + "beep.wav";
	def_snd[EventJoin] = datapath + "join.wav";
	def_snd[EventLeave] = datapath + "leave.wav";
	def_snd[EventMessage] = datapath + "beep.wav";
	def_snd[EventRegexpMatch] = datapath + "beep.wav";

	//
	// get hostname
	char tmp[255];
	gethostname(tmp, 254);
	def_hostname = QString(tmp);

	//
	// get username
	char *login = getenv("LOGNAME");
	def_nick = QString(login);

	//
	// get eth0 parameters
	def_broadcast = "255.255.255.255";
	def_mcastgrp = "227.0.0.2";
	def_ip = "127.0.0.1";
	def_port = 8167; 

	struct ifconf ifc;
	int sockfd = socket(AF_INET, SOCK_DGRAM, 0);

	ifc.ifc_req = new ifreq [16];
	ifc.ifc_len = 16*sizeof(struct ifreq);

	if (ioctl(sockfd, SIOCGIFCONF, &ifc)==0)
	{
		status = true;

		const int num = ifc.ifc_len/sizeof(struct ifreq);
		struct ifreq *req = ifc.ifc_req; 
		for (int i=0; i<num; i++)
		{
			if (strcmp(req[i].ifr_name, "eth0")==0)
			{
				if (ioctl(sockfd, SIOCGIFBRDADDR , &req[i])==0) 
					def_broadcast = QString(inet_ntoa(*(in_addr *)(&req[i].ifr_broadaddr.sa_data[2])));
				else
					status = false;
				
				if (ioctl(sockfd, SIOCGIFADDR , &req[i])==0) 
					def_ip = QString(inet_ntoa(*(in_addr *)(&req[i].ifr_addr.sa_data[2])));
				else
					status = false;
			}
		}
		
	}
	delete [] ifc.ifc_req;
	close(sockfd);
	return status;
}/*}}}*/

bool Settings::load()/*{{{*/
{
	bool ok;

	getDefaults();

	cfg.beginGroup("/Network");
		nick = cfg.readEntry("/Nick", def_nick);
		hostname = cfg.readEntry("/Hostname", def_hostname);
		ip.setAddress(cfg.readEntry("/IP", def_ip));
		port = cfg.readNumEntry("/Port", def_port);
		broadcast = cfg.readEntry("/Broadcast", def_broadcast);
		mcastgrp.setAddress(cfg.readEntry("/MulticastGroup", def_mcastgrp));
		refresh = cfg.readNumEntry("/Refresh", 300);
	cfg.endGroup();
	cfg.beginGroup("/Misc");
		encoding = cfg.readEntry("/Encoding", "");
		useutf = cfg.readBoolEntry("/UTF", true);
		hide_channels = cfg.readBoolEntry("/HideChannels", false);
		popup = cfg.readBoolEntry("/Popup", true);
		nopopup_dnd = cfg.readBoolEntry("/NoPopupDND", true);
	cfg.endGroup();
	cfg.beginGroup("/General");
		version = cfg.readEntry("/Version", "0", &ok);
		if (!ok)
			cfg.writeEntry("/Version", VYQCHAT_VERSION);
		uuid = cfg.readEntry("/UUID", QString::null, &ok);
		if (!ok)
			cfg.writeEntry("/UUID", uuid.asString());
		regexp_on = cfg.readBoolEntry("/RegExpOn", false);
		toolbar_on = cfg.readBoolEntry("/ToolbarOn", true);
		userslist_on = cfg.readBoolEntry("/UsersListOn", true);
		nickcombo_on = cfg.readBoolEntry("/NickComboOn", true);
		confirm_exit = cfg.readBoolEntry("/ConfirmExit", true);
		topic = cfg.readEntry("/Topic", def_topic);
		keep_status = cfg.readBoolEntry("/KeepStatus", true);
		status = cfg.readNumEntry("/Status", STATUS_NORMAL);
	cfg.endGroup();
	cfg.beginGroup("/Sound");
		sound = static_cast<Sound::SoundType>(cfg.readNumEntry("/sound", def_sound));
		playcmd = cfg.readEntry("/PlayCommand", def_playcmd);
	cfg.endGroup();
	cfg.beginGroup("/Wavs");
		for (int i=0; i<EventDummy; i++)
			snd[i] = cfg.readEntry(getEventKey(static_cast<VyEvent>(i)), def_snd[static_cast<VyEvent>(i)]);
	cfg.endGroup();
	cfg.beginGroup("/WavFlags");
		for (int i=0; i<EventDummy; i++)
			sndp[i] = cfg.readBoolEntry(getEventKey(static_cast<VyEvent>(i)), false);
	cfg.endGroup();
	cfg.beginGroup("/Scripts");
		for (int i=0; i<EventDummy; i++)
			scr[i] = cfg.readEntry(getEventKey(static_cast<VyEvent>(i)), QString::null);
	cfg.endGroup();
	cfg.beginGroup("/ScriptFlags");
		for (int i=0; i<EventDummy; i++)
			scrf[i] = cfg.readBoolEntry(getEventKey(static_cast<VyEvent>(i)), false);
	cfg.endGroup();

	cfg.beginGroup("/Window");
		x = cfg.readNumEntry("/x", -1);
		y = cfg.readNumEntry("/y", -1);
		width = cfg.readNumEntry("/Width", 480);
		height = cfg.readNumEntry("/Height", 360);
		listwidth = cfg.readNumEntry("/ListWidth", 120);
	cfg.endGroup();
	cfg.beginGroup("/Look");
		icontheme = cfg.readEntry("/IconTheme", "default");
	cfg.endGroup();
	return true;
}/*}}}*/

const QString Settings::getEventKey(VyEvent s) const/*{{{*/
{
	switch (s)
	{
		case EventChatline: return QString("/Chatline"); break;
		case EventMessage: return QString("/Message"); break;
		case EventJoin: return QString("/Join"); break;
		case EventLeave: return QString("/Leave"); break;
		case EventBeep: return QString("/Beep"); break;
		case EventRegexpMatch: return QString("/RegexpMatch"); break;
		default: break;
	}
	return QString::null;
}/*}}}*/

void Settings::setSampleFileName(VyEvent s, const QString &f)/*{{{*/
{
	const QString key = getEventKey(s);
	if (!key.isEmpty())
	{
		cfg.beginGroup("/Wavs");
		cfg.writeEntry(key, snd[s] = f);
		cfg.endGroup();
	}
}/*}}}*/

void Settings::setSampleEnabled(VyEvent s, bool f)/*{{{*/
{
	const QString key = getEventKey(s);
	if (!key.isEmpty())
	{
		cfg.beginGroup("/WavFlags");
		cfg.writeEntry(key, sndp[s] = f);
		cfg.endGroup();
	}
}/*}}}*/
	
void Settings::setScriptFileName(VyEvent e, const QString &f)/*{{{*/
{
	const QString key = getEventKey(e);
	if (!key.isEmpty())
	{
		cfg.beginGroup("/Scripts");
		cfg.writeEntry(key, scr[e] = f);
		cfg.endGroup();
	}
}/*}}}*/

void Settings::setScriptEnabled(VyEvent e, bool f)/*{{{*/
{
	const QString key = getEventKey(e);
	if (!key.isEmpty())
	{
		cfg.beginGroup("/ScriptFlags");
		cfg.writeEntry(key, scrf[e] = f);
		cfg.endGroup();
	}

}/*}}}*/

