/*
 * This file is a part of VyQChat.
 *
 * Copyright (C) 2002-2004 Pawel Stolowski <yogin@linux.bydg.org>
 *
 * VyQChat is free software; you can redestribute it and/or modify it
 * under terms of GNU General Public License by Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY. See GPL for more details.
 */

#include <qpopupmenu.h>
#include "chattab.h"
#include "icons.h"

ChatTab::ChatTab(QWidget *parent): QTabWidget(parent), prevdel(NULL)/*{{{*/
{
	setFocusPolicy(QWidget::NoFocus);
	setSizePolicy(QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding));
	connect(this, SIGNAL(currentChanged(QWidget *)), this, SLOT(tabChanged(QWidget *)));
}/*}}}*/

ChatTab::~ChatTab()/*{{{*/
{
}/*}}}*/

bool ChatTab::isPrivate(const QString &name)/*{{{*/
{
	ChannelMap::iterator it = cmap.find(name);
	return (it!=cmap.end()) ? it.data()->isPrivate() : false;
} /*}}}*/

bool ChatTab::addChannel(const QString &name, const QString &topic) /*{{{*/
{
	QString n = (name[0]=='#') ?  name : "#" + name;

	if (exists(n))
		return false;
	Channel *c = new Channel(this, n, topic);
	
	addTab(c, Icons::icon_channel_normal, n);
	cmap[n] = c;
	showPage(c);
	connect(c, SIGNAL(leave(const QString&)), this, SIGNAL(leave(const QString&)));
	
	return true;
} /*}}}*/

bool ChatTab::del(const QString &name)/*{{{*/
{
	//
	// deleting is delayed to avoid problem with
	// deleting Channel that send 'leave' signal
	if (prevdel)
	{
		delete prevdel;
		prevdel = NULL;
	}
	
	ChannelMap::iterator it = cmap.find(name);
	if (it != cmap.end())
	{
		prevdel = it.data();
		removePage(it.data());
		cmap.remove(it);
		return true;
	}
	return false;
}/*}}}*/

bool ChatTab::delChannel(const QString &name)/*{{{*/
{
	return del((name[0]=='#') ?  name : "#" + name);
}/*}}}*/

bool ChatTab::addPrivate(const QString &name)/*{{{*/
{
	const QString &n = name;
	
	if (exists(n))
		return false;

	Channel *c = new Channel(this, n, true);
	addTab(c, Icons::icon_private, n);
	cmap[n] = c;
	showPage(c);
	
	connect(c, SIGNAL(leave(const QString&)), this, SIGNAL(leave(const QString&)));

	return true;
}/*}}}*/

bool ChatTab::delPrivate(const QString &name)/*{{{*/
{
	return del(name);
}/*}}}*/

QString ChatTab::getTopic(const QString &name)/*{{{*/
{
	ChannelMap::iterator it = cmap.find(name);
	return (it == cmap.end()) ? QString::null : it.data()->getTopic();
}/*}}}*/

bool ChatTab::setTopic(const QString &name, const QString &topic)/*{{{*/
{
	ChannelMap::iterator it = cmap.find(name);
	if (it != cmap.end())
	{
		it.data()->setTopic(topic);
		return true;
	}
	return false;
}/*}}}*/

bool ChatTab::write(const QString &channel, const QString &str, QColor color)/*{{{*/
{
	ChannelMap::iterator it = cmap.find(channel);
	if (it != cmap.end())
	{
		Channel *c = it.data();

		c->setColor(color);
		c->append(str);
		
		if (c != currentPage())
		{
			if (c->isPrivate())
			{
				if (!c->getActivity())
					setTabIconSet(c, Icons::icon_private_alert);
			}
			else
			{
				if (!c->getActivity())
					setTabIconSet(c, Icons::icon_channel_alert);
			}
			c->setActivity(true);
		}
		return true;
	}
	return false;
}/*}}}*/

void ChatTab::clearCurrent()/*{{{*/
{
	currentPage()->clear();
}/*}}}*/

QString ChatTab::channelsList()/*{{{*/
{
	QString list;
	for (ChannelMap::iterator it = cmap.begin(); it != cmap.end(); it++ )
		if (! it.data()->isPrivate())
			list += it.data()->name();
	return list + "#";
}/*}}}*/

void ChatTab::tabChanged(QWidget *widget)/*{{{*/
{
	Channel *c = static_cast<Channel *>(widget);
	
	if (c->isPrivate())
	{
		if (c->getActivity())
			setTabIconSet(c, Icons::icon_private);

	}
	else
	{
		if (c->getActivity())
			setTabIconSet(c, Icons::icon_channel_normal);
	}
	c->setActivity(false);
}/*}}}*/

