//  Copyright (C) 2011 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <config.h>

#include <fstream>
#include <gtkmm.h>
#include <glibmm.h>
#include <giomm.h>
#include "main.h"
#include "vcma-window.h"
#include "ucompose.hpp"
#include "track.h"
#include "track-list.h"
#include "track-list-loader.h"

VCMAWindow* VCMAWindow::create()
{
  VCMAWindow *window;
  Glib::RefPtr<Gtk::Builder> xml = 
    Gtk::Builder::create_from_file(Main::get_glade_path() + "/vcma.gtk");

  xml->get_widget_derived("window", window);
  window->on_window_loaded();
  window->signal_delete_event().connect(
	sigc::mem_fun(*window, &VCMAWindow::on_delete_event));
  return window;
}

VCMAWindow::VCMAWindow(BaseObjectType* baseObject, 
                             const Glib::RefPtr<Gtk::Builder>& xml)
 : Gtk::Window(baseObject), duration_column("Duration", duration_renderer)
{

  xml->get_widget("contents", contents);

  set_icon_from_file(Main::get_data_path() + "/icon.png");
  xml->get_widget("treeview", treeview);
  treeview->signal_button_press_event().connect_notify
    (sigc::mem_fun(*this, &VCMAWindow::on_treeview_clicked));
  songs_list = Gtk::ListStore::create(songs_columns);
  sorted_songs_list = Gtk::TreeModelSort::create(songs_list);
  sorted_songs_list->set_sort_column(songs_columns.duration, Gtk::SORT_ASCENDING);

  treeview->set_model(sorted_songs_list);
  treeview->get_selection()->set_mode(Gtk::SELECTION_SINGLE);
  treeview->set_headers_clickable(true);
  //the name entry box
  duration_column.set_cell_data_func
    (duration_renderer, sigc::mem_fun(*this, &VCMAWindow::cell_data_duration));
  treeview->append_column(duration_column);
  treeview->append_column("Genre", songs_columns.genre);
  treeview->append_column("Artist", songs_columns.artist_name);
  treeview->append_column("Song", songs_columns.song_title);
  treeview->signal_row_activated().connect
    (sigc::mem_fun(*this, &VCMAWindow::on_song_activated));

  Gtk::Menu::MenuList& menulist = treeview_context_menu.items();
  menulist.push_back
    (Gtk::Menu_Helpers::MenuElem
     ("_Play Song", 
      sigc::mem_fun(*this, &VCMAWindow::on_play_song_clicked)));
  menulist.push_back
    (Gtk::Menu_Helpers::MenuElem
     ("_View Album on the Web", sigc::mem_fun(*this, &VCMAWindow::on_view_album_clicked)));
  treeview_context_menu.accelerate(*this);

  treeview->get_column(0)->set_sort_column(songs_columns.duration);
  treeview->get_column(1)->set_sort_column(songs_columns.genre);
  treeview->get_column(1)->set_fixed_width(100);
  treeview->get_column(1)->set_resizable(true);
  treeview->get_column(2)->set_sort_column(songs_columns.song_title);
  treeview->get_column(2)->set_fixed_width(100);
  treeview->get_column(2)->set_resizable(true);
  treeview->get_column(3)->set_sort_column(songs_columns.artist_name);
  treeview->get_column(3)->set_fixed_width(100);
  treeview->get_column(3)->set_resizable(true);

  //menubar callbacks
  xml->get_widget("search_button", search_button);
  search_button->signal_clicked().connect(sigc::mem_fun(*this,&VCMAWindow::on_search_clicked));
  xml->get_widget("quit_button", quit_button);
  quit_button->signal_clicked().connect(sigc::mem_fun(*this,&VCMAWindow::on_exit_clicked));
  xml->get_widget("minute_spinbutton", minute_spinbutton);
  minute_spinbutton->signal_changed().connect
    (sigc::mem_fun(*this, &VCMAWindow::on_minute_spinbutton_changed));
  minute_spinbutton->signal_insert_text().connect
         (sigc::mem_fun(this, &VCMAWindow::on_minute_text_changed));
  xml->get_widget("second_spinbutton", second_spinbutton);
  second_spinbutton->signal_changed().connect
    (sigc::mem_fun(*this, &VCMAWindow::on_second_spinbutton_changed));
  second_spinbutton->signal_insert_text().connect
         (sigc::mem_fun(this, &VCMAWindow::on_second_text_changed));
  xml->get_widget("spinner_container_box", spinner_container_box);
  xml->get_widget("minute_label", minute_label);
  xml->get_widget("seconds_label", seconds_label);
}

void VCMAWindow::on_window_loaded()
{
  d_track_list = new TrackList();
  d_connected = false;
}

VCMAWindow::~VCMAWindow()
{
}

void VCMAWindow::load_catalogue_in_the_background(TrackList *track_list, guint32 secs)
{
  TrackListLoader loader;
  loader.load("http://img.jamendo.com/data/dbdump.en.xml.gz", track_list,
              sigc::bind(sigc::mem_fun(this, &VCMAWindow::on_catalogue_loaded), secs),
              sigc::mem_fun(this, &VCMAWindow::on_load_catalogue_aborted));
}

void VCMAWindow::on_load_catalogue_aborted()
{
  contents->set_sensitive(true);
  unshow_spinner();
}

void VCMAWindow::on_catalogue_loaded(TrackList *track_list, guint32 secs)
{
  contents->set_sensitive(true);
  d_connected = true;
  d_track_list->add(track_list);
  Glib::signal_idle().connect_once(sigc::bind(sigc::mem_fun(*this, &VCMAWindow::search_tracks_in_the_background), d_track_list, secs));
}

bool VCMAWindow::quit()
{
  exit(0); //makes us exit quicker
  return true;
}
    
bool VCMAWindow::on_delete_event(GdkEventAny *event)
{
  return quit();
}

void VCMAWindow::on_exit_clicked()
{
  quit();
}

void VCMAWindow::on_song_activated(const Gtk::TreeModel::Path& path,
                                   Gtk::TreeViewColumn *col)
{
  on_play_song_clicked();
}

void VCMAWindow::on_play_song_clicked()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      guint32 track_id = row[songs_columns.track_id];
      Glib::ustring url = String::ucompose("http://www.jamendo.com/get/track/id/track/audio/redirect/%1/?aue=ogg2", track_id);

      printf("trying %s\n", std::string(url).c_str());
      Glib::ustring cmd = Glib::find_program_in_path("totem") + " " + url;
      Glib::spawn_command_line_async(cmd);
    }
}

void VCMAWindow::on_view_album_clicked()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      guint32 album_id = row[songs_columns.album_id];
      Glib::ustring url = String::ucompose("http://www.jamendo.com/en/album/%1", album_id);
      Glib::ustring cmd = Glib::find_program_in_path("gvfs-open") + " " + url;
      Glib::spawn_command_line_async(cmd);
    }
}

void VCMAWindow::on_search_clicked()
{
  contents->set_sensitive(false);
  show_spinner();
  songs_list->clear();
  guint32 secs = (minute_spinbutton->get_value() * 60) + second_spinbutton->get_value();
  if (d_connected == false)
    Glib::signal_idle().connect_once(sigc::bind(sigc::mem_fun(*this, &VCMAWindow::load_catalogue_in_the_background), d_track_list, secs));
  else
    Glib::signal_idle().connect_once(sigc::bind(sigc::mem_fun(*this, &VCMAWindow::search_tracks_in_the_background), d_track_list, secs));
}

void VCMAWindow::search_tracks_in_the_background(TrackList *track_list, guint32 duration)
{
  guint32 min = duration;
  if (min > 0)
    min--;
  guint32 max = duration;
  if (max < 86400)
    max++;

  TrackList songs;
  track_list->find_tracks_with_duration(min, max, songs);

  guint32 count = 0;
  for (TrackList::const_iterator it = songs.get_tracks().begin(); 
       it != songs.get_tracks().end(); it++)
    {
      add_song_row(**it);
      count++;
    }

  unshow_spinner();
  return;
}
    
void VCMAWindow::add_song_row(const Track &track)
{
  Gtk::TreeIter i = songs_list->append();
  (*i)[songs_columns.duration] = track.get_duration();
  (*i)[songs_columns.genre] = track.get_genre();
  (*i)[songs_columns.song_title] = track.get_song_title();
  (*i)[songs_columns.artist_name] = track.get_artist_name();
  (*i)[songs_columns.album_id] = track.get_album_id();
  (*i)[songs_columns.track_id] = track.get_track_id();
}

void VCMAWindow::cell_data_duration(Gtk::CellRenderer *renderer, const Gtk::TreeIter &i)
{
  Gtk::CellRendererText*cell = dynamic_cast<Gtk::CellRendererText*>(renderer);
  guint32 mins = (*i)[songs_columns.duration] / 60;
  guint32 secs = (*i)[songs_columns.duration] % 60;
  if (secs < 10)
    cell->property_text() = String::ucompose("%1:0%2", mins, secs);
  else
    cell->property_text() = String::ucompose("%1:%2", mins, secs);
}

void VCMAWindow::on_treeview_clicked(GdkEventButton* event)
{
  if (event->type == GDK_BUTTON_PRESS && event->button == 3)
    treeview_context_menu.popup(event->button, event->time);
}
    
void VCMAWindow::on_minute_spinbutton_changed()
{
  if (minute_spinbutton->get_value() == 1)
    minute_label->set_text("minute, and");
  else
    minute_label->set_text("minutes, and");
}

void VCMAWindow::on_second_spinbutton_changed()
{
  if (second_spinbutton->get_value() == 1)
    seconds_label->set_text("second long.");
  else
    seconds_label->set_text("seconds long.");
}

void VCMAWindow::on_second_text_changed(const Glib::ustring &s, int *p)
{
  second_spinbutton->set_value(atoi(second_spinbutton->get_text().c_str()));
  on_second_spinbutton_changed();
}

void VCMAWindow::on_minute_text_changed(const Glib::ustring &s, int *p)
{
  minute_spinbutton->set_value(atoi(minute_spinbutton->get_text().c_str()));
  on_minute_spinbutton_changed();
}

void VCMAWindow::show_spinner()
{
  Gtk::Spinner *spinner = new Gtk::Spinner();
  spinner->property_active() = true;
  spinner_container_box->pack_start(*spinner, true, false, 0);
  spinner_container_box->show_all();
}

void VCMAWindow::unshow_spinner()
{
  std::list<Gtk::Widget*> widgets = spinner_container_box->get_children();
  spinner_container_box->remove(*widgets.back());
  spinner_container_box->show_all();
}
