/* Copyright (c) 2012 - 2013 Samuel Klipphahn
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* $Id: $ */
/**
 * @file
 * @brief Simple Routing Protocoll.
 * @_addtogroup grplwMesh
 */


/* === includes ============================================================ */
#include "lw_mesh.h"
#include "lw_route.h"
#include "lw_cmd.h"
//#include "ioutil.h"
//#include "transceiver.h"
#include <string.h>
#include "timer.h"

/* === macros ============================================================== */

/* === types =============================================================== */

/* === globals ============================================================= */
static uint8_t lw_seq_nr;
static uint16_t lw_short_addr;
static bool (*endpoints[NWK_MAX_ENDPOINTS_AMOUNT])(NWK_DataInd_t *);

static NWK_Tasklist_t *tasks[NWK_BUFFERS_AMOUNT];

/* === prototypes ========================================================== */
static void proc_data_req(NWK_Tasklist_t *task);
static void proc_data_ind(NWK_Tasklist_t *task);

static NWK_Ind_Opt_t get_data_ind_opt(NWK_FrameFormat_t *frame);

static void lw_send_frame(NWK_Tasklist_t *task, uint8_t lw_pl_size,
        uint8_t options);
static void data_confirm(NWK_DataReq_t *req, NWK_Stat_t lw_status);

static void handle_next_task();
static void tasks_init();
static uint8_t get_empty_task_id();

static time_t lw_ack_timer_expired(timer_arg_t task_id);
static void lw_send_ack_frame(NWK_Tasklist_t *task);

static inline uint8_t get_lw_endpts(uint8_t src_endpoint
        , uint8_t dst_endpoint);
static inline uint8_t get_src_ep(uint8_t lw_endpts);
static inline uint8_t get_dst_ep(uint8_t lw_endpts);

/* === functions =========================================================== */

void lw_mesh_init(uint16_t pan_id, uint16_t short_addr, uint8_t channel)
{
    /* init global variables */
    lw_seq_nr = 0;
    lw_short_addr = short_addr;
    tasks_init();
    timer_init();

    int i=0;
    for(; i < NWK_MAX_ENDPOINTS_AMOUNT; i++)
    {
        endpoints[i] = NULL;
    }

    /* init lw_cmd, lw_route and lw_mac*/
    lwc_init();
    lwr_init();
    lwm_init(pan_id, short_addr, channel);
}

void lw_mesh_task_handler()
{
    /* handle all tasks in queue */
    uint8_t handled = 0;
    while(handled < NWK_BUFFERS_AMOUNT)
    {
        handle_next_task();
        handled++;
    }
}

/** handles next task in tasks-array could be NWK_TASK_NO_TASK */
static void handle_next_task()
{
    static uint8_t next_task = 0;
    switch(tasks[next_task]->type)
    {
    case NWK_TASK_DATA_REQ:
        proc_data_req(tasks[next_task]);
        break;

    case NWK_TASK_TX_PENDING:
        /* gets handled in lw_tx_done by call back, so keep in queue */
        break;

    case NWK_TASK_ACK_PENDING:
        /* a data request waiting for acknowledgment, so keep it in the queue
         * it is removed by the call back function */
        break;

    case NWK_TASK_DATA_IND:
        proc_data_ind(tasks[next_task]);
        break;

    case NWK_TASK_NO_TASK:
        break;
    }

    /* check for overflow */
    if(next_task == NWK_BUFFERS_AMOUNT - 1)
    {
        next_task = 0;
    } else {
        next_task++;
    }
}

/** function for sending frames */
void lw_mesh_data_req(NWK_DataReq_t *data_req)
{
    uint8_t this_task = get_empty_task_id();

    if(this_task != NWK_NO_TASK_ID)
    {
        uint8_t acceptedSize;
        if(data_req->size <= NWK_MAX_PAYLOAD_SIZE)
        {
            acceptedSize = data_req->size;
        } else {
            acceptedSize = NWK_MAX_PAYLOAD_SIZE;
        }

        tasks[this_task]->type   = NWK_TASK_DATA_REQ;

        /* copy data for data request */
        NWK_FrameFormat_t *frame = malloc(sizeof(NWK_FrameFormat_t));
        memcpy(&frame->lw_payload, data_req->data, acceptedSize);
        tasks[this_task]->frame  = frame;

        NWK_DataReq_t *request  = &tasks[this_task]->info.req;
        request->dstAddr        = data_req->dstAddr;
        request->dstEndpoint    = data_req->dstEndpoint;
        request->srcEndpoint    = data_req->srcEndpoint;
        request->options        = data_req->options;
        request->data           = frame->lw_payload;
        request->size           = acceptedSize;
        request->confirm        = data_req->confirm;
    } else {
        /* no buffer available, so call back application */
        data_confirm(data_req, NWK_OUT_OF_MEMORY_STATUS);
    }
}

/** function for endpoint registration (register to receive frames) */
void lw_mesh_open_endpoint(uint8_t endpoint_id,
        bool (*data_ind_callback)(NWK_DataInd_t *))
{
    if(endpoint_id < NWK_MAX_ENDPOINTS_AMOUNT)
    {
        endpoints[endpoint_id] = data_ind_callback;
    }
}

void lw_mesh_set_ack_control(uint8_t control)
{
    lwc_set_ack_control(control);
}

static void proc_data_req(NWK_Tasklist_t *task)
{
    /* application wants to send data */
    task->frame->lw_fctl    = task->info.req.options & NWK_NFCTL_MASK;
    task->frame->lw_seq     = lw_seq_nr++;
    task->frame->lw_srcAddr = lw_short_addr;
    task->frame->lw_dstAddr = task->info.req.dstAddr;
    task->frame->lw_endpts  = get_lw_endpts(task->info.req.srcEndpoint,
            task->info.req.dstEndpoint);
    if(task->info.req.options & NWK_OPT_LINK_LOCAL)
    {
        task->frame->m_dstAddr  = task->info.req.dstAddr;
    } else {
        task->frame->m_dstAddr = lwr_get_next_hop_addr(task->info.req.dstAddr);
    }

    /* remember frame information in duplicate rejection table, do not send, if
     * table is full */
    if(!lwr_is_duplicate_frame(task->frame->lw_seq, task->frame->lw_srcAddr))
    {
        lw_send_frame(task, task->info.req.size, task->info.req.options);
    } else {
        data_confirm(&task->info.req, NWK_ERROR_STATUS);
        free(task->frame);
        task->type = NWK_TASK_NO_TASK;
    }
}

static void lw_send_frame(NWK_Tasklist_t *task, uint8_t lw_pl_size,
        uint8_t options)
{
    radio_cca_t cca = lwm_send_frame(lw_pl_size, task->frame,
            task->task_id, options);
    if(cca == RADIO_CCA_FREE)
    {
        /* wait for tx callback */
        task->type = NWK_TASK_TX_PENDING;
    } else {
        /* report access failure to application */
        data_confirm(&task->info.req, NWK_PHY_CHANNEL_ACCESS_FAILURE_STATUS);
        free(task->frame);
        task->type = NWK_TASK_NO_TASK;
    }
}

void lw_data_indication(NWK_FrameFormat_t *frame, uint8_t lw_payload_size,
        uint8_t lqi, uint8_t rssi)
{
    if(!lwr_is_duplicate_frame(frame->lw_seq, frame->lw_srcAddr))
    {
        uint8_t this_task = get_empty_task_id();
        if(this_task != NWK_NO_TASK_ID)
        {
            tasks[this_task]->type  = NWK_TASK_DATA_IND;
            tasks[this_task]->frame = frame;
            NWK_DataInd_t *d_ind  = &tasks[this_task]->info.ind;

            d_ind->srcAddr        = frame->lw_srcAddr;
            d_ind->dstEndpoint    = get_dst_ep(frame->lw_endpts);
            d_ind->srcEndpoint    = get_src_ep(frame->lw_endpts);
            d_ind->options        = get_data_ind_opt(frame);
            d_ind->data           = frame->lw_payload;
            d_ind->size           = lw_payload_size;
            d_ind->lqi            = lqi;
            d_ind->rssi           = rssi;
        } else {
            /* no buffer for frame reception */
            free(frame);
        }
    } else {
        /* reject duplicate frame */
        free(frame);
    }
}

static NWK_Ind_Opt_t get_data_ind_opt(NWK_FrameFormat_t *frame)
{
    NWK_Ind_Opt_t opt = 0;
    if(frame->lw_fctl & NWK_OPT_ACK_REQUEST)
    {
        opt |= NWK_IND_OPT_ACK_REQUEST;
    }
    if(frame->lw_fctl & NWK_OPT_ENABLE_SECURITY)
    {
        opt |= NWK_IND_OPT_SECURED;
    }
    if(frame->m_dstAddr & NWK_BROADCAST_ADDR)
    {
        opt |= NWK_IND_OPT_BROADCAST;
    }
    if(frame->lw_srcAddr == frame->m_srcAddr)
    {
        opt |= NWK_IND_OPT_LOCAL;
    }
    if(frame->m_pid == NWK_BROADCAST_PID)
    {
        opt |= NWK_IND_OPT_BROADCAST_PAN_ID;
    }
    if(frame->lw_fctl & NWK_OPT_LINK_LOCAL)
    {
        opt |= NWK_IND_OPT_LINK_LOCAL;
    }
    return opt;
}

static void proc_data_ind(NWK_Tasklist_t *task)
{
    /* we received a frame, decide about routing and application call back */
    NWK_DataInd_t *d_ind = &task->info.ind;
    lwr_save_routes(task->frame->lw_srcAddr, task->frame->m_srcAddr,
            d_ind->lqi);
    if(task->frame->lw_dstAddr == lw_short_addr)
    {
        /* this is a frame for this node, so call back application*/
        if(endpoints[d_ind->dstEndpoint] != NULL)
        {
            /* known endpoint */
            bool ack_req = endpoints[d_ind->dstEndpoint](d_ind);
            if(ack_req
                    || (task->frame->lw_fctl & NWK_OPT_ACK_REQUEST)
                    || task->frame->m_dstAddr == NWK_BROADCAST_ADDR)
            {
                /* send acknowledgment */
                lw_send_ack_frame(task);
                /* caution: indication buffer is reused for ack frame */
                /* TODO: maybe find another way without buffer reuse */
            }else {
                free(task->frame);
                task->type = NWK_TASK_NO_TASK;
            }
        } else {
            /* unknown endpoint, discard frame*/
            free(task->frame);
            task->type = NWK_TASK_NO_TASK;
        }
    } else {
        /* not for this node check options if frame propagation is allowed */
        if(!(task->info.ind.options & (NWK_IND_OPT_LINK_LOCAL)))
        {
            /* propagate frame */
            if(task->frame->m_dstAddr == NWK_BROADCAST_ADDR)
            {
                /* broadcast frame */
                uint8_t pl_size = task->info.ind.size;
                task->type = NWK_TASK_DATA_REQ;
                task->info.req.dstAddr       = task->frame->lw_dstAddr;
                task->info.req.dstEndpoint =
                                        get_dst_ep(task->frame->lw_endpts);
                task->info.req.srcEndpoint =
                                        get_src_ep(task->frame->lw_endpts);
                task->info.req.options = task->frame->lw_fctl & NWK_NFCTL_MASK;
                task->info.req.data          = task->frame->lw_payload;
                task->info.req.size          = pl_size;
                task->info.req.confirm       = lwr_data_confirm;
                lw_send_frame(task, task->info.req.size
                        , task->info.req.options);
            } else {
                /* route frame */
                if(lwr_prep_r_frame(task))
                {
                    uint8_t pl_size = task->info.ind.size;
                    task->type = NWK_TASK_DATA_REQ;
                    task->info.req.dstAddr       = task->frame->lw_dstAddr;
                    task->info.req.dstEndpoint =
                                        get_dst_ep(task->frame->lw_endpts);
                    task->info.req.srcEndpoint =
                                        get_src_ep(task->frame->lw_endpts);
                    task->info.req.options =
                                        task->frame->lw_fctl & NWK_NFCTL_MASK;
                    task->info.req.data          = task->frame->lw_payload;
                    task->info.req.size          = pl_size;
                    task->info.req.confirm       = lwr_data_confirm;
                    lw_send_frame(task, task->info.req.size
                            , task->info.req.options);
                }
            }
        }
    }
}

static time_t lw_ack_timer_expired(timer_arg_t task_id)
{
    if(task_id != NWK_NO_TASK_ID
            && tasks[task_id]->type == NWK_TASK_ACK_PENDING)
    {
        data_confirm(&tasks[task_id]->info.req, NWK_NO_ACK_STATUS);
        free(tasks[task_id]->frame);
        tasks[task_id]->type = NWK_TASK_NO_TASK;
    }
    return 0;
}

void lw_ack_received(uint8_t lw_seq, uint8_t lw_cmd_ctrl_message)
{
    uint8_t task_id = 0;
    for(; task_id < NWK_BUFFERS_AMOUNT; task_id++)
    {
        if(tasks[task_id]->type == NWK_TASK_ACK_PENDING
                && tasks[task_id]->frame->lw_seq == lw_seq)
        {
            timer_stop(tasks[task_id]->timer_id);
            tasks[task_id]->info.req.control = lw_cmd_ctrl_message;
            data_confirm(&tasks[task_id]->info.req, NWK_SUCCESS_STATUS);
            free(tasks[task_id]->frame);
            tasks[task_id]->type = NWK_TASK_NO_TASK;
            break;
        }
    }
}

void lw_tx_done(uint8_t task_id, radio_tx_done_t tx_status)
{
    if(task_id != NWK_NO_TASK_ID
            && tasks[task_id]->type == NWK_TASK_TX_PENDING)
    {
        NWK_Tasklist_t *task = tasks[task_id];
        switch(tx_status)
        {
        case TX_OK:
            /* transmission completed successfully */
            if((task->frame->lw_fctl & NWK_OPT_ACK_REQUEST)
                    || task->frame->m_dstAddr == NWK_BROADCAST_ADDR)
            {
                /* wind up timer and wait for network acknowledgment */
                task->timer_id = timer_start(lw_ack_timer_expired,
                        MSEC(NWK_ACK_WAIT_TIME), task_id);
                task->type = NWK_TASK_ACK_PENDING;
            }else {
                /* no network ack requested */
                data_confirm(&task->info.req, NWK_SUCCESS_STATUS);
                free(task->frame);
                task->type = NWK_TASK_NO_TASK;
            }
            break;

        case TX_CCA_FAIL:
            /* channel was busy (TX_AUTO only) */
            data_confirm(&task->info.req
                    , NWK_PHY_CHANNEL_ACCESS_FAILURE_STATUS);
            free(tasks[task_id]->frame);
            task->type = NWK_TASK_NO_TASK;
            break;

        case TX_NO_ACK:
            /* no ACK received (TX_AUTO only) */
            data_confirm(&task->info.req, NWK_PHY_NO_ACK_STATUS);
            /*TODO: update routing information to avoid this node */
            /*TODO: maybe resend frame */
            free(task->frame);
            task->type = NWK_TASK_NO_TASK;
            break;

        case TX_FAIL:
            /* unexpected error */
            data_confirm(&task->info.req, NWK_ERROR_STATUS);
            free(task->frame);
            task->type = NWK_TASK_NO_TASK;
            break;
        }
    }
}

static void lw_send_ack_frame(NWK_Tasklist_t *task)
{
    task->type              = NWK_TASK_DATA_REQ;
    task->info.req.dstAddr       = task->frame->lw_srcAddr;
    task->info.req.dstEndpoint   = NWK_ENDPOINT_ID_LW_COMMAND;
    task->info.req.srcEndpoint   = NWK_ENDPOINT_ID_LW_COMMAND;
    task->info.req.options       = 0;
    task->info.req.data          = task->frame->lw_payload;
    task->info.req.size          = sizeof(NWK_cmd_ack_t);
    task->info.req.confirm       = lwc_data_confirm;

    lwc_prep_ack_pl(task->frame, (NWK_cmd_ack_t *) &task->frame->lw_payload);

    proc_data_req(task);
}

static void data_confirm(NWK_DataReq_t *req, NWK_Stat_t lw_status)
{
    req->status = lw_status;

    /* inform routing */
    lwr_data_confirm(req);

    /* inform application */
    if(req->confirm != NULL)
    {
        req->confirm(req);
    }
}

static void tasks_init()
{
    uint8_t task_id = 0;
    for(; task_id < NWK_BUFFERS_AMOUNT; task_id++)
    {
        tasks[task_id] = malloc(sizeof(NWK_Tasklist_t));
        tasks[task_id]->task_id = task_id;
        tasks[task_id]->type = NWK_TASK_NO_TASK;
    }
}

static uint8_t get_empty_task_id()
{
    uint8_t ret = 0;
    cli();
    for(; ret < NWK_BUFFERS_AMOUNT; ret++){
        if(tasks[ret]->type == NWK_TASK_NO_TASK)
        {
            break;
        }
    }
    if(ret >= NWK_BUFFERS_AMOUNT)
    {
        ret = NWK_NO_TASK_ID;
    }
    sei();
    return ret;
}

static inline uint8_t get_lw_endpts(uint8_t src_ep, uint8_t dst_ep)
{
    return (((0x0F & src_ep) << 4) | (0x0F & dst_ep));
}

static inline uint8_t get_src_ep(uint8_t lw_endpts)
{
    return ((0xF0 & lw_endpts) >> 4);
}

static inline uint8_t get_dst_ep(uint8_t lw_endpts)
{
    return (0x0F & lw_endpts);
}
