/* Copyright (c) 2011 Axel Wachtler, Daniel Thiele
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* $Id$ */
/**
 * @file
 * @brief ....
 * @_addtogroup grpApp...
 */


/* === includes ============================================================ */
#include <avr/io.h>
#include <avr/interrupt.h>

/* === macros ============================================================== */

/* === types =============================================================== */

/* === globals ============================================================= */
extern volatile uint8_t joystick;

/* === prototypes ========================================================== */

/* === functions =========================================================== */


ISR(TIMER1_OVF_vect, ISR_BLOCK)
{
}

ISR(TIMER1_COMPA_vect, ISR_BLOCK)
{
}

ISR(TIMER1_COMPB_vect, ISR_BLOCK)
{
}

ISR(TIMER0_COMP_vect, ISR_BLOCK)
{
}

ISR(ADC_vect, ISR_BLOCK)
{
	if(0 == (ADMUX & 0x1F)){ /* microphone */
		OCR2A = (127 - (ADCH & 0x7F)) / 8;
	}else if(1 == (ADMUX & 0x1F)){ /* joystick */
		if((0 <= ADCH) && (30 > ADCH)){
			joystick = _BV(0);
		}else if((30 <= ADCH) && (96 > ADCH)){
			joystick = _BV(1);
		}else if((96 <= ADCH) && (166 > ADCH)){
			joystick = _BV(2);
		}else if((166 <= ADCH) && (240 > ADCH)){
			joystick = _BV(3);
		}else{ /* no button */
			joystick = 0;
		}
	}
}

void audio_powerup(void)
{
	DDRE |= _BV(PE7);
	PORTE &= ~_BV(PE7);
}

void audio_powerdown(void)
{
	PORTE |= _BV(PE7);
}

void audio_init(void)
{
	DIDR0 |= _BV(ADC0D);
	ADCSRA = _BV(ADEN) | _BV(ADATE) | _BV(ADIE) | _BV(ADPS2);
	ADCSRB = _BV(ADTS1) | _BV(ADTS0); /* Timer0 CompA as trigger source */

	DDRB |= _BV(PB5); /* speaker PWM output */

	PORTE |= _BV(PE7); /* initially off */
	DDRE |= _BV(PE7);

	TCCR1A = 0;
	TCCR1B = 0;
	TCCR1C = 0;
	OCR1A = 0;
	OCR1B = 0;
	ICR1 = 0;
	TIMSK1 = 0;
	TIFR1 = 0xFF;
	
	TCCR1A = _BV(COM1A1) | _BV(COM1A0) | _BV(WGM10);
	TCCR1B = _BV(CS10);
}

void joystick_sampling_start(uint16_t freq_Hz)
{
	uint8_t ps; /* prescaler value */
	uint8_t cs; /* prescaler register setting */
	if(freq_Hz > 4000){
		ps=8;
		cs= _BV(CS01);
	}else{
		ps=64;
		cs = _BV(CS01) | _BV(CS00);
	}

	ADMUX = _BV(REFS0) | _BV(ADLAR) | _BV(MUX0); /* reference AVCC, ADC1 as input */
	OCR0A = (F_CPU / (2UL*ps*freq_Hz)) - 1;
	TCCR0A = _BV(WGM01) | cs;
	TIMSK0 |= _BV(OCIE0A);
}

/*
 * \brief Start Sampling
 *
 * @param freq_Hz Sampling frequency [Hz]
 *
 */
void audio_sampling_start(uint16_t freq_Hz)
{
	uint8_t ps; /* prescaler value */
	uint8_t cs; /* prescaler register setting */
	if(freq_Hz > 4000){
		ps=8;
		cs= _BV(CS01);
	}else{
		ps=64;
		cs = _BV(CS01) | _BV(CS00);
	}
	ADMUX = _BV(REFS0) | _BV(ADLAR); /* reference AVCC, ADC0 as input */


	OCR0A = (F_CPU / (2UL*ps*freq_Hz)) - 1;
	TCCR0A = _BV(WGM01) | cs;
	TIMSK0 = _BV(OCIE0A);
}

/*
 * \brief Stop Sampling
 */
void audio_sampling_stop(void)
{
	TCCR0A = 0; /* stop timer */
	TIMSK0 &= _BV(OCIE0A);
}

/* EOF */
