#   Copyright (c) 2011 Axel Wachtler
#   All rights reserved.
#
#   Redistribution and use in source and binary forms, with or without
#   modification, are permitted provided that the following conditions
#   are met:
#
#   * Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#   * Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in the
#     documentation and/or other materials provided with the distribution.
#   * Neither the name of the authors nor the names of its contributors
#     may be used to endorse or promote products derived from this software
#     without specific prior written permission.
#
#   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
#   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
#   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
#   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
#   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
#   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
#   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
#   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
#   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
#   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
#   POSSIBILITY OF SUCH DAMAGE.

# $Id$
"""

uracoli Sniffer Proxy

The script dumps a PCAP file with the frames received
from the serial port to stdout.

Usage: 
    python sniffer.py [OPTIONS] 
    
Options:
    -p PORT:
        initial data rate.
    -c CHANNEL:
        initial channel to be used.
    -r RATE:
        initial data rate to be used.
    -h:
        Show help and exit.
    -V:
        show version and exit.

Examples:
    python sniffer.py -p COM1 -c 17 > dump.pcap
    
    Open the GUI and portdump the incoming frames to the file 
    pcap.
    
    python sniffer.py -p COM1 -c 17 | wireshark -i - -k
    
     Open the GUI and send the frames to wireshark.

"""
VERSION = "0.42"


# === import ==================================================================
import sys, os, ctypes
import traceback, getopt, threading

import ieee802154_base as base
from sniffer_io import PortIn

from Tkinter import *

CTX = dict(rate = None, channel = None, port = None)
DEVOUT = None


## Socket class
# this class umplemenportts the interface to the
# libpcap/wireshark application.
class StdOut(base.PcapBase):
    VERBOSE = 0
    ## conctructor
    def open(self, sockname, devin):
        self.stdout = os.fdopen(sys.stdout.fileno(),"wb")    
        self.InputDev = devin
        self.TxThread = threading.Thread(target = self.__tx__)
        self.TxThread.setDaemon(1)
        self.TxThread.start()
        self.TxThread.setName("TX")

    ## closing the socket
    def close(self):
        if self.TxThread != None:
            self.InputDev.reset()
            self.TxThread.join(self.TMO)
            self.TxThread = None

    ## regular messagporte
    def message(self,lvl,msg,*args):
        if self.VERBOSE >= lvl:
            prompt = "+"*(lvl)
            msg = msg%args
            msg = msg.replace("\n","\n> ")
            sys.stderr(prompt+msg)
            sys.stderr.flush()

    ## reopen fifo and place pcap header
    def reopen(self):
        self.InputDev.reset()
        self.stdout.write(self.InputDev.pcap_get_header())
        self.stdout.flush()
        self.InputDev.sniff()

    ## periodically send data retrived from the input device
    # (a instance of @ref PcapFile or @ref PcapFile).
    def __tx__(self):
        cnt = 0        
        self.reopen()
        while 1:
            try:
                d = self.InputDev.read_packet()
                if d:
                    sys.stdout.write(d)
                    sys.stdout.flush()
                    self.message(1,"cnt=%d pack=%d len=%d", cnt, self.InputDev.FCNT, len(d))
                    if self.VERBOSE > 1:
                        self.message(self.VERBOSE,":%08d:P:% 3d:%s",
                                     self.InputDev.FCNT,len(d),str(self.InputDev))
                        self.message(self.VERBOSE,":".join(map(hex,map(ord,d))))
                    cnt += 1
            except:
                traceback.print_exc()
                #self.exc_handler("End Thread cnt=%d" % cnt)
                break

class CtrlGui(threading.Thread):
    def __init__(self, devin, devout):
        self.devin = devin
        self.devout = devout
        self.root=Tk()
        
        master = self.root
        master.title("uracoli sniffer")
        devcfg = self.devin.info()
        
        width=30
        Label(master, 
              text="uracoli Sniffer Interface\n\n"\
                   "%(platform)s @ %(port)s\n" % devcfg).pack()

        b = Button(master,text='REOPEN', command=self.reopen, width = width)
        b.pack()

        # select the channel
        self.VarChannel = StringVar(master)
        self.VarChannel.set(devcfg['chan'])
        of = Frame(master, width=width)
        Label(of,text="Channel:").pack(side=LEFT, expand=YES, fill=BOTH)
        self.OmChannel = OptionMenu( of, self.VarChannel, 
                            *tuple(devcfg['clist']), 
                            command=self.update_channel)
        self.OmChannel.pack()
        of.pack()

        # select the rate
        rates = devcfg['rates'].split()
        self.VarRate = StringVar(master)
        self.VarRate.set(str(devcfg['crate']))
        of = Frame(master, width=width)
        Label(of, text="Rate:").pack(side=LEFT, expand=YES, fill=X)
        self.OmRate = OptionMenu( of, self.VarRate,
                            *tuple(rates), 
                            command=self.update_rate)
        self.OmRate.pack(expand=YES, fill=X)
        of.pack()
        # message window
        tf = Text(master,width=width,height=10)
        tf.pack(expand=YES, fill=X)
        self.TxtLog = tf 
        b = Button(master, text='QUIT', width = width, command = master.quit)
        b.pack()
        threading.Thread.__init__(self)
        
    def message(self,txt):
        self.TxtLog.configure(state=NORMAL)                    
        self.TxtLog.insert(END, txt.strip()+"\n")
        self.TxtLog.yview(END)
        self.TxtLog.configure(state=DISABLED)
        
    def update_channel(self, channel):
        self.message("set channel %s\n" % channel)
        self.devin.set_channel(channel)

    def update_rate(self, rate):
        self.message("portset rate %s\n" % rate)
        self.devin.set_rate(rate)
        
    def reopen(self):
        self.message("send PCAP header to wireshark")
        self.devout.reopen()

def interactive_inspect_mode():
    flagPtr = ctypes.cast(ctypes.pythonapi.Py_InteractiveFlag, 
                         ctypes.POINTER(ctypes.c_int))
    return flagPtr.contents.value > 0 or bool(os.environ.get("PYTHONINSPECT",False))

if __name__ == "__main__":
    try:
        import rlcompleter
        import readline
        readline.parse_and_bind("tab:complete")
    except:
        pass
    opts,args = getopt.getopt(sys.argv[1:],"hvp:c:r:")
    do_exit = False
    for o,v in opts:
        if o == "-p":
            CTX["port"] = v
        elif o == "-r":
            CTX["rate"] = v
        elif o == "-c":
            CTX["channel"] = eval(v)
        elif o == "-h":
            print __doc__
            do_exit = True
        elif o == "-v":
            print VERSION
            do_exit = True
    if do_exit:
        sys.exit()        
                        
    
    DEVIN = PortIn()
    DEVIN.open(CTX["port"])
    DEVOUT = StdOut()
    DEVOUT.open("xxx",DEVIN)
    DEVOUT.verbose = 2
    if CTX["channel"]:
        DEVIN.set_channel(CTX["channel"])
    if CTX["rate"]:
        DEVIN.set_channel(CTX["rate"])

    CGUI = CtrlGui(DEVIN, DEVOUT)
    if not interactive_inspect_mode():
        CGUI.root.mainloop()

