#   Copyright (c) 2008 Axel Wachtler
#   All rights reserved.
#
#   Redistribution and use in source and binary forms, with or without
#   modification, are permitted provided that the following conditions
#   are met:
#
#   * Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#   * Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in the
#     documentation and/or other materials provided with the distribution.
#   * Neither the name of the authors nor the names of its contributors
#     may be used to endorse or promote products derived from this software
#     without specific prior written permission.
#
#   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
#   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
#   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
#   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
#   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
#   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
#   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
#   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
#   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
#   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
#   POSSIBILITY OF SUCH DAMAGE.

# $Id: ieee802154.py,v 1.4 2008/07/02 20:10:17 awachtler Exp $
##
# @file
# @brief capture converter script
#
# @par Usage of the Adapter Script
#
#   <tt>python ieee802154.py -p PORT|-r FILE [OPTIONS]</tt>
#
# @b Options
#
# - @c -i socket name [ieee802154]
# - @c -xxxx
#

# === import ==================================================================
import time, socket, sys, os, time
import threading, traceback, struct, getopt, Queue, serial

# === globals =================================================================

## name of the file socket, overwritten by -i option
SOCKNAME='ieee802154'
## name of the capture file, overwritten by -r option
CAPFILE=None
## name of the capture port, overwritten by -p option
PORT=None
## single step flag
SINGLESTEP = 0
## channel number to scan
CHANNEL = 15

VERBOSE = 0
MAXPACKETS = 0

# === classes =================================================================
##
# @brief Base Class for PcapFile and PcapPort
class PcapBase:
    TMO = 1
    VERBOSE = 0
    def __init__(self):
        self.fname = None
    ## reporting an error to stderr.
    def error(self,msg,*args):
        sys.stderr.write("ERROR:"+msg%args+"\n")
        sys.stderr.flush()
    ## regular message
    def message(self,lvl,msg,*args):
        if self.VERBOSE >= lvl:
            print "+"*(lvl+1),msg%args
            sys.stdout.flush()
    def setverbose(self,verbose):
        self.VERBOSE = verbose

    def exc_handler(self, diag):
        self.message(0,diag)
        if self.VERBOSE>0:
            traceback.print_exc()
##
# Class that reads a capture file.
#
class PcapFile(PcapBase):
    ## Cstruct format string for pcap file header
    PCAP_HEADER="LHHLLLL"
    ## Cstruct format string for pcap packet header
    PCAP_PACKET="QLL"

    def open(self, fname, mode="r"):
        self.fname = fname
        self.fh = open(fname,mode)
        d = self.fh.read()
        self.header,self.frames = self.pcap_parse_data(d)
        self.frameidx = 0

    def close(self):
        self.fh.close()

    def set_channel(self, channel):
        self.error("can not set channel on a file")

    def read_packet(self):
        try:
            ret = self.frames[self.frameidx]
            self.frameidx += 1
            self.FCNT += 1
        except:
            ret = None
        return ret

    def pcap_parse_data(self,data):
        ret = []
        hsz = struct.calcsize(self.PCAP_HEADER)
        psz = struct.calcsize(self.PCAP_PACKET)
        o = hsz
        hdr, data = data[:o], data[o:]
        while(len(data) > psz):
            o = psz
            ph, data = data[:o], data[o:]
            ts,cl,fc = struct.unpack(self.PCAP_PACKET , ph)
            if cl != fc:
                self.message(1,"MISMATCH cl=%x fc=%x", cl, fc)
            o = cl
            frm, data = data[:o], data[o:]
            l = len(frm)
            frm = chr(1) + frm + chr(4)
            ret.append(frm)
        return hdr,ret


##
# Class that reads packet data from a serial interface.
#
class PcapPort(PcapBase):
    FCNT = 0
    UNSYNC = 0
    SYNCED = 1
    IDXERR = 0
    BAUDRATE = 38400
    def __init__(self):
        PcapBase.__init__(self)
        self.RxThread = None
        self.RxQueue = Queue.Queue()
        self.channel = 0

    def open(self, fname, mode = 'r'):
        if self.RxThread:
            self.RxThread.join()
        self.fname = fname
        self.sport = serial.Serial(fname, self.BAUDRATE)
        self.sport.open()
        self.RxThread=threading.Thread(target = self.__rx__)
        self.RxThread.setDaemon(1)
        self.RxThread.start()
        self.init()

    def init(self):
        t = int(time.time())
        print "timeset %s" % t
        self.sport.write("\nidle\ntimeset %s\nchan %d\nsniff\n" % (t,self.channel))

    def close(self):
        self.RxThread.join(self.TMO)
        self.RxThread = None
        self.sport.close()

    def __rx__(self):
        state = self.UNSYNC
        frm = ""
        while 1:
            frm += self.sport.read()
            if state == self.UNSYNC:
                p = self.sync_search(frm)
                if type(p) == type(1):
                    frm = frm[p:]
                    state = self.SYNCED
            if state == self.SYNCED:
                frm = self.packetizer(frm)

    def sync_search(self,frm):
        ret = None
        p = 0
        for idx in range(frm.count('\x01')):
            try:
                p += frm[p:].index('\x01')
                l = ord(frm[p+1])
                pe = l + p + 2
                self.message(2,"syncing : i=%d, p=%d, l=%d, pe=%d len=%d",idx,p,l,pe,len(frm))
                if pe < len(frm):
                    if(ord(frm[pe]) == 4):
                        ret = p
                        self.message(1,"synced : i=%d, p=%d, l=%d, pe=%d len=%d", idx,p,l,pe,len(frm))
                        break
                p += 1
            except:
                self.exc_handler("sync_search")
                break
        return ret

    def packetizer(self,frm):
        ret = self.SYNCED
        while len(frm) > 2:
            try:
                l = ord(frm[1])
                if frm[l+2] == '\x04':
                    packet,frm = frm[:l+3],frm[l+3:]
                    self.RxQueue.put(packet)
                    self.message(1,"Found Packet l=%d qsize: %d", len(packet),self.RxQueue.qsize())
                else:
                    ret = self.UNSYNC
                    raise "UnSync"
            except IndexError:
                self.IDXERR += 1
                break;
            except:
                self.exc_handler("packetizer,Other")
                break
        return frm

    def set_channel(self, channel):
        self.channel = channel
        self.init()

    def read_packet(self):
        if self.RxQueue.empty():
            ret = None
        else:
            ret = self.RxQueue.get()
        return ret

class PcapSocket(PcapBase):
    def __init__(self, sockname, device):
        self.sockname = sockname
        self.InputDev = device
        if os.path.exists(self.sockname):
            self.message(1, "remove old socket")
            os.remove(self.sockname)
        self.TxSocket = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
        self.TxSocket.bind(self.sockname)
        self.TxThread=threading.Thread(target = self.__tx__)
        self.TxThread.setDaemon(1)
        self.TxThread.start()

    def close(self):
        self.TxSocket.close()
        self.TxThread.join(self.TMO)
        self.TxThread = None

    def __tx__(self):
        run = 1
        cnt = 0
        while run:
            self.message(1,"listen at %s", self.sockname)
            self.TxSocket.listen(1)
            conn, addr = self.TxSocket.accept()
            self.message(1,"accepted connection sock=%s addr=%s", conn, addr)
            while 1:
                try:
                    d = self.InputDev.read_packet()
                    if d:
                        conn.sendall(d)

                        self.message(1,"cnt=%d pack=%d len=%d", cnt, self.InputDev.FCNT, len(d))
                        if self.VERBOSE > 1:
                            self.message(self.VERBOSE,":%08d:P:% 3d:%s", self.InputDev.FCNT,len(d),str(self.InputDev))
                            self.message(self.VERBOSE,":".join(map(hex,map(ord,d))))
                        cnt += 1
                except socket.error:
                    si = sys.exc_info()
                    self.exc_handler("SocketError: %s" % str(si[1].args))
                    break
                except:
                    self.exc_handler("End Thread cnt=%d" % cnt)
                    run = 0
                    break
        self.message(1,"Packets: fcnt=%d", self.InputDev.FCNT)
        os.remove(SOCKNAME)


# === functions ===============================================================
def do_quit(device,outsocket):
    device.close()
    outsocket.close()

def process_command_line_args(argv):
    global SOCKNAME, CAPFILE, PORT, SINGLESTEP, VERBOSE, MAXPACKETS
    opts,args = getopt.getopt(argv,"i:p:r:st:vN:c:")
    for o,v in opts:
        print o,v
        if o == "-i":
            v = v.strip()
            if v.find("ieee802154")<0:
                print "Interface needs to have string ieee802154 in name"
                sys.exit(1)
            if os.path.isfile(v) or os.path.islink(v):
                print "Can't overwrite existing file or link with socket"
                sys.exit(1)
            SOCKNAME = v.strip()
        if o == "-r":
            CAPFILE = v.strip()
            PORT = None
        if o == "-p":
            CAPFILE = None
            PORT = v.strip()
        if o == "-s":
            SINGLESTEP = -1
        if o == "-t":
            SINGLESTEP = eval(v)
        if o == "-v":
            VERBOSE += 1
        if o == "-N":
            MAXPACKETS = eval(v)
        if o == "-c":
            CHANNEL = eval(v)

    if not CAPFILE and not PORT:
        print "You need to specifiy either a file (-r) or a port (-p) as data source"
        sys.exit(1)




# === init ====================================================================

if __name__ == "__main__":
    try:
        import rlcompleter
        import readline
        readline.parse_and_bind("tab:complete")
    except:
        pass

    process_command_line_args(sys.argv[1:])

    # create input reader classes
    if CAPFILE:
        device = PcapFile()
        device.open(CAPFILE)
    if PORT:
        device = PcapPort()
        device.open(PORT)
        device.set_channel(CHANNEL)
    outsocket = PcapSocket(SOCKNAME, device)

    outsocket.setverbose(VERBOSE)
    device.setverbose(VERBOSE)

    run = 1
    while run:
        try:
            cmd = raw_input("cmd:")
            print cmd
            if cmd == "quit":
                print "QUIT!!!"
                do_quit(device,outsocket)
                run = 0
        except:
            traceback.print_exc()
            break
