/* Copyright (c) 2007 Axel Wachtler
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* $Id: radio.h,v 1.1 2008/03/21 11:14:26 awachtler Exp $ */
/**
 * @file
 * @brief Radio function interface.
 *
 */
#ifndef RADIO_H
#define RADIO_H

/* === Includes ============================================================== */
#include <stdbool.h>
#include <util/crc16.h>
#include "board.h"
#include "transceiver.h"

/* === Externals ============================================================= */

/* === Types ================================================================= */

/**
 * Radio state enumeration
 * @ingroup grpRadio
 */
typedef enum SHORTENUM
{
    STATE_OFF = 0,  /**< TRX is in off state */
    STATE_TX,       /**< TRX is in transmit state
                         (a toggle at SLP_TR pin sends the fifo buffer)*/
    STATE_RX,       /**< TRX is in receive state
                         (incoming frames are signalled with the RX_START and TRX_END IRQ)*/
    STATE_SLEEP,    /**< TRX is in sleep state (lowest power consumption) */
} radio_state_t;


/**
 * error codes for tx done event
 * @ingroup grpRadio
 */
typedef enum SHORTENUM
{
    TX_OK,
    TX_CCA_FAIL,
    TX_NO_ACK,
} radio_tx_done_t;

/**
 * codes for CCA
 * @ingroup grpRadio
 */
typedef enum SHORTENUM
{
    /** The CCA measurement estimates, that the channel is free. */
    CCA_FREE,
    /** The CCA measurement estimates, that the channel is busy. */
    CCA_BUSY,
    /** The CCA measurement was not finished. */
    CCA_FAIL,
} radio_cca_t;



/** Enumeration to identify radio attributes.
 * @ingroup grpRadio
 */
typedef enum SHORTENUM
{
    phyCurrentChannel        =  0x00,
    phyChannelsSupported,
    phyTransmitPower,
    phyCCAMode,
    phyRxOnIdle,

} radio_attribute_t;


/**
 * @brief Container for handover of radio parameter values.
 * @ingroup grpRadio
 */
typedef union
{
    uint8_t  curr_channel;  /**< Value for current radio channel.
                                (@ref MIN_CHANNEL ... @ref MAX_CHANNEL) */
    uint32_t supp_channels; /**< Value for supported channels. */
    uint8_t  tx_pwr;        /**< Value for transmit power.
                                - 0 : 3.2 dBm
                                - 1 : ... see datasheet
                             */
    uint8_t  rxon_idle;     /**< after TX go to RX state, if this value is true */
    uint8_t  cca_mode;      /**< Value for cca mode. */
} radio_param_t;



/** @brief Error codes.
 * @ingroup grpRadio
 */
typedef enum SHORTENUM
{
    SUCCESS = 0,        /**< OK Code*/
    STATE_SET_FAILED,   /**< function radio_set_state failed */
    SET_PARM_FAILED,    /**< function radio_set_param failed */
    GET_PARM_FAILED,    /**< function radio_get_param failed */
    GENERAL_ERROR,      /**< something unexpected happened */
} radio_error_t;


/**
 * @brief Structure for storage of radio parameters.
 * @ingroup grpRadio
 */
typedef struct
{
    uint8_t  curr_channel; /**< Current radio channel.
                                (@ref MIN_CHANNEL ... @ref MAX_CHANNEL) */
    uint8_t  tx_pwr;       /**< Current transmit power. */
    uint8_t  cca_mode;     /**< Current cca mode. */
    uint8_t  state;        /**< Current transceiver state. */
    uint8_t  rxon_idle;    /**< after TX go to RX state, if this value is true */
    uint8_t  rxrssi;       /**< RSSI value of last received frame (see also @ref VOID_RSSI). */
    uint8_t  rxlqi;        /**< LQI value of last received frame. */
    uint8_t  *rxframe;     /**< Pointer for frame data storage. */
    uint8_t  rxframesz;    /**< Length of the buffer rxframesz */
} radio_status_t;

/* === Macros ================================================================ */
/** Lowest available channel for 2.4GHz band.
 * @ingroup grpRadio
 */
#define MIN_CHANNEL (11)
/** Highest available channel for 2.4GHz band.
 * @ingroup grpRadio
 */
#define MAX_CHANNEL (26)
/** Code for invalid RSSI value.
 * @ingroup grpRadio
 */
#define VOID_RSSI (0xff)

#define CRC_CCITT_UPDATE(crc, data) _crc_ccitt_update(crc, data)

#ifndef RADIO_CFG_EEOFFSET
/** offset of radio config data in EEPROM */
#define RADIO_CFG_EEOFFSET (8)
#endif

#ifndef RADIO_CFG_DATA
/** a default radio configuration data structure */
#define RADIO_CFG_DATA {chan: 16, txp: 0, cca: 1, edt: 11, clkm: 0, crc: 0xab12}
#endif

/* === Prototypes ============================================================ */
#ifdef __cplusplus
extern "C" {
#endif


/**
 * @brief Radio related ressource initialization
 *
 * The function initializes all IO ressources,
 * needed for the usage of the radio and performs
 * a reset.
 *
 * @param rxbuf A buffer for the receive frames.
 *              This buffer needs to be static, and
 *              of size MAX_FRAME_SIZE
 *              (see also function usr_radio_receive_frame).
 * @param rxbufsz maximum size of the rx buffer, frames longer
 *                then rxbufsz will be ignored
 * @ingroup grpRadio
 */
void radio_init(uint8_t * rxbuf, uint8_t rxbufsz);


/**
 * @brief Bring the the radio in the requested state.
 * @param state  requested radio state
 * @ingroup grpRadio
 */
void radio_set_state(radio_state_t state);

/**
 * @brief Set a radio parameter.
 * @param attr  attribute parameter (enumeration value radio_attribute_t)
 * @param parm  pointer to parameter value (union type radio_param_t)
 *
 * @ingroup grpRadio
 */
void radio_set_param(radio_attribute_t attr, radio_param_t *parm);


/**
 * @brief Get value of radio parameter
 *
 * @ingroup grpRadio
 */
void radio_get_param(radio_attribute_t attr, radio_param_t *parm);


/**
 * @brief Frame tranmission
 * @todo add csma ca
 * @ingroup grpRadio
 */
void radio_send_frame(uint8_t len, uint8_t *frm, uint8_t compcrc);


/**
 * @brief Perform CCA Measure
 * @return cca status (see @ref radio_cca_t)
 * @ingroup grpRadio
 */
radio_cca_t radio_do_cca(void);

/* application call back functions to radio library */


/**
 * Error callback function which has to be implemented in the application.
 *
 * This function is called, when a fatal error occurs.
 * see also @ref radio_error_t.
 * @ingroup grpRadio
 */
void usr_radio_error(radio_error_t err);

/**
 * Interrupt callback function.
 * @ingroup grpRadio
 */
void usr_radio_irq(uint8_t cause);


/**
 * Frame reception callback function.
 * @ingroup grpRadio
 */
uint8_t * usr_radio_receive_frame(uint8_t len, uint8_t *frm, uint8_t lqi, uint8_t rssi, uint8_t crc);


/**
 * Transmit done callback function.
 * @ingroup grpRadio
 */
void usr_radio_tx_done(radio_tx_done_t status);


/**
 * Read radio configuration from eeprom or flash and program it to the transceiver.
 * @ingroup grpRadio
 */
trx_cfg_t radio_config_recall(void);


/**
 * Read radio configuration from transceiver and save it to eeprom.
 * @ingroup grpRadio
 */

trx_cfg_t radio_config_store(void);


/** @todo radio_set/get_channel() */
/** @todo radio_set/get_pwr() */
/** @todo radio_set/get_cca() */

#ifdef __cplusplus
} /* extern "C" */
#endif
#endif  /* #ifndef RADIO_H */
/* EOF */
