/* Copyright (c) 2007 Axel Wachtler
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* The function keys_debounced() use the code given in Peter Fleury's
   avr-gcc examples. File: avrgcc-examples/debounce_keys.c */

/****************************************************************************
 Title:    Debouncing 8 Keys
 Author:   Peter Fleury <pfleury@gmx.ch> http://jump.to/fleury,
           based on algorithm of Peter Dannegger <danni@specs.de>
 Date:     December 2003
 Software: AVR-GCC 3.3
 Hardware: AT90S8515 at 4 Mhz, STK200 compatible starter kit

*****************************************************************************/

/* $Id: ioutil.h,v 1.1 2008/03/21 11:14:26 awachtler Exp $ */
/**
 * @file
 * @brief IO Utility Interface definition.
 */

#ifndef IOUTIL_H
#define IOUTIL_H

/* === Includes ============================================================== */
#include "board.h"
#include "hif.h"


/* === Externals ============================================================= */

/* === Types ================================================================= */

/* === Macros ================================================================ */

/* === LED Handling === */
#if defined(NO_LEDS) || defined (DOXYGEN)
    /**
     * @addtogroup grpIoUtilLed */
    /** @{ */

    /** Initialisation of the LED port.
     */
# define LED_INIT()
    /** Display a numeric value on the LED port. */
# define LED_SET_VALUE(x)
    /** Read back the current numeric value from the LED port. */
# define LED_GET_VALUE() (0)
    /** Switch the LED with the number  @e n ON. */
# define LED_SET(ln)
    /** Switch the LED with the number  @e n OFF. */
# define LED_CLR(ln)
# define LED_VAL(msk,val)
    /** @} */
#else
# if INVERSE_LEDS == 0
#  define LED_INIT() DDR_LED |= (MASK_LED); PORT_LED &= ~(MASK_LED)
#  define LED_SET_VALUE(x) \
            do {\
                PORT_LED = (PORT_LED & ~MASK_LED) | ((x<<SHIFT_LED) & MASK_LED);\
            }while(0)
#  define LED_GET_VALUE()  ((PORT_LED & MASK_LED) >> SHIFT_LED)
#  define LED_SET(ln)      PORT_LED |= (_BV(ln+SHIFT_LED) & MASK_LED)
#  define LED_CLR(ln)      PORT_LED &= ~(_BV(ln+SHIFT_LED) & MASK_LED)
#  define LED_VAL(msk,val) PORT_LED |= ((MASK_LED|msk) << SHIFT_LED); \
                          PORT_LED |= ~((val << SHIFT_LED)& (MASK_LED|(msk<<SHIFT_LED)) )
# else
#  define LED_INIT() DDR_LED |= (MASK_LED); PORT_LED |= (MASK_LED)
#  define LED_SET_VALUE(x) do {\
            PORT_LED = (PORT_LED & ~MASK_LED) | ((~x<<SHIFT_LED) & MASK_LED);\
            }while(0)
#  define LED_GET_VALUE()  ((~PORT_LED & MASK_LED) >> SHIFT_LED)
#  define LED_SET(ln)      PORT_LED &= ~(_BV(ln+SHIFT_LED) & MASK_LED)
#  define LED_CLR(ln)      PORT_LED |= (_BV(ln+SHIFT_LED) & MASK_LED)
#  define LED_VAL(msk,val) PORT_LED &= ~(MASK_LED|(msk<<SHIFT_LED)); PORT_LED |= ~(val & (MASK_LED|msk))

# endif
#endif

/** Toggle the LED with the number  @e n. */
#define LED_TOGGLE(ln) PORT_LED ^= (_BV(ln+SHIFT_LED) & MASK_LED)

/* === KEY Handling === */
#if defined(NO_KEYS) || defined (DOXYGEN)
  /* @ingroup grpIoUtilKey */
  /** @{ */
  /** Initialisation of the KEY Port */
# define KEY_INIT()
  /** Reading of the KEY Port */
# define KEY_GET() (0)
  /** @} */
#else
# if PULLUP_KEYS != 0
#  define PULL_MASK (MASK_KEY)
# else
#  define PULL_MASK (0)
# endif
# define KEY_INIT()  do{PORT_KEY |= PULL_MASK; DDR_KEY &= ~(MASK_KEY); }while(0)
# if INVERSE_KEYS == 0
#  define KEY_GET()\
                ((PIN_KEY & MASK_KEY) >> SHIFT_KEY)
# else
#  define KEY_GET()\
                ((~PIN_KEY & MASK_KEY) >> SHIFT_KEY)
# endif
#endif



/**
 * @brief Debounce Key values, returned from the macro KEY_GET()
 * @return status ot the debounced key
 * @ingroup grpIoUtilKey
 */
static inline uint8_t keys_debounced(void)
{
  static uint8_t key_state;		// debounced and inverted key state:
  static uint8_t ct0, ct1;      // holds two bit counter for each key
  uint8_t i;


  /*
   * read current state of keys (active-low),
   * clear corresponding bit in i when key has changed
   */
  i = key_state ^ KEY_GET();   // key changed ?

  /*
   * ct0 and ct1 form a two bit counter for each key,
   * where ct0 holds LSB and ct1 holds MSB
   * After a key is pressed longer than four times the
   * sampling period, the corresponding bit in key_state is set
   */
  ct0 = ~( ct0 & i );			// reset or count ct0
  ct1 = (ct0 ^ ct1) & i;	    // reset or count ct1
  i &= ct0 & ct1;			    // count until roll over ?
  key_state ^= i;			    // then toggle debounced state

  /*
   * To notify main program of pressed key, the correspondig bit
   * in global variable key_press is set.
   * The main loop needs to clear this bit
   */
  return key_state & i;	// 0->1: key press detect

}


/* === Bootloader Interface === */
#if BOOT_LOADER_ADDRESS != 0

#define JUMP_BOOT_LOADER() \
    do {\
        void (*funcptr)( uint8_t flag ) = BOOT_LOADER_ADDRESS;\
        funcptr(42);\
    }while(0)
#else
#define JUMP_BOOT_LOADER()
#endif /* BOOT_LOADER_ADDRESS != 0 */




/* === Prototypes ============================================================ */
#ifdef __cplusplus
extern "C" {
#endif

#ifdef __cplusplus
} /* extern "C" */
#endif

#endif  /* #ifndef IOUTIL_H */
/* EOF */
