/* Copyright (c) 2007 Axel Wachtler
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* $Id: hif.h,v 1.1 2008/03/21 11:14:26 awachtler Exp $ */
/**
 * @file
 * @brief Functions for Host interface
 *
 * @ingroup grpHIF
 */
#ifndef HIF_H
#define HIF_H


/* === types =========================================== */
#include <stdarg.h>
#include <avr/pgmspace.h>
#define FLASH_STRING_T  PGM_P
#define FLASH_STRING(x) PSTR(x)
#define PRINTF(fmt, ...) hif_printf(FLASH_STRING(fmt), __VA_ARGS__)
#define PRINT(fmt) hif_echo(FLASH_STRING(fmt))
#define DUMP(sz,ptr) hif_dump(sz,ptr)
#define HIF_PUTS_NEWLINE() hif_puts_p(FLASH_STRING("\n\r"))


/* === Prototypes ====================================== */

/**
 * @brief Initialize host interface
 *
 * @param baudrate  data rate of the interface in bit/s
 *
 * @ingroup grpHIF
 */
void hif_init(const uint16_t baudrate);

/**
 * @brief Send a programm memory string to the interface.
 *
 * @param progmem_s  pointer to a null terminated  string,
 *           which is located in program memory.
 *
 * @ingroup grpHIF
 */
void hif_puts_p(const char *progmem_s);

/**
 * @brief Send string to the interface.
 *
 * @param s  pointer to a null terminated  string,
 *           which is located in RAM.
 *
 * @ingroup grpHIF
 */
void hif_puts(const char *s );

/**
 * @brief Send a block of characters to the interface.
 *
 * @param data  pointer to the data array.
 * @param size  size of the block.
 * @return num  number of bytes, which was send.
 * @ingroup grpHIF
 */
uint8_t hif_put_blk(unsigned char *data, uint8_t size);

/**
 * @brief Send a character to the interface.
 *
 * @param data  8 bit value, which is transfered.
 *
 * @ingroup grpHIF
 */
void hif_putc(unsigned char data);

/**
 * @brief Print a string to the interface.
 *
 * @param str  string, which is located in flash memory
 *
 * @ingroup grpHIF
 */
void hif_echo(FLASH_STRING_T str);

/**
 * @brief Print a formated string to the interface.
 *
 * @param fmt  format string, which is located in flash memory
 * @param ...  variable argument list
 *
 * @ingroup grpHIF
 */
void hif_printf(FLASH_STRING_T fmt, ...);

/**
 * @brief Print hexdump of a data array to the interface.
 *
 * @param sz  number of bytes, that will be dumped.
 * @param d   pointer to the data array, that will be dumped.
 *
 * @ingroup grpHIF
 */
void hif_dump(uint16_t sz, uint8_t *d);

/**
 * @brief Get a charakter byte from the host interface.
 *
 * @return 16 bit value, lower byte contains character, high
 *         byte contains error code, if 0, no error occured.
 * @ingroup grpHIF
 */
unsigned int hif_getc(void);

/**
 * @brief Get a block of bytes from the host interface.
 *
 * @param data     buffer where the bytes are stored
 * @param max_size maximum number of bytes, which can be stored in the buffer.
 * @return  number of bytes stored in the buffer
 * @ingroup grpHIF
 */
uint8_t hif_get_blk(unsigned char *data, uint8_t max_size);

#endif /* HIF_H */
