/* Copyright (c) 2007 Axel Wachtler
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* $Id: board_stk541.h,v 1.1 2008/03/21 11:14:27 awachtler Exp $ */
/**
 * @defgroup grpSTK541 STK541
 * @brief This board is describes the STK541 that comes with Atmels
 * ATAVRRZ541 Packet Sniffer Kit.
 *
 * @ingroup grpBoard
 *
 * The board wiring fits the Atmel radio development kit
 * hardware and compatibles. It uses a FTDI245 chip as
 * Host Interface.
 *
 * The wiring of the radio and the ATmega is shown below:
 *
<pre>
     AVR      RF230
     ---      -----
     PB4  -->  SLPTR
     P??  <--  MCLK
     PD4  <--  IRQ (ICP1)
     PB5  -->  RSTN
     PB0  -->  SS
     PB2  -->  MOSI
     PB3  <--  MISO
     PB1  -->  SCK


    PD6:7 -> LEDS
    PD0   -> KEY


Fuses/Locks:
     LF: 0xe2 - 8MHz internal RC Osc.
     HF: 0x11 - without boot loader
     HF: 0x10 - with boot loader
     EF: 0xff
     LOCK: 0xef - protection of boot section

Bootloader:
    Start at byte=0x1e000, address=0xf000, size = 4096 instructions/ 8192 bytes

    @image html atmel_stk541.jpg "Atmel STK541 and Radio Controller Board"
    @image latex atmel_stk541.jpg "Atmel STK541 and Radio Controller Board"
</pre>
 */

/** ID String for this hardware */
#define BOARD_NAME_STK541 "STK541"

#ifndef BOARD_STK541_H
#define BOARD_STK541_H

#define BOARD_TYPE (STK_541)
#define BOARD_NAME BOARD_NAME_STK541

/*=== Compile time parameters ========================================*/
#ifndef MAX_FRAME_SIZE
# define MAX_FRAME_SIZE (127) /**< maximum allowed frame size */
#endif

#ifndef F_CPU
# error "F_CPU is undefined"
#endif

#ifndef DEFAULT_SPI_RATE
# define DEFAULT_SPI_RATE  (SPI_RATE_1_2)
#endif

/*=== radio interface definition =====================================*/
#include "base_rdk230.h"

/*=== LED access macros ==============================================*/
#define PORT_LED      PORTD
#define DDR_LED       DDRD
#define MASK_LED      (0xc0)
#define SHIFT_LED     (6)
#define INVERSE_LEDS  (1)

/*=== KEY access macros ==============================================*/
#define PORT_KEY      PORTD
#define PIN_KEY       PIND
#define DDR_KEY       DDRD
#define MASK_KEY      (0x01)
#define SHIFT_KEY     (0)
#define INVERSE_KEYS  (1)

/*=== Host Interface ================================================*/
#define HIF_TYPE   (HIF_FT245)

#define USB_FIFO_AD  0xF000
#define XRAM_ENABLE() XMCRA |= (1 << SRE); XMCRB = (1 << XMBK)
#define HIF_NO_DATA   (0x0100)
#define FT245_DDR    DDRE
#define FT245_PIN    PINE
#define FT245_TXE    _BV(6)
#define FT245_RXF    _BV(7)
#define FT245_INIT() do { \
           FT245_DDR &= ~(FT245_TXE|FT245_RXF);\
        } while(0)
#define TX_IS_READY        (0)
#define TX_IS_BLOCKED      (FT245_TXE)
#define RX_HAS_DATA        (0)
#define RX_HAS_NO_DATA     (FT245_RXF)
#define FT245_RX_DATA()  ((FT245_PIN & FT245_RXF))
#define FT245_TX_DATA()  ((FT245_PIN & FT245_TXE))

/*=== TIMER Interface ===============================================*/
#define  TIMER_TICK (0xFFFFUL * 1000.0 / F_CPU) /** tick value in ms*/
# define TIMER_POOL_SIZE  (4)
# define TIMER_INIT() \
    do{ \
        TCCR1B |= (_BV(CS10)); \
        TIMSK1 |= _BV(TOIE1); \
    }while(0)
# define TIMER_IRQ_vect   TIMER1_OVF_vect

/*=== OSCCAL tuning =================================================*/

#ifndef TUNED_OSCCAL
# define TUNED_OSCCAL (0xbf)  /* default is 0xb1, but @2.9V 0xbf is better */
#endif

#endif /*BOARD_STK541_H*/
