/* Copyright (c) 2007 Axel Wachtler
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* $Id: board_stkm8.h,v 1.10 2009/05/18 05:26:26 awachtler Exp $ */
/**
 * @ingroup grpBoard
 * @file
 * @brief AT86RF230 adapter board wired to STK500 with Atmega8(L)/Atmega88(L).
 *
 *
 * The wiring of the radio and the ATmega is shown below:
<pre>
     AVR         RF230
     ---         -----
     PB0:1
     PB2    -->  /SEL
     PB3    -->  MOSI
     PB4    <--  MISO
     PB5    -->  SCK
     PB6    -->  RSTN
     PB7    -->  SLPTR

     PC0:6

     PD0:1
     PD2    <--  IRQ (INT2)
     PD3:4
     PD5    <--  MCLK
     PD6:7

</pre>

  @par Build Options

 - stkm8 : AT86RF230 adapter board wired to STK500 with Atmega8(L)/Atmega88(L).

 */

#define BOARD_NAME_STKM8 "stkm8"

#ifndef BOARD_STK_M8_H
#define BOARD_STK_M8_H

#define BOARD_TYPE  (STK500_MEGA8)
#define BOARD_NAME  BOARD_NAME_STKM8

/*=== Compile time parameters ========================================*/
#ifndef MAX_FRAME_SIZE
# define MAX_FRAME_SIZE (127)
#endif

#ifndef DEFAULT_SPI_RATE
# define DEFAULT_SPI_RATE  (SPI_RATE_1_2)
//# define DEFAULT_SPI_RATE  (SPI_RATE_1_128)
#endif

/*=== Hardware Components ============================================*/
#define CPU_TYPE   (CPU_M16)
#define RADIO_TYPE (RADIO_AT86RF230)

/*=== TRX pin access macros ==========================================*/

#define DDR_TRX_RESET   DDRB
#define PORT_TRX_RESET  PORTB
#define MASK_TRX_RESET  (_BV(PB3))
#define TRX_RESET_INIT() DDR_TRX_RESET |= MASK_TRX_RESET
#define TRX_RESET_HIGH() PORT_TRX_RESET |= MASK_TRX_RESET
#define TRX_RESET_LOW()  PORT_TRX_RESET &= ~MASK_TRX_RESET

#define PORT_TRX_SLPTR  PORTB
#define DDR_TRX_SLPTR   DDRB
#define MASK_TRX_SLPTR  (_BV(PB0))

#define TRX_SLPTR_INIT() DDR_TRX_SLPTR |= MASK_TRX_SLPTR
#define TRX_SLPTR_HIGH() PORT_TRX_SLPTR |= MASK_TRX_SLPTR
#define TRX_SLPTR_LOW()  PORT_TRX_SLPTR &= ~MASK_TRX_SLPTR


/*=== IRQ access macros ==============================================*/
# define TRX_IRQ         _BV(INT0)
# define TRX_IRQ_vect    INT0_vect
# define TRX_IRQ_INIT()  do{\
                            MCUCR  |= _BV(ISC01) | _BV(ISC00);\
                          } while(0) /** rising edge triggers INT0 */

#define DI_TRX_IRQ() {GICR &= (~(TRX_IRQ));}
#define EI_TRX_IRQ() {GICR |= (TRX_IRQ);}
/** timestamp register for RX_START event */
#define TRX_TSTAMP_REG TCNT1

/*=== SPI access macros ==============================================*/
#define DDR_SPI  (DDRB)
#define PORT_SPI (PORTB)

#define SPI_MOSI _BV(PB5)
#define SPI_MISO _BV(PB6)
#define SPI_SCK  _BV(PB7)
#define SPI_SS   _BV(PB4)

#define SPI_DATA_REG SPDR

static inline void SPI_INIT(uint8_t spirate)
{
    /* first configure SPI Port, then SPCR */
    PORT_SPI |= SPI_SCK | SPI_SS;
    DDR_SPI  |= SPI_MOSI | SPI_SCK | SPI_SS;
    DDR_SPI  &= ~SPI_MISO;

    SPCR = (_BV(SPE) | _BV(MSTR));

    SPCR &= ~(_BV(SPR1) | _BV(SPR0) );
    SPSR &= ~_BV(SPI2X);

    SPCR |= (spirate & 0x03);
    SPSR |= ((spirate >> 2) & 0x01);

}

#define SPI_SELN_LOW()       uint8_t sreg = SREG; cli(); PORT_SPI &=~SPI_SS
#define SPI_SELN_HIGH()      PORT_SPI |= SPI_SS; SREG = sreg
#define SPI_WAITFOR()        do { while((SPSR & _BV(SPIF)) == 0);} while(0)

/*=== LED access macros ==============================================*/
#define LED_PORT PORTC
#define LED_DDR  DDRC
#define LED_MASK 0x3f
#define LED_SHIFT  (0)
#define LED_NUMBER (0)
#define NO_LEDS (1)
#define NO_KEYS (1)

/*=== Host Interface ================================================*/
#define HIF_TYPE    HIF_UART_0


/*=== TIMER Interface ===============================================*/
/**
 * Mode: normal
 * Prescaler: 1
 * Overflow: 0xFFFFUL
 */
#define HWTMR_PRESCALE  (1)
#define HWTIMER_TICK    ((1.0*HWTMR_PRESCALE)/F_CPU)
#define HWTIMER_TICK_NB (0xFFFFUL)
#define HWTIMER_REG     (TCNT1)
#define TIMER_TICK      (HWTIMER_TICK_NB * HWTIMER_TICK)
#define TIMER_POOL_SIZE (4)
#define TIMER_INIT() \
    do{ \
        TCCR1B |= (_BV(CS10)); \
        TIMSK |= _BV(TOIE1); \
    }while(0)
# define TIMER_IRQ_vect   TIMER1_OVF_vect

/*=== OSCCAL tuning =================================================*/

#ifndef TUNED_OSCCAL
# define TUNED_OSCCAL (0xbf)  /* default is 0xb1, but @2.9V 0xbf is better */
#endif
#endif /* BOARD_STK_M8_H */
