/*
 * Copyright (C) 2011, Joel Klinghed.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef DAEMON_PROTO_H
#define DAEMON_PROTO_H

/* Sent when a daemon gets notified about a new service or a new server connects
 * (then both the servers send all currently known services to each other).
 * The service_id is generated by the "owning" daemon. */
typedef struct
{
    uint32_t service_id;
    char* usn;
    char* location;
    char* service;
    char* server; /* May be NULL */
    char* opt; /* May be NULL */
    char* nls; /* May be NULL */
} pkg_new_service_t;

/* Sent when a daemon expires a service.
 * The service_id must earlier has been sent by the same daemon with a
 * pkg_new_service_t package. */
typedef struct
{
    uint32_t service_id;
} pkg_old_service_t;

/* Sent when a daemon accepts a connection to a service and want to start to
 * tunnel it. The tunnel_id is generated by the requesting daemon.
 * If must be unique for the connection and daemon. */
typedef struct
{
    uint32_t service_id;
    uint32_t tunnel_id;
    /* Host this tunnel has on its originating daemon. The receiving tunnel
     * needs this to replace addresses */
    char* host;
    /* Port the server listens on for a connection. May be 0 which means
     * the receive server must response with a setup_tunnel. */
    uint16_t port;
} pkg_create_tunnel_t;

/* Response to create_tunnel */
typedef struct
{
    uint32_t tunnel_id;
    bool ok;
    /* Port the receiving daemon listens to to setup a tunnel.
     * May be 0 if create_tunnel.port != 0. */
    uint16_t port;
} pkg_setup_tunnel_t;

/* Can be sent by either daemon to signal that the tunnel now is closed */
typedef struct
{
    uint32_t tunnel_id;
} pkg_close_tunnel_t;

typedef enum
{
    PKG_NEW_SERVICE,
    PKG_OLD_SERVICE,
    PKG_CREATE_TUNNEL,
    PKG_SETUP_TUNNEL,
    PKG_CLOSE_TUNNEL,
} pkg_type_t;

typedef struct
{
    pkg_type_t type;
    union
    {
        pkg_new_service_t new_service;
        pkg_old_service_t old_service;
        pkg_create_tunnel_t create_tunnel;
        pkg_setup_tunnel_t setup_tunnel;
        pkg_close_tunnel_t close_tunnel;
    } content;
    size_t tmp1;
    bool tmp2;
} pkg_t;

#include "buf.h"

/* Will only copy the pointers, not their data */
void pkg_new_service(pkg_t* pkg, uint32_t service_id, char* usn, char* location,
                     char* service, char* server, char* opt, char* nls);
void pkg_old_service(pkg_t* pkg, uint32_t service_id);
void pkg_create_tunnel(pkg_t* pkg, uint32_t service_id, uint32_t tunnel_id, char* host, uint16_t port);
void pkg_setup_tunnel(pkg_t* pkg, uint32_t tunnel_id, bool ok, uint16_t port);
void pkg_close_tunnel(pkg_t* pkg, uint32_t tunnel_id);

/* Duplicate the given package */
pkg_t* pkg_dup(const pkg_t* pkg);
void pkg_free(pkg_t* pkg);

/* If pkg_write returns true, the pointers in pkg is now OK to free.
 * If pkg_write returns false, the buffer is full */
bool pkg_write(buf_t buf, pkg_t* pkg);
/* If pkg_peek returns true, pkg now contains a full package.
 * If pkg_peek returns false, more data is needed to make a full package */
bool pkg_peek(buf_t buf, pkg_t* pkg);
/* If pkg_peek returned true, call pkg_read. Any pointers in pkg is now freed */
void pkg_read(buf_t buf, pkg_t* pkg);

#endif /* DAEMON_PROTO_H */
