;;  Copyright (C) 2013 Aljosha Papsch <misc@rpapsch.de>
;;
;;  This file is part of Upmf.
;;
;;  Upmf is free software: you can redistribute it and/or modify
;;  it under the terms of the GNU General Public License as published by
;;  the Free Software Foundation, either version 3 of the License, or
;;  (at your option) any later version.
;;
;;  Upmf is distributed in the hope that it will be useful,
;;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;  GNU General Public License for more details.
;;
;;  You should have received a copy of the GNU General Public License
;;  along with Upmf.  If not, see <http://www.gnu.org/licenses/>.

(use-modules (ice-9 getopt-long))
(use-modules (ice-9 popen))

(load "utils.scm")
(load "install-modes.scm")
(load "find-utils.scm")
(load "dependency.scm")
(load "fetch.scm")
(load "package.scm")
(load "patch.scm")

(define (_ msg) (gettext msg))
(define help-message (_ "\
upmf [options]
  -i, --install PACKAGELIST            Install PACKAGELIST (comma separated list)
  -g, --install-group PACKAGEGROUPLIST Install groups of packages (comma separated list)
  -r, --remove  PACKAGELIST            Remove PACKAGELIST (comma separated list)
  -s, --search  PACKAGE                Search for PACKAGE
  -V, --version                        Display version
  -h, --help                           Display this help

A package string looks like: SECTION/NAME[:VERSION]
For example: core/autoconf:2.69
A package group string looks like: NAME
For example: core, gnome, x11))"))
(define license-message (_ "\
Upmf 0.5.1
Copyright (C) 2013 Aljosha Papsch
License GPLv3+: GNU GPL version 3 or later <http://gnu.org/licenses/gpl.html>.
This is free software: you are free to change and redistribute it.
There is NO WARRANTY, to the extent permitted by law."))

(define install-package
  (lambda (package-string)
    (if (eq? package-string '())
	'()
    (begin (validate-package-string package-string 'no-version)
	   (let* ((package-with-ver (string-split package-string #\:))
		  (package-name (list-ref package-with-ver 0))
		  (package-release (if (eq? (length package-with-ver) 2)
				       (list-ref package-with-ver 1)
				       "latest"))
		  (package-file (find-package-file package-name))
		  (package-obj (load package-file))
		  (release-to-install (get-release-for package-obj package-release 'verbose))
		  (extracted-directory (fetch-and-extract (cdr release-to-install)))
		  (pkg-dest-dir (string-append package-dest-dir "/"
					       (assq-ref package-obj 'section)
					       "/" (assq-ref package-obj 'name)
					       "/" (car release-to-install)))
		  (installed-list '())
		  (current-installed #f))
	     (display (string-append ">> Installing package:\t" package-name
				     " (version: " package-release ")"))
	     (newline)
	     (check-dependencies (assq-ref package-obj 'dependencies))
	     (apply-patches extracted-directory (assq-ref package-obj 'patches)
			    (car release-to-install))
	     (install-with-mode (assq-ref package-obj 'mode)
				package-obj extracted-directory
				pkg-dest-dir))))))

(define remove-package
  (lambda (package-string)
    (validate-package-string package-string 'with-version)
    (let* ((pkg-and-ver (string-split package-string #\:))
	   (dir-to-remove (string-append package-dest-dir "/"
					 (list-ref pkg-and-ver 0) "/"
					 (list-ref pkg-and-ver 1))))
      (display (string-append ">> Removing package:\t" package-string))
      (newline)
      (if (access? dir-to-remove F_OK)
	  (begin (display (string-append "Removing directory "
					 dir-to-remove))
		 (newline))
	  (errormsg (string-append "Release not installed: " package-string)
		    'fatal))
      (if (exec (string-append "rm -r " dir-to-remove))
	  (display (string-append "Package successfully removed: "
				  package-string))
	  (errormsg (string-append "Unable to remove directory: "
				   dir-to-remove) 'nofatal))
      (newline))))

(define print-release
  (lambda (release)
    (newline)
    (display (string-append "Release:\t" (car release)
			    " (" (cdr release) ")"))))

;; Print information about the package on the standard port
(define print-package-stat
  (lambda (package)
    (display (string-append "Package:\t" (assq-ref package 'section)
			    "/" (assq-ref package 'name)
			    "\nDescription:\t" (assq-ref package 'description)))
    (map print-release (assq-ref package 'releases))
    (newline)(newline)))

(define install-handler
  (lambda (install-list)
    (let ((installed-list (make-list (length install-list))))
      (do ((k 0 (1+ k)))



	  ((> k (- (length install-list) 1)))
	(list-set! installed-list k
		   (install-package (list-ref install-list k))))
      
      (delq '() installed-list)
      (newline)
      (display "Successfully installed packages:")
      (newline)
      (do ((k 0 (1+ k)))
	  ((> k (- (length installed-list) 1)))
	(display (string-append "Package: " (list-ref installed-list k)))
	(newline))
      (newline))))

(define remove-handler
  (lambda (remove-list)
    (do ((k 0 (1+ k)))
	((> k (- (length remove-list) 1)))
      (remove-package (list-ref remove-list k)))))

(define (upmf-start)
  (let* ((option-spec '((install (single-char #\i) (value #t))
			(install-group (single-char #\g) (value #t))
			(remove (single-char #\r) (value #t))
			(search (single-char #\s) (value #t))
			(version (single-char #\V) (value #f))
			(help (single-char #\h) (value #f))))
	 (options (getopt-long (command-line) option-spec))
	 (install-wanted (option-ref options 'install #f))
	 (group-wanted (option-ref options 'install-group #f))
	 (remove-wanted (option-ref options 'remove #f))
	 (search-wanted (option-ref options 'search #f))
	 (help-wanted (option-ref options 'help #f))
	 (version-wanted (option-ref options 'version #f)))
    (if (or install-wanted group-wanted remove-wanted search-wanted
	    help-wanted version-wanted)
	(begin
	  (if remove-wanted
	      (remove-handler (string-split remove-wanted #\,)))
	  (if group-wanted
	      (install-group (string-split group-wanted #\,)))
	  (if install-wanted
	      (install-handler (string-split install-wanted #\,)))
	  (if search-wanted
	      (find-package search-wanted))
	  (if help-wanted
	      (begin (display help-message)(newline)(quit)))
	  (if version-wanted
	      (begin (display license-message)(newline)(quit))))
	(begin (display help-message)(newline)))))
