/* UniEXP - Universo Experimental
 * Copyright (C) 1999,2002,2003,2004,2006,2007 Silvio Almeida
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


#include <iostream>

#include <uniexp/UniEXP_opengl.h>


using std::cin;
using std::cout;
using std::cerr;
using std::endl;

using std::istream;
using std::ostream;

using namespace ux;


UniEXP_opengl::UniEXP_opengl(const Attrs* attrs)
  : Art_VA("UniEXP_opengl.Art_VA", "single") {
  if (SDL_Init(SDL_INIT_VIDEO /* | SDL_INIT_NOPARACHUTE */ ) < 0) {
    cerr<<"Video initialization failed: "<<SDL_GetError()<<endl;
  }
  const SDL_VideoInfo* videoinfo = SDL_GetVideoInfo();
  if (! videoinfo) {
    /* This should probably never happen. */
    cerr<<"Video query failed: "<<SDL_GetError()<<endl;
  }
  /**
   * from SDL tutorial:
   *
   * Now, we want to setup our requested
   * window attributes for our OpenGL window.
   * We want *at least* 5 bits of red, green
   * and blue. We also want at least a 16-bit
   * depth buffer.
   *
   * The last thing we do is request a double
   * buffered window. '1' turns on double
   * buffering, '0' turns it off.
   *
   * Note that we do not use SDL_DOUBLEBUF in
   * the flags to SDL_SetVideoMode. That does
   * not affect the GL attribute state, only
   * the standard 2D blitting setup.
   */

  SDL_GL_SetAttribute(SDL_GL_RED_SIZE, 5);
  SDL_GL_SetAttribute(SDL_GL_GREEN_SIZE, 5);
  SDL_GL_SetAttribute(SDL_GL_BLUE_SIZE, 5);
  SDL_GL_SetAttribute(SDL_GL_DEPTH_SIZE, 16);
  SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, 1);

  /* base arguments setup */
  if (attrs) {
    _attrs = attrs;
  }
  Attrs* alias = _attrs.ptr();

  if (! alias->exists<Uint32>("width"))
    alias->newAttribute<Uint32>("width", static_cast<Uint32>(800));
  if (! alias->exists<Uint32>("height"))
    alias->newAttribute<Uint32>("height", static_cast<Uint32>(600));
  /**
   * from SDL tutorial:
   *
   * On X11, VidMode can't change resolution,
   * so this is probably being overly safe.
   * Under Win32, ChangeDisplaySettings can change the bpp.
   */
  if (! alias->exists<Uint32>("bpp"))
    alias->newAttribute<Uint32>("bpp", static_cast<Uint32>(videoinfo->vfmt->BitsPerPixel));
  if (! alias->exists<Uint32>("resizable"))
    alias->newAttribute<Uint32>("resizable", SDL_RESIZABLE);
  if (! alias->exists<Uint32>("fullscreen"))
    alias->newAttribute<Uint32>("fullscreen", SDL_FULLSCREEN);
  if (! alias->exists<Uint32>("framespsec"))
    alias->newAttribute<float>("framespsec", 18.f);

  cout<<"UniEXP_opengl() width:"<<width()<<", height:"<<height()<<", bpp:"<<bpp()<<", resisable:"<<resizable()<<", full:"<<fullscreen()<<endl;
  if (SDL_SetVideoMode(width(), height(), bpp(), SDL_OPENGL | resizable() | fullscreen()) == 0) {
    /**
     * from SDL tutorial:
     *
     * This could happen for a variety of reasons,
     * including DISPLAY not being set, the specified
     * resolution not being available, etc.
     */
    fprintf(stderr, "Video mode set failed: %s\n", SDL_GetError());
    exit (1);
  } else {
    GLint val;
    SDL_GL_GetAttribute(SDL_GL_RED_SIZE    ,&val); cout<<"SDL_GL_RED_SIZE: "    <<val<<endl;
    SDL_GL_GetAttribute(SDL_GL_GREEN_SIZE  ,&val); cout<<"SDL_GL_GREEN_SIZE: "  <<val<<endl;
    SDL_GL_GetAttribute(SDL_GL_BLUE_SIZE   ,&val); cout<<"SDL_GL_BLUE_SIZE: "   <<val<<endl;
    SDL_GL_GetAttribute(SDL_GL_DEPTH_SIZE  ,&val); cout<<"SDL_GL_DEPTH_SIZE: "  <<val<<endl;
    SDL_GL_GetAttribute(SDL_GL_DOUBLEBUFFER,&val); cout<<"SDL_GL_DOUBLEBUFFER: "<<val<<endl;
  }

  /** limita frames por segundo a framespsec */
  realTime.setFPS(framespsec());
}

UniEXP_opengl::~UniEXP_opengl ()
{
}


UniEXP_opengl* UniEXP_opengl::singleton = 0;
UniEXP_opengl* UniEXP_opengl::Instance(const Attrs* attrs) {
  if (singleton == 0) {
    singleton = new UniEXP_opengl(attrs);
  }
  return singleton;
}


void UniEXP_opengl::setupScene()
{
  /* inicialização geométrica */
  camera.setEye(0.0, -14.0, 2.5);
  camera.setAt(0.0, 0.0, 8.0);

  camera.setFOV(67.0);
  camera.setClipPlanes(1, 100.0);
  camera.setViewport(width(), height()); // atualiza camera.aspect

  spinner.setAngles(0.0, 0.0, 0.0);
  spinner.setSpinX(0.00);
  spinner.setSpinY(0.00);
  spinner.setSpinZ(0.20);
  spinner.doSpin(); //0); // ();
  
  /* Set the clear color and depth test */
  glClearColor(0.22, 0.22, 0.22, 1.0);
  glEnable(GL_DEPTH_TEST);
  
  /* Setup our viewport */
  camera.gluPerspective();
  camera.glViewport();
}

void UniEXP_opengl::setupLights()
{
  glEnable(GL_LIGHTING);
  float ambi[4] = {0.0, 0.0, 0.0, 1.0};
  float spec[4] = {1.0, 1.0, 1.0, 1.0};
  float sdir[4] = {0.0, 0.0, -1.0, 1.0};
  float sexp = 36.0;
  float scut = 90.0;
  float attc = 0.0;
  float attl = 0.0;
  float attq = 0.0001;

  float diff0[4] = {0.7, 0.4, 0.2, 1.0};
  float posi0[4] = {10.0, -10.0, 20.0, 1.0};
  glEnable(GL_LIGHT0);
  glLightfv(GL_LIGHT0, GL_DIFFUSE, diff0);
  glLightfv(GL_LIGHT0, GL_POSITION, posi0);
  glLightfv(GL_LIGHT0, GL_AMBIENT, ambi);
  glLightfv(GL_LIGHT0, GL_SPECULAR, spec);
  glLightfv(GL_LIGHT0, GL_SPOT_DIRECTION, sdir);
  glLightf(GL_LIGHT0, GL_SPOT_EXPONENT, sexp);
  glLightf(GL_LIGHT0, GL_SPOT_CUTOFF, scut);
  glLightf(GL_LIGHT0, GL_CONSTANT_ATTENUATION, attc);
  glLightf(GL_LIGHT0, GL_LINEAR_ATTENUATION, attl);
  glLightf(GL_LIGHT0, GL_QUADRATIC_ATTENUATION, attq);

  float diff1[4] = {0.3, 0.6, 0.8, 1.0};
  float posi1[4] = {-10.0, -10.0, 20.0, 1.0};
  glEnable(GL_LIGHT1);
  glLightfv(GL_LIGHT1, GL_DIFFUSE, diff1);
  glLightfv(GL_LIGHT1, GL_POSITION, posi1);
  glLightfv(GL_LIGHT1, GL_AMBIENT, ambi);
  glLightfv(GL_LIGHT1, GL_SPECULAR, spec);
  glLightfv(GL_LIGHT1, GL_SPOT_DIRECTION, sdir);
  glLightf(GL_LIGHT1, GL_SPOT_EXPONENT, sexp);
  glLightf(GL_LIGHT1, GL_SPOT_CUTOFF, scut);
  glLightf(GL_LIGHT1, GL_CONSTANT_ATTENUATION, attc);
  glLightf(GL_LIGHT1, GL_LINEAR_ATTENUATION, attl);
  glLightf(GL_LIGHT1, GL_QUADRATIC_ATTENUATION, attq);

  float diff2[4] = {0.3, 0.8, 0.3, 1.0};
  float posi2[4] = {10.0, 10.0, 20.0, 1.0};
  glEnable(GL_LIGHT2);
  glLightfv(GL_LIGHT2, GL_DIFFUSE, diff2);
  glLightfv(GL_LIGHT2, GL_POSITION, posi2);
  glLightfv(GL_LIGHT2, GL_AMBIENT, ambi);
  glLightfv(GL_LIGHT2, GL_SPECULAR, spec);
  glLightfv(GL_LIGHT2, GL_SPOT_DIRECTION, sdir);
  glLightf(GL_LIGHT2, GL_SPOT_EXPONENT, sexp);
  glLightf(GL_LIGHT2, GL_SPOT_CUTOFF, scut);
  glLightf(GL_LIGHT2, GL_CONSTANT_ATTENUATION, attc);
  glLightf(GL_LIGHT2, GL_LINEAR_ATTENUATION, attl);
  glLightf(GL_LIGHT2, GL_QUADRATIC_ATTENUATION, attq);

  float diff3[4] = {0.9, 0.5, 0.1, 1.0};
  float posi3[4] = {-10.0, 10.0, 20.0, 1.0};
  glEnable(GL_LIGHT3);
  glLightfv(GL_LIGHT3, GL_DIFFUSE, diff3);
  glLightfv(GL_LIGHT3, GL_POSITION, posi3);
  glLightfv(GL_LIGHT3, GL_AMBIENT, ambi);
  glLightfv(GL_LIGHT3, GL_SPECULAR, spec);
  glLightfv(GL_LIGHT3, GL_SPOT_DIRECTION, sdir);
  glLightf(GL_LIGHT3, GL_SPOT_EXPONENT, sexp);
  glLightf(GL_LIGHT3, GL_SPOT_CUTOFF, scut);
  glLightf(GL_LIGHT3, GL_CONSTANT_ATTENUATION, attc);
  glLightf(GL_LIGHT3, GL_LINEAR_ATTENUATION, attl);
  glLightf(GL_LIGHT3, GL_QUADRATIC_ATTENUATION, attq);
}

void UniEXP_opengl::setupMaterials()
{
  /* Our shading model--Gouraud (smooth) */
  glShadeModel(GL_SMOOTH);

  /* Culling */
  glCullFace(GL_BACK);
  glFrontFace(GL_CCW);
  glEnable(GL_CULL_FACE);

  glColorMaterial(GL_FRONT, GL_SPECULAR);
  glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE);
}

void UniEXP_opengl::setup() {
  setupScene();
  setupLights();
  setupMaterials();
}

void UniEXP_opengl::setupGeom(const Attrs* src) {
  if (! src)
    src = new Attrs("rootAttrs", "UniEXP_opengl");
  /** defaults da biblioteca */
  int numTrees = 1; /**< número máximo de árvores */
  int leafs = 6; /**< número máximo de folhas por nó */
  int deep = 6; /**< profundidade máxima de iteração */
  int maxi = 1000; /**< número máximo de elementos por árvore */
  Coord conic = 17.0; /**< ângulo máximo de dispersão germinativa */
  Coord ratio = 0.5; /**< relação entre X e Y ?? */
  int random = -100; /**< randomicidade gloabal / random seed
		      * random == 0 : simetria e previsibilidade
		      * random > 0 : semente randômica
		      * random < 0 : randomicidade máxima / time seed
		      */
  Coord3 randLCR = Coord3(0.09, 0.67, 0.21);
  unsigned int randSeed = RANDOM_SEED; /**< 0 ou RANDOM_SEED por exemplo */

  /** atributos de Art_VA* raiz */
  Attrs* alias = _attrs.ptr();

  alias->setAttribute<int>("numTrees", src->exists<int>("numTrees") ? src->getAttribute<int>("numTrees") : numTrees);
  alias->setAttribute<int>("leafs", src->exists<int>("leafs") ? src->getAttribute<int>("leafs") : leafs);
  alias->setAttribute<int>("deep", src->exists<int>("deep") ? src->getAttribute<int>("deep") : deep);
  alias->setAttribute<int>("maxi", src->exists<int>("maxi") ? src->getAttribute<int>("maxi") : maxi);
  alias->setAttribute<Coord>("conic", src->exists<Coord>("conic") ? src->getAttribute<Coord>("conic") : conic);
  alias->setAttribute<Coord>("ratio", src->exists<Coord>("ratio") ? src->getAttribute<Coord>("ratio") : ratio);
  alias->setAttribute<int>("random", src->exists<int>("random") ? src->getAttribute<int>("random") : random);
  alias->setAttribute<Coord3>("randLCR", src->exists<Coord3>("randLCR") ? src->getAttribute<Coord3>("randLCR") : randLCR);

  random = alias->getAttribute<int>("random");
  alias->setAttribute<unsigned int>("randSeed",
					src->exists<unsigned int>("randSeed") ?
					src->getAttribute<unsigned int>("randSeed") :
					(random < 0 ? 0 : (random == 0 ? randSeed : random)));

  setupGeom(alias->getAttribute<int>("numTrees"),
	    alias->getAttribute<int>("leafs"),
	    alias->getAttribute<int>("deep"),
	    alias->getAttribute<int>("maxi"),
	    alias->getAttribute<Coord3>("randLCR"),
	    alias->getAttribute<unsigned int>("randSeed"),
	    alias->getAttribute<Coord>("conic"),
	    alias->getAttribute<Coord>("ratio")
	    );
}

#define MIN(a, b) ((b < a ? b : a))
#define MAX(a, b) ((b > a ? b : a))

void UniEXP_opengl::setupGeom(int numTrees,
			      size_t maxChild, size_t maxDeep, size_t maxDesc,
			      Coord3 randLCR, unsigned int randomSeed,
			      Coord conic, Coord ratio) {
  Attrs* alias = _attrs.ptr();
  Random randFn(0.0, 1.0, randomSeed);
  int mode = 5;
  numTrees = ( numTrees<0 ? static_cast<int>(randFn() * MIN(-numTrees, UX_UNIEXP_OPENGL_MAX_RAND_TREES)) : numTrees );

  alias->setAttribute<int>("vento", mode);
  alias->setAttribute<AllocC<_fn_createGeom>*>("createFn", new AllocC<_fn_createGeom>(_fn_createRandomGeom("uniexp.createFn",
													   maxChild, maxDeep, maxDesc,
													   randLCR, randomSeed,
													   conic, ratio)));
  alias->setAttribute<AllocC<_fn_calcGeom>*>("calcFn", new AllocC<_fn_calcGeom>(_fn_calcGeom("uniexp.calcFn", mode, randomSeed)));
  alias->setAttribute<AllocC<_fn_drawGeom>*>("drawFn", new AllocC<_fn_drawGeom>(_fn_drawGeom("uniexp.drawFn", mode, randomSeed)));
  alias->setAttribute<AllocC<_fn_popGeom>*>("popFn", new AllocC<_fn_popGeom>(_fn_popGeom("uniexp.popFn", mode, randomSeed)));

  clear();
  for (int i=0; i<numTrees; i++) {
    std::string fnId("loop["+_str(i)+"]");
    cout<<numTrees<<" trees; calculando "<<i<<"... fnId:"<<fnId<<endl;
    Attrs attrs("Attrs:UniEXP_opengl.setupGeom()", fnId);
    mode = static_cast<int>(randFn() * mode);
    attrs.setAttribute<int>("vento", mode);
    attrs.setAttribute<Coord3>("base", Coord3((UX_UNIEXP_OPENGL_MAX_RAND_X / 2) - randFn() * UX_UNIEXP_OPENGL_MAX_RAND_X,
					      (UX_UNIEXP_OPENGL_MAX_RAND_Y / 2) - randFn() * UX_UNIEXP_OPENGL_MAX_RAND_Y,
					      (UX_UNIEXP_OPENGL_MAX_RAND_Z / 2) - randFn() * UX_UNIEXP_OPENGL_MAX_RAND_Z));
    attrs.setAttribute<AllocC<_fn_createGeom>*>("createFn", alias->getAttribute<AllocC<_fn_createGeom>*>("createFn"));
    attrs.setAttribute<AllocC<_fn_calcGeom>*>("calcFn", alias->getAttribute<AllocC<_fn_calcGeom>*>("calcFn"));
    attrs.setAttribute<AllocC<_fn_drawGeom>*>("drawFn", alias->getAttribute<AllocC<_fn_drawGeom>*>("drawFn"));
    attrs.setAttribute<AllocC<_fn_popGeom>*>("popFn", alias->getAttribute<AllocC<_fn_popGeom>*>("popFn"));
    Art_VA geom(attrs);
    ad_fim(&geom);
    cout<<"FIM DO SETUP_GEOM( item "<<i<<" ): ";
    if (back().ptr())
      cout<<*dynamic_cast<Art_VA*>(back().ptr())<<"\nFIM FIM FIM"<<endl;
    else
      cout<<"NULL\nFIM FIM FIM"<<endl;
  }
}

void UniEXP_opengl::createGeom(Attrs* attrs) {
  _fn_createRandomGeom* createFn = dynamic_cast<_fn_createRandomGeom*>(_attrs.ptr()->getAttribute<AllocC<_fn_createGeom>*>("createFn")->ptr());
  if (createFn && ! empty()) {
    for (_iv_AA i=begin(); i!=end(); i++) {
      createFn->init();
      i->ptr()->xx(createFn, attrs);
    }
  }
}

void UniEXP_opengl::drawInit() {
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  camera.gluLookAt();
  spinner.glRotate();
}

void UniEXP_opengl::calcGeom(Attrs* attrs) {
  _fn_calcGeom* calcFn = _attrs.ptr()->getAttribute<AllocC<_fn_calcGeom>*>("calcFn")->ptr();
  if (calcFn && ! empty()) {
    for (_iv_AA i=begin(); i!=end(); i++) {
      calcFn->init();
      i->ptr()->xx(calcFn, attrs);
    }
  }
}

void UniEXP_opengl::drawGeom(Attrs* attrs) {
  _fn_drawGeom* drawFn = _attrs.ptr()->getAttribute<AllocC<_fn_drawGeom>*>("drawFn")->ptr();
  _fn_popGeom* popFn = _attrs.ptr()->getAttribute<AllocC<_fn_popGeom>*>("popFn")->ptr();
  if (drawFn && popFn && ! empty()) {
    for (_iv_AA i=begin(); i!=end(); i++) {
      drawFn->init();
      popFn->init();
      i->ptr()->pp(drawFn, popFn, attrs);
    }
  }
}

void UniEXP_opengl::drawSwap() {
  realTime.keepFPS();
  SDL_GL_SwapBuffers();
}


Uint32 UniEXP_opengl::width() const { return _attrs.ptr()->getAttribute<Uint32>("width"); }
Uint32 UniEXP_opengl::height() const { return _attrs.ptr()->getAttribute<Uint32>("height"); }
Uint32 UniEXP_opengl::bpp() const { return _attrs.ptr()->getAttribute<Uint32>("bpp"); }
Uint32 UniEXP_opengl::resizable() const { return _attrs.ptr()->getAttribute<Uint32>("resizable"); }
Uint32 UniEXP_opengl::fullscreen() const { return _attrs.ptr()->getAttribute<Uint32>("fullscreen"); }
float  UniEXP_opengl::framespsec() const { return _attrs.ptr()->getAttribute<float>("framespsec"); }


istream& ux::operator>>(istream& is, UniEXP_opengl& ux) { return is; }
ostream& ux::operator<<(ostream& os, const UniEXP_opengl& ux) {
  os<<"<uniexp_opengl tp=\""<<ux.tp()<<"\" id=\""<<ux.id()<<"\">";
  return (dynamic_cast<const Art_VA&>(ux).output(os))<<"</uniexp_opengl>";
}

