/* UniEXP - Universo Experimental
 * Copyright (C) 1999,2002,2003,2004,2006,2007 Silvio Almeida
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <iostream>
#include <cmath>

#include <uniexp/Functors_opengl.h>


using std::cout;
using std::cerr;
using std::endl;
using namespace ux;


std::istream& ux::operator>>(std::istream& is, _fn_createGeom& fn) { return is; }
std::ostream& ux::operator<<(std::ostream& os, const _fn_createGeom& fn) {
  return os<<"<_fn_createGeom count=\""<<fn._count<<"\" leafs=\""<<fn._leafs<<"\" deep=\""<<fn._deep<<"\" maxi=\""<<fn._maxi<<"\" conic=\""<<fn._conic<<"\" ratio=\""<<fn._ratio<<"\">:"<<static_cast<const _fn_varr&>(fn)<<"</_fn_createGeom>";
}

_fn_createGeom::_fn_createGeom()
  : _fn_varr("_fn_createGeom", "noid"),
    _leafs(0), _deep(0), _maxi(0),
    _conic(0), _ratio(0)
{ }
_fn_createGeom::_fn_createGeom(const _fn_createGeom& src)
  : _fn_varr(src.tp(), src.id()+".C"),
    _leafs(src._leafs), _deep(src._deep), _maxi(src._maxi),
    _conic(src._conic), _ratio(src._ratio)
{ }
_fn_createGeom::_fn_createGeom(const std::string& tp, const std::string& id)
  : _fn_varr(tp, id),
    _leafs(0), _deep(0), _maxi(0),
    _conic(0), _ratio(0)
{ }
_fn_createGeom::_fn_createGeom(const std::string& id,
			       size_t leafs, size_t deep, size_t maxi,
			       Coord conic, Coord ratio)
  : _fn_varr("_fn_createGeom", id),
    _leafs(leafs), _deep(deep), _maxi(maxi),
    _conic(conic), _ratio(ratio) {
}

_fn_createGeom::~_fn_createGeom()
{ }

void _fn_createGeom::operator() (Art_0* art, Attrs* xxxxxxxx_attrs) {

  if (_count == 0)
    art->getAttrs()->setAttribute<Coord3>("baseCoords", Coord3(0.0, 0.0, 1.0));

  if (art->prof() >= _deep)
    return;

  Coord high = 1.0 / tan(_conic * M_PI / 180.0);
  for (size_t i=0; i<_leafs && _count<_maxi; i++) {
    Coord ang = 2.0 * M_PI * (static_cast<Coord>(i) / static_cast<Coord>(_leafs));
    Attrs attrs("Attrs_" + _str(_count+1));
    Coord3 base(sin(ang), cos(ang), high);
    attrs.setAttribute<Coord3>("baseCoords", base.multiply(_ratio));
    art->ad_fim( new Art_VA(attrs) );
    _count++;
  }
}



std::istream& ux::operator>>(std::istream& is, _fn_createRandomGeom& fn) { return is; }
std::ostream& ux::operator<<(std::ostream& os, const _fn_createRandomGeom& fn) {
  return os<<"<_fn_createRandomGeom randLeafConicRatio=\""<<fn._randLeafConicRatio<<"\" alea=\""<<fn._alea<<"\">:"<<static_cast<const _fn_createGeom&>(fn)<<"</_fn_createRandomGeom>";
}

_fn_createRandomGeom::_fn_createRandomGeom()
  : _fn_createGeom("_fn_createRandomGeom", "noid"),
    _randLeafConicRatio(0.0, 0.0, 0.0), _alea(0.0, 1.0)
{ }

_fn_createRandomGeom::_fn_createRandomGeom(const _fn_createRandomGeom& src)
  : _fn_createGeom(src.id()+".C",
		   src._leafs, src._deep, src._maxi,
		   src._conic, src._ratio),
    _randLeafConicRatio(src._randLeafConicRatio), _alea(0.0, 1.0, src._alea.getSeed())
{ }

_fn_createRandomGeom::_fn_createRandomGeom(const std::string& tp, const std::string& id)
  : _fn_createGeom(tp, id),
     _randLeafConicRatio(0.0, 0.0, 0.0), _alea(0.0, 1.0, 0)
{ }

_fn_createRandomGeom::_fn_createRandomGeom(const std::string& id,
					   size_t leafs, size_t deep, size_t maxi,
					   Coord3 randLCR, unsigned int seed,
					   Coord conic, Coord ratio)
  : _fn_createGeom(id,
		   leafs, deep, maxi,
		   conic, ratio),
    _randLeafConicRatio(randLCR), _alea(0.0, 1.0, seed) {
  tp("_fn_createRandomGeom");
}

_fn_createRandomGeom::~_fn_createRandomGeom()
{ }

void _fn_createRandomGeom::operator() (Art_0* art_0, Attrs* xxxxxxxx_attrs) {

  Art_VA* art = dynamic_cast<Art_VA*>(art_0);
  if (! art) {
    cerr<<"ERRO: _fn_createRandomGeom::operator(): Art_VA* == 0"<<endl;
    return;
  }

  Coord randConic;
  Coord randRatio;
  Coord high;
  Coord ang;
  Coord deep = art->prof();

  if (_count == 0) {
    randConic = _conic * (0.75 + _randLeafConicRatio.Y() * _alea());
    randRatio = _ratio * (0.75 + _randLeafConicRatio.Z() * _alea());
    high = 1.0 / tan(_conic * M_PI / 180.0);
    art->getAttrs()->setAttribute<Coord3>("baseCoords", Coord3(0.0, 0.0, high * randRatio));
  }

  if (deep >= _deep)
    return;

  int randChild = static_cast<int>(_leafs * (1.1 - _randLeafConicRatio.X() * _alea()));

  for (int i=0; i<randChild && _count<_maxi; i++) {
    randConic = _conic * (0.75 + _randLeafConicRatio.Y() * _alea());
    randRatio = _ratio * (0.75 + _randLeafConicRatio.Z() * _alea());
    high = 1.0 / tan(_conic * M_PI / 180.0);
    ang = 2.0 * M_PI * (static_cast<Coord>(i) / static_cast<Coord>(randChild));
    ang = ang * (1 + (0.5 - _alea()));
    Attrs attrs("artAttrs", "attr" + _str(_count+1));
    Coord3 base(sin(ang), cos(ang), high);
    attrs.setAttribute<Coord3>("baseCoords", base.multiply(randRatio));
    Art_VA artVA(attrs);
    art->ad_fim(&artVA); /** não adquire o ponteiro */
    _count++;
  }
}



std::istream& ux::operator>>(std::istream& is, _fn_calcGeom& fn) { return is; }
std::ostream& ux::operator<<(std::ostream& os, const _fn_calcGeom& fn) { return os<<"++{fn=>_fn_calcGeom}"; }

_fn_calcGeom::_fn_calcGeom()
  : _fn_varr("_fn_calcGeom", "noid"),
    _mode(0), _alea(-0.00002, 0.00002)
{ }
_fn_calcGeom::_fn_calcGeom(const _fn_calcGeom& src)
  : _fn_varr(src.id()+".C", src.tp()),
    _mode(src._mode), _alea(-0.00002, 0.00002, src._alea.getSeed())
{ }
_fn_calcGeom::_fn_calcGeom(const std::string& tp, const std::string& id)
  : _fn_varr(tp, id),
    _mode(0), _alea(-0.00002, 0.00002)
{ }
_fn_calcGeom::_fn_calcGeom(const std::string& id,
			   size_t mode, unsigned int seed)
  : _fn_varr(id, "_fn_calcGeom"),
    _mode(mode), _alea(-0.00002, 0.00002, seed)
{ }

_fn_calcGeom::~_fn_calcGeom()
{ }

void _fn_calcGeom::operator() (Art_0* art, Attrs* xxxxxxxx_attrs)
{ }



std::istream& ux::operator>>(std::istream& is, _fn_popGeom& fn) { return is; }
std::ostream& ux::operator<<(std::ostream& os, const _fn_popGeom& fn) { return os<<"++{fn=>_fn_popGeom}"; }

_fn_popGeom::_fn_popGeom()
  : _fn_varr("_fn_popGeom", "noid"),
    _mode(0), _alea(-1, 1),
    stackDepth(0)
{
  glGetIntegerv(GL_MAX_MODELVIEW_STACK_DEPTH, &maxStackDepth);
}

_fn_popGeom::_fn_popGeom(const _fn_popGeom& src)
  : _fn_varr(src.tp(), src.id()+".C"),
    _mode(src._mode), _alea(-1, 1, src._alea.getSeed()),
    maxStackDepth(src.maxStackDepth), stackDepth(0)
{ }

_fn_popGeom::_fn_popGeom(const std::string& tp, const std::string& id)
  : _fn_varr(tp, id),
    _mode(0), _alea(-1, 1),
    stackDepth(0) {
  glGetIntegerv(GL_MAX_MODELVIEW_STACK_DEPTH, &maxStackDepth);
}

_fn_popGeom::_fn_popGeom(const std::string& id, size_t mode, unsigned int seed)
  : _fn_varr("_fn_popGeom", id),
    _mode(mode), _alea(-1, 1, seed),
    stackDepth(0) {
  glGetIntegerv(GL_MAX_MODELVIEW_STACK_DEPTH, &maxStackDepth);
}

_fn_popGeom::~_fn_popGeom()
{ }

void _fn_popGeom::init() {
  _fn_varr::init();
  stackDepth = 0;
}

void _fn_popGeom::operator() (Art_0* art, Attrs* xxxxxxxx_attrs) {
  glGetIntegerv(GL_MODELVIEW_STACK_DEPTH, &stackDepth);
  if (stackDepth > 0) {
    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();
  } else {
    cerr<<"ERRO de esvaziamento de pilha GL_MODELVIEW_STACK em "<<*art<<endl;
    return;
  }
}
			       

std::istream& ux::operator>>(std::istream& is, _fn_pushGeom& fn) { return is; }
std::ostream& ux::operator<<(std::ostream& os, const _fn_pushGeom& fn) { return os<<"++{fn=>_fn_pushGeom}"; }

_fn_pushGeom::_fn_pushGeom()
  : _fn_popGeom("_fn_pushGeom", "noid")
{ }
_fn_pushGeom::_fn_pushGeom(const _fn_pushGeom& src)
  : _fn_popGeom(src)
{ }
_fn_pushGeom::_fn_pushGeom(const std::string& tp, const std::string& id)
  : _fn_popGeom(tp, id)
{ }
_fn_pushGeom::_fn_pushGeom(const std::string& id, size_t mode, unsigned int seed)
  : _fn_popGeom(id, mode, seed)
{ }

_fn_pushGeom::~_fn_pushGeom()
{ }

void _fn_pushGeom::operator() (Art_0* art, Attrs* xxxxxxxx_attrs) {
  glGetIntegerv(GL_MODELVIEW_STACK_DEPTH, &stackDepth);
  if (stackDepth < maxStackDepth) {
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
  } else {
    cerr<<"ERRO de pilha cheia GL_MODELVIEW_STACK em "<<*art<<endl;
    return;
  }
  Coord3 item = art->getAttrs()->getAttribute<Coord3>("baseCoords");
  glTranslate(item.X(), item.Y(), item.Z());
}



std::istream& ux::operator>>(std::istream& is, _fn_drawGeom& fn) { return is; }
std::ostream& ux::operator<<(std::ostream& os, const _fn_drawGeom& fn) { return os<<"++{fn=>_fn_createGeom}"; }

_fn_drawGeom::_fn_drawGeom()
  : _fn_pushGeom("_fn_drawGeom", "noid") {
  _alea.setRange(-0.00002, 0.00002);
  qobj = gluNewQuadric();
}
_fn_drawGeom::_fn_drawGeom(const _fn_drawGeom& src)
  : _fn_pushGeom(src) {
  _alea.setRange(-0.00002, 0.00002);
  qobj = gluNewQuadric();
}
_fn_drawGeom::_fn_drawGeom(const std::string& tp, const std::string& id)
  : _fn_pushGeom(tp, id) {
  _alea.setRange(-0.00002, 0.00002);
  qobj = gluNewQuadric();
}
_fn_drawGeom::_fn_drawGeom(const std::string& id,
			   size_t mode, unsigned int seed)
  : _fn_pushGeom(id, mode, seed) {
  _alea.setRange(-0.00002, 0.00002);
  qobj = gluNewQuadric();
}

_fn_drawGeom::~_fn_drawGeom() {
  gluDeleteQuadric(qobj);
}

void _fn_drawGeom::operator() (Art_0* art, Attrs* xxxxxxxx_attrs) {
  Attrs* artAttrs = art->getAttrs();
  Coord3 item = artAttrs->getAttribute<Coord3>("baseCoords");
  /* cache */
  Coord deep = 1 + art->prof();
  Coord factor = deep + 1;
  factor *= deep;
  Coord x, y, z = item.Z();
  if (_mode) {
    x = _mode * factor * _alea() + item.X();
    y = _mode * factor * _alea() + item.Y();
  } else {
    x = item.X();
    y = item.Y();
  }
  /* vento */
  artAttrs->setAttribute<Coord3>("baseCoords", item.set(x, y, z));
  Coord3 perp(0.0, 0.0, 1.0);
  glMatrixMode(GL_MODELVIEW);
  glGetIntegerv(GL_MODELVIEW_STACK_DEPTH, &stackDepth);
  if (stackDepth < maxStackDepth) {
    glPushMatrix();
  } else {
    cerr<<"ERRO de pilha cheia GL_MODELVIEW_STACK em "<<*art<<endl;
    return;
  }
  const Coord3* base = artAttrs->getAttributePtr<Coord3>("base");
  if (base) {
    glTranslate(base->X(), base->Y(), base->Z());
  }
  if (! perp.isParallel(item)) {
    Coord ang = acos(perp.dot(item)) * 180.0 * M_1_PI;
    // ambti-gimbal : 
    Coord test = ang;
    if (ang < 88 || ang > 92) {
      perp.cross(item);
      glRotate(ang, perp.nX(), perp.nY(), perp.nZ());
      // bizarro
      // glRotate(ang, -item.Y(), item.X(), 0.0);
      }
  }
  Coord radius = z * .52 / factor;
  gluCylinder(qobj, radius, radius, z, 7, 7);
  glTranslate(0.0, 0.0, z);
}



