/* UniEXP - Universo Experimental
 * Copyright (C) 1999,2002,2003,2004,2006,2007 Silvio Almeida
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <cmath>

#include <uniexp/Vectors.h>


using std::cout;
using std::endl;


Coord3::Coord3(Coord x, Coord y, Coord z) {
  coords[0] = x;
  coords[1] = y;
  coords[2] = z;
  normal = 0;
}

Coord3::Coord3(const Coord* vec3) {
  coords[0] = vec3[0];
  coords[1] = vec3[1];
  coords[2] = vec3[2];
  normal = 0;
}

Coord3::Coord3(const Coord3& src) {
  coords[0] = src.coords[0];
  coords[1] = src.coords[1];
  coords[2] = src.coords[2];
  normal = 0;
  if (src.normal) {
    cleanNormal();
    normal[0] = src.normal[0];
    normal[1] = src.normal[1];
    normal[2] = src.normal[2];
  }
}

Coord3::~Coord3() {
  deleteNormal();
}

const Coord3& Coord3::operator=(const Coord3& src) {
  if (&src != this) {
    coords[0] = src.coords[0];
    coords[1] = src.coords[1];
    coords[2] = src.coords[2];
    if (normal) {
      magnitude();
    }
  }
  return *this;
}

const Coord3 Coord3::operator+(const Coord3& vec3) const {
  Coord3 ret(*this);
  ret.coords[0] += vec3.coords[0];
  ret.coords[1] += vec3.coords[1];
  ret.coords[2] += vec3.coords[2];
  if (ret.normal) {
    ret.magnitude();
  }
  return ret;
}

const Coord3 Coord3::operator-(const Coord3& vec3) const {
  Coord3 ret(*this);
  ret.coords[0] -= vec3.coords[0];
  ret.coords[1] -= vec3.coords[1];
  ret.coords[2] -= vec3.coords[2];
  if (ret.normal) {
    ret.magnitude();
  }
  return ret;
}

const Coord3 Coord3::operator/(const Coord& k) const {
  Coord3 ret(*this);
  ret.coords[0] /= k;
  ret.coords[1] /= k;
  ret.coords[2] /= k;
  if (ret.normal) {
    ret.magnitude();
  }
  return ret;
}

const Coord3 Coord3::operator*(const Coord& k) const {
  Coord3 ret(*this);
  ret.coords[0] *= k;
  ret.coords[1] *= k;
  ret.coords[2] *= k;
  if (ret.normal) {
    ret.magnitude();
  }
  return ret;
}

const Coord3& Coord3::set(Coord x, Coord y, Coord z) {
  coords[0] = x;
  coords[1] = y;
  coords[2] = z;
  if (normal) {
    magnitude();
  }
  return *this;
}

const Coord3& Coord3::set(Coord* n) {
  coords[0] = n[0];
  coords[1] = n[1];
  coords[2] = n[2];
  if (normal) {
    magnitude();
  }
  return *this;
}

const Coord3& Coord3::setX(Coord x) {
  coords[0] = x;
  if (normal) {
    magnitude();
  }
  return *this;
}

const Coord3& Coord3::setY(Coord y) {
  coords[1] = y;
  if (normal) {
    magnitude();
  }
  return *this;
}

const Coord3& Coord3::setZ(Coord z) {
  coords[2] = z;
  if (normal) {
    magnitude();
  }
  return *this;
}

const Coord3& Coord3::setW(Coord w) {
  normalize();
  normal[3] = w;
  coords[0] = normal[0] * normal [3];
  coords[1] = normal[1] * normal [3];
  coords[2] = normal[2] * normal [3];
  return *this;
}

const Coord* Coord3::get() const {
  return const_cast<const Coord*>(coords);
}
const Coord* Coord3::getN() const {
  return const_cast<const Coord*>(normal);
}

Coord Coord3::X() const { return coords[0]; }
Coord Coord3::Y() const { return coords[1]; }
Coord Coord3::Z() const { return coords[2]; }
Coord Coord3::W() {
  if (! normal || normal[0] < -1.0) {
    magnitude();
  }
  return normal[3];
}
Coord Coord3::nX() const { return normal[0]; }
Coord Coord3::nY() const { return normal[1]; }
Coord Coord3::nZ() const { return normal[2]; }

const Coord3& Coord3::add(const Coord3& v3) {
  coords[0] += v3.coords[0];
  coords[1] += v3.coords[1];
  coords[2] += v3.coords[2];
  if (normal) {
    normal[0] = -8;
    return this->normalize();
  }
  return *this;
}

const Coord3& Coord3::subtract(const Coord3& v3) {
  coords[0] -= v3.coords[0];
  coords[1] -= v3.coords[1];
  coords[2] -= v3.coords[2];
  if (normal) {
    normal[0] = -8;
    return this->normalize();
  }
  return *this;
}

const Coord3& Coord3::multiply(Coord f) {
  coords[0] *= f;
  coords[1] *= f;
  coords[2] *= f;
  if (normal) {
    normal[0] = -8;
    return this->normalize();
  }
  return *this;
}

const Coord3& Coord3::divide(Coord f) {
  coords[0] /= f;
  coords[1] /= f;
  coords[2] /= f;
  if (normal) {
    normal[0] = -8;
    return this->normalize();
  }
  return *this;
}

const Coord3& Coord3::negate() {
  coords[0] = -coords[0];
  coords[1] = -coords[1];
  coords[2] = -coords[2];
  if (normal) {
    normal[0] = -8;
    return this->normalize();
  }
  return *this;
}

const Coord3& Coord3::normalize() {
  if (normal && normal[0] >= -1.0) {
    return *this;
  }
  return magnitude();
}

const Coord3& Coord3::magnitude() {
  if (! normal) {
    normal = new Coord[4];
  }
  normal[3] = sqrt(coords[0] * coords[0] + coords[1] * coords[1] + coords[2] * coords[2]);
  normal[0] = coords[0] / normal[3];
  normal[1] = coords[1] / normal[3];
  normal[2] = coords[2] / normal[3];
  return *this;
}

const Coord3& Coord3::cleanNormal() {
  if (! normal) {
    normal = new Coord[4];
  }
  normal[0] = -8.0;
  normal[1] = 0.0;
  normal[2] = 0.0;
  normal[3] = 0.0;
  return *this;
}

const Coord3& Coord3::deleteNormal() {
  if (normal) {
    delete [] normal;
    normal = 0;
  }
  return *this;
}

Coord Coord3::getMagnitude() {
  normalize();
  return normal[3];
}

Coord3 Coord3::getNormal() {
  normalize();
  Coord3 ret(normal[0], normal[1], normal[2]);
  return ret;
}

Coord Coord3::dot(Coord3& other) {
  normalize();
  other.normalize();
  return
    normal[0] * other.normal[0] +
    normal[1] * other.normal[1] +
    normal[2] * other.normal[2];
}

const Coord3& Coord3::cross(Coord3& other) {
  this->normalize();
  other.normalize();
  Coord tmp[3];
  tmp[0] = normal[1] * other.normal[2] + normal[2] * other.normal[1];
  tmp[1] = normal[0] * other.normal[2] + normal[2] * other.normal[0];
  tmp[2] = normal[0] * other.normal[1] + normal[1] * other.normal[0];
  coords[0] = tmp[0] * normal[3];
  coords[1] = tmp[1] * normal[3];
  coords[2] = tmp[2] * normal[3];
  normal[0] = -8;
  return normalize();
}

int Coord3::isParallel(Coord3& other) {
  if (coords[0] == 0 && other.coords[0] == 0)
    if ( (coords[1] == 0 && other.coords[1] == 0) ||
	 (coords[2] == 0 && other.coords[2] == 0) )
      return 1;
    else
      if (coords[1] / other.coords[1] == coords[2] / other.coords[2])
	return 1;
      else
	return 0;
  if (coords[1] == 0 && other.coords[1] == 0)
    if (coords[2] == 0 && other.coords[2] == 0)
      return 1;
    else
      if (coords[0] / other.coords[0] == coords[2] / other.coords[2])
	return 1;
      else
	return 0;
  normalize();
  other.normalize();
  if (normal[0] == other.normal[0] &&
      normal[1] == other.normal[1] &&
      normal[2] == other.normal[2])
    return 1;
  return 0;
}


std::istream& operator>>(std::istream& is, Coord3& c3) {
  Coord n[3];
  is >> n[0] >> n[1] >> n[2];
  c3.set(n);
  return is;
}

std::ostream& operator<<(std::ostream& os, const Coord3& c3) {
  os<<c3.coords[0]<<" "<<c3.coords[1]<<" "<<c3.coords[2];
  if (c3.normal)
    os<<"["<<c3.normal[0]<<" "<<c3.normal[1]<<" "<<c3.normal[2]<<" "<<c3.normal[3]<<"]";
  return os;
}
