/* UniEXP - Universo Experimental
 * Copyright (C) 1999,2002,2003,2004,2006,2007 Silvio Almeida
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#ifndef UX_VECTORS_H
#define UX_VECTORS_H


#include <uniexp/Numbers.h>


/** Vetor de coordenadas espacias.
    Coord3 é um vetor desenhado para representar coordenadas
    espaciais. Possui um mecanismo de normalização otimizado.
 */
class Coord3 {
  Coord coords[3];
  Coord* normal;
  friend std::istream& operator>>(std::istream& is, Coord3& coord3);
  friend std::ostream& operator<<(std::ostream& os, const Coord3& coord3);
protected:
  /** Normalizar um Coord3 é criar um vetor auxiliar que guarda as coordenadas
      normalizadas e o módulo, recalculados somente quando necessário.
      Se Coord3 já está normalizado recalcula a normalização quando normal[0] < 1.0.
      Retorna uma referência para o Coord3 normalizado.
  */
  const Coord3& normalize();
  /** força cálculo de normal */
  const Coord3& magnitude();
  /** instancia normal zerado, zera valores se existente, init -8 */
  const Coord3& cleanNormal();
  /** deleta vetor normal */
  const Coord3& deleteNormal();
public:
  Coord3(Coord x = 0.0, Coord y = 0.0, Coord z = 0.0);
  /** vec precisa ter size 3, inicializa sem normal */
  Coord3(const Coord* vec3);
  Coord3(const Coord3& vec3);
 ~Coord3();
  const Coord3& operator=(const Coord3& vec3);
  const Coord3 operator+(const Coord3& vec3) const;
  const Coord3 operator-(const Coord3& vec3) const;
  const Coord3 operator/(const Coord& k) const;
  const Coord3 operator*(const Coord& k) const;
  /** metodo set padrão */
  const Coord3& set(Coord x = 0.0, Coord y = 0.0, Coord z = 0.0);
  /** vec precisa ter size 4 */
  const Coord3& set(Coord* vec);
  const Coord3& setX(Coord x = 0.0);
  const Coord3& setY(Coord y = 0.0);
  const Coord3& setZ(Coord z = 0.0);
  const Coord3& setW(Coord w = 1.0);
  const Coord* get() const; /**< retorna Coord[3] */
  const Coord* getN() const; /**< retorna Coord[4] normal(x,y,z) + magnitude, assume normal */
  Coord X() const; /**< retorna X */
  Coord Y() const; /**< retorna Y */
  Coord Z() const; /**< retorna Z */
  Coord W(); /**< retorna magnitude e colateralmente liga auto-normal */
  Coord nX() const; /**< retorna X normalizado, assume normal */
  Coord nY() const; /**< retorna Y normalizado, assume normal */
  Coord nZ() const; /**< retorna Z normalizado, assume normal */
  const Coord3& negate();
  const Coord3& add(const Coord3& vec3);
  const Coord3& subtract(const Coord3& vec3);
  const Coord3& multiply(Coord f);
  const Coord3& divide(Coord f);
  Coord getMagnitude();
  /** Executa normalize() e retorna normal[] como um novo Coord3, descartando o módulo */
  Coord3 getNormal();
  /** Este Coord3 e vec3 são normalizados, caso ainda não estejam. */
  Coord dot(Coord3& vec3);
  /** Este Coord3 e vec3 são normalizados, caso ainda não estejam. */
  const Coord3& cross(Coord3& vec3);
  /** Este Coord3 e vec3 serão normalizados, caso ainda não estejam. */
  int isParallel(Coord3& vec3);
};



template<typename T>
class Vec3 {
  T vec[3];
public:
  Vec3(T x = 0, T y = 0, T z = 0);
  const Vec3& operator=(const Vec3& vec);
  const Vec3& set (T x = 0.0, T y = 0.0, T z = 0.0);
  const Vec3& setX(T x = 0.0);
  const Vec3& setY(T y = 0.0);
  const Vec3& setZ(T z = 0.0);
  T X() const;
  T Y() const;
  T Z() const;
};

template<typename T>
Vec3<T>::Vec3(T x, T y, T z) {
  vec[0] = x;
  vec[1] = y;
  vec[2] = z;
}

template<typename T>
const Vec3<T>& Vec3<T>::operator=(const Vec3& vec3) {
  if (&vec3 != this) {
    vec[0] = vec3.vec[0];
    vec[1] = vec3.vec[1];
    vec[2] = vec3.vec[2];
  }
  return *this;
}

template<typename T>
const Vec3<T>& Vec3<T>::set (T x, T y, T z) {
  vec[0] = x;
  vec[1] = y;
  vec[2] = z;
  return *this;
}

template<typename T> const Vec3<T>& Vec3<T>::setX(T x) { vec[0] = x; return *this; }
template<typename T> const Vec3<T>& Vec3<T>::setY(T y) { vec[1] = y; return *this; }
template<typename T> const Vec3<T>& Vec3<T>::setZ(T z) { vec[2] = z; return *this; }

template<typename T> T Vec3<T>::X() const { return vec[0]; }
template<typename T> T Vec3<T>::Y() const { return vec[1]; }
template<typename T> T Vec3<T>::Z() const { return vec[2]; }


#endif /* UX_VECTORS_H */

