/* UniEXP - Universo Experimental
 * Copyright (C) 1999,2002,2003,2004,2006,2007 Silvio Almeida
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#ifndef UX_UXAUXL_H
#define UX_UXAUXL_H


#include <iostream>
#include <vector>
#include <stdexcept>
#include <cassert>
#include <cerrno>


/** namespace _ux::
 *  símbolos internos do UniEXP
 */
namespace ux {


/** exceções */
class Excc : public std::runtime_error {
  int e;
public:
  Excc(const std::string& tx): std::runtime_error(tx), e(errno) { }
  int         num() const { return e     ; }
  const char* txt() const { return what(); }
};



/** @name Funções de Conversão
 *  funções para converter de/para std::string,
 *  emitem Excc se não podem converter.
 */
/*@{*/
char               _str_chr  (const std::string &);
unsigned char      _str_uchr (const std::string &);
short              _str_shr  (const std::string &);
unsigned short     _str_ushr (const std::string &);
int                _str_int  (const std::string &);
unsigned int       _str_uint (const std::string &);
long               _str_lng  (const std::string &);
unsigned long      _str_ulng (const std::string &);
long long          _str_llg  (const std::string &);
unsigned long long _str_ullg (const std::string &);
float              _str_flt  (const std::string &);
double             _str_dbl  (const std::string &);
std::string _str(const char               &);
std::string _str(const unsigned char      &);
std::string _str(const short              &);
std::string _str(const unsigned short     &);
std::string _str(const int                &);
std::string _str(const unsigned int       &);
std::string _str(const long               &);
std::string _str(const unsigned long      &);
std::string _str(const long long          &);
std::string _str(const unsigned long long &);
std::string _str(const float              &);
std::string _str(const double             &);
std::string _str(const char               *);
/*@}*/


/** @name objetos para iterar em std::ostream */
/*@{*/

/** std::for_each - prefixo, ITEM, sufixo */
template<typename I> struct OS_item: public std::unary_function<I, void> {
  std::ostream& os; std::string p1, p2;
  OS_item(std::ostream& o, const std::string& pre="", const std::string& pos="\n")
  :  os(o), p1(pre), p2(pos) { }
  void operator() (I i) { os<<p1<<i<<p2; }
};

/** std::for_each - prefixo, PAR.first, meio, PAR.second, sufixo */
template<typename P> struct OS_par: public std::unary_function<P, void> {
  std::ostream& os; std::string p1, p2, p3;
  OS_par(std::ostream& o,            const std::string& pre="",
     const std::string& meio=" => ", const std::string& pos="\n")
  :  os(o), p1(pre), p2(meio), p3(pos) { }
  void operator() (P i) { os<<p1<<i.first<<p2<<i.second<<p3; }
};

/*@}*/



/** Este recurso parece ser necessário ao copy contructor de Alloc<T> */
class Clonable {
public:
  /** redefinir em cada subclasse */
  virtual Clonable* factory() const = 0;
  /** redefinir em cada subclasse */
  virtual Clonable* clone() const = 0;
  /** não precisa redefinir nas subclasses de TPID */
  virtual std::istream& input(std::istream& is) = 0;
  /** não precisa redefinir nas subclasses de TPID */
  virtual std::ostream& output(std::ostream& os) const = 0;
};


/** instância verídica do abstrato Clonable */
class Virtual : public Clonable {
  friend std::istream& operator>>(std::istream& is, Virtual&) { return is; }
  friend std::ostream& operator<<(std::ostream& os, const Virtual&) { return os; }
public:
  Virtual() { }
  Virtual(const Virtual& src) { }
  virtual ~Virtual() { }
  const Virtual& operator=(const Virtual& src) { }
  /** redefinir em cada subclasse */
  virtual Clonable* factory() const { return new Virtual(); }
  /** redefinir em cada subclasse */
  virtual Clonable* clone() const { return new Virtual(*this); }
  /** não precisa redefinir nas subclasses */
  virtual std::istream& input(std::istream& is) { return is>>*this; }
  /** não precisa redefinir nas subclasses */
  virtual std::ostream& output(std::ostream& os) const { return os<<*this; }
};


class TP {
  std::string _tp;
  friend std::istream& operator>>(std::istream&, TP&);
  friend std::ostream& operator<<(std::ostream&, const TP&);
protected:
  void tp(const std::string& tt) { _tp = tt; }
public:
  TP(const std::string& tt = "_tp_") { _tp = tt; }
  TP(const TP& src) { tp(src.tp()); }
  virtual ~TP();
  const TP& operator= (const TP& src) { if (&src != this) { tp(src.tp()); } return *this; }
  std::string tp() const { return _tp; }
};

class ID {
  std::string _id;
  friend std::istream& operator>> (std::istream&, ID&);
  friend std::ostream& operator<< (std::ostream&, const ID&);
protected:
  void id(const std::string& ii) { _id = ii; }
public:
  ID(const std::string& ii = "_id_") { _id = ii; }
  ID(const ID& src) { id(src.id()); }
  /** Destrutor de virtual ID */
  virtual ~ID();
  const ID& operator= (const ID& src) { if (&src != this) { id(src.id()); } return *this; }
  std::string id() const { return _id; }
};

class TPID: public Clonable, public TP, public ID {
  friend std::istream& operator>> (std::istream&, TPID&);
  friend std::ostream& operator<< (std::ostream&, const TPID&);
protected:
  void tpid(const std::string& t, const std::string& i) { tp(t); id(i); }
public:
  TPID(const std::string& tp = "_tp_", const std::string& id = "_id_"): TP(tp), ID(id) { }
  TPID(const TPID& src) { tpid(src.tp(), src.id()); }
  /** Destrutor de virtual TPID */
  virtual ~TPID();
  const TPID& operator= (const TPID& src) { if (&src != this) { tp(src.tp()); id(src.id()); } return *this; }
  std::string tpid() const { return tp()+':'+id(); }
  /** redefinir em cada subclasse */
  virtual Clonable* factory() const { return new TPID(); }
  /** redefinir em cada subclasse */
  virtual Clonable* clone() const { return new TPID(*this); }
  /** não precisa redefinir nas subclasses */
  virtual std::istream& input(std::istream& is) { return is>>*this; }
  /** não precisa redefinir nas subclasses */
  virtual std::ostream& output(std::ostream& os) const { return os<<*this; }
};



}; /* namespace ux */


#endif  /* UX_UXAUXL_H */

