/* UniEXP - Universo Experimental
 * Copyright (C) 1999,2002,2003,2004,2006,2007 Silvio Almeida
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#ifndef UX_ATTRS_H
#define UX_ATTRS_H


#include <map>

#include <uniexp/Alloc.h>



/** dispositivos de qualificação de objetos.
 *  todo objeto do UniEXP possui um conjunto de características
 *  que o definem. Tais características são gerenciadas e
 *  manipuladas por objetos ux::Attrs.
 */

namespace ux {


/** implementação de um Atributo.
 *  esta classe implementa a interface Attr_0 para tipos arbitrários.
 *
 *  há um construtor e um operador= que recebem ponteiros
 *  e usam uma referência para argumento ao invés da
 *  alocação dinâmica local. A regra aqui é: se você criar um
 *  Attr A como cópia de um ponteiro Attr* B, ou se modificar
 *  A usando operator=(Attr* B), você é responsável por efetuar
 *  delete B quando A (ou outro objeto) não o referenciar mais.
 *  
 *  para alocar um Attr<X> com um tipo X que não implementa
 *  os operadores padrão de inserção e extração,  deve ser
 *  suficiente escrevê-los em função da especialização Attr<X>.
 */
template<typename T>
class Attr : public Virtual, public Alloc<T> {
  friend class Attrs;
  template <typename TT> friend std::istream& operator>> (std::istream&, Attr<TT>&);
  template <typename TT> friend std::ostream& operator<< (std::ostream&, const Attr<TT>&);
  /** construtor padrão privado */
  Attr() { }
public:
  /** construtor por cópia */
  Attr (const Attr<T>& src);
  /** construtor por cópia */
  Attr (const T& val);
  /** construtor por ponteiro (adquire) */
  Attr (T* val);
  /** construtor para referência automática em target */
  Attr (Attr<T>* target);
  virtual ~Attr() { }
  const Attr<T>& operator=(const Attr<T>& src);
  const Attr<T>& operator=(Attr<T>* src);

  virtual Clonable* factory() const { return new Attr<T>(*this); }
  virtual Clonable* clone() const { return new Attr<T>(*this); }
  virtual std::istream& input(std::istream& is) { return is>>*this; }
  virtual std::ostream& output(std::ostream& os) const { return os<<*this; }

  virtual const T* ptr() const;
  virtual operator T ();
};


template<typename T>
Attr<T>::Attr(const Attr<T>& attr)
  : Alloc<T>(*(attr.ptr()))
{ }

template<typename T>
Attr<T>::Attr(const T& value)
  : Alloc<T>(value)
{ }

template<typename T>
Attr<T>::Attr(T* value)
  : Alloc<T>(value)
{ }

template<typename T>
Attr<T>::Attr(Attr<T>* attr)
  : Alloc<T>(static_cast<Alloc<T>*>(attr))
{ }

template <typename T>
const Attr<T>& Attr<T>::operator=(const Attr<T>& src) {
  if (src != this) {
    static_cast<Alloc<T>*>(this)->operator=(src);
  }
  return *this;
}

template <typename T>
const Attr<T>& Attr<T>::operator=(Attr<T>* src) {
  if (src != this) {
    static_cast<Alloc<T>*>(this)->operator=(src);
  }
  return *this;
}

template <typename T>
const T* Attr<T>::ptr() const {
  return static_cast<const Alloc<T>*>(this)->ptr();
}

template <typename T>
Attr<T>::operator T () {
  T ret;
  const T* ptr = this->ptr();
  if (ptr) {
    ret = *ptr;
  }
  return ret;
}

template<typename T>
std::istream& operator>>(std::istream& is, Attr<T>& attr) { return is; }

template<typename T>
std::ostream& operator<<(std::ostream& os, const Attr<T>& attr) {
  const T* aPtr = attr.ptr();
  if (aPtr) {
    return os<<*aPtr;
  } else {
    return os<<"NULL";
  }
}



typedef std::map<std::string, Virtual*> _m_sV;
typedef _m_sV::iterator _im_sV;
typedef _m_sV::const_iterator _cim_sV;
typedef _m_sV::value_type _pm_sV;

/** Attrs, um conjunto determinado de atributos.
 */
class Attrs : public TPID, public _m_sV {
  friend std::istream& operator>>(std::istream&, Attrs&);
  friend std::ostream& operator<<(std::ostream&, const Attrs&);
public:
  Attrs(const Attrs& src);
  Attrs(const std::string& tp = "Attrs", const std::string& id = "noid");
  Attrs(const _m_sV& m, const std::string& tp = "Attrs", const std::string& id = "noid");
  ~Attrs();
  const Attrs& operator=(const Attrs& src);

  virtual Clonable* factory() const { return new Attrs(); }
  virtual Clonable* clone() const { return new Attrs(*this); }
  virtual std::istream& input(std::istream& is) { return is>>*this; }
  virtual std::ostream& output(std::ostream& os) const { return os<<*this; }

  template<typename T> bool exists(const std::string& attrName) const;
  template<typename T> T getAttribute(const std::string& attrName) const;
  template<typename T> const T* getAttributePtr(const std::string& attrName) const;
  template<typename T> bool newAttribute(const std::string& attrName, const T& attrValue);
  template<typename T> bool setAttribute(const std::string& attrName, const T& attrValue);
  template<typename T> int rmAttribute(const std::string& attrName);
};

template<typename T> bool Attrs::exists(const std::string& attrName) const {
  _cim_sV i = find(attrName);
  if (i != end()) {
    try {
      if (dynamic_cast<const Attr<T>*>(i->second)) {
	return true;
      } else {
	return false;
      }
    } catch (...) {
      std::cerr<<"Exception: Attr<>::exists("<<attrName<<")"<<std::endl;
      return false;
    }
  } else {
    return false;
  }
}

template<typename T> T Attrs::getAttribute(const std::string& attrName) const {
  T ret;
  _cim_sV i = find(attrName);
  if (i != end()) {
    Virtual* base = i->second;
    const Attr<T>* attrPtr;
    try {
      attrPtr = dynamic_cast<Attr<T>*>(base);
      if (attrPtr) {
	ret = *(attrPtr->ptr());
      }
      return ret;
    } catch (...) {
      std::cerr<<"throw exception, attrName aponta para nao Attr<T>"<<std::endl;
      return ret;
    }
  } else {
    return ret;
  }
}

template<typename T> const T* Attrs::getAttributePtr(const std::string& attrName) const {
  _cim_sV i = find(attrName);
  if (i != end()) {
    return dynamic_cast<const Attr<T>*>(i->second)->ptr();
  } else {
    return 0;
  }
}


template<typename T> bool Attrs::setAttribute(const std::string& attrName, const T& attrValue) {
  _im_sV i = find(attrName);
  if (i != end()) {
    Virtual* ptr = i->second;
    if (ptr) {
      delete ptr;
    }
    erase(i);
  }
  Attr<T>* attr = new Attr<T>(attrValue);
  return insert(std::pair<std::string, Virtual*>(attrName, attr)).second;
}

template<typename T>
bool Attrs::newAttribute(const std::string& attrName, const T& attrValue) {
  Attr<T>* attr = new Attr<T>(attrValue);
  return insert(std::pair<std::string, Virtual*>(attrName, attr)).second;
}

template<typename T> int Attrs::rmAttribute(const std::string& attrName) {
  /* esta implementação está sujeita a segfault ou leak? */
  _im_sV i = find(attrName);
  if (i != end()) {
    erase(i);
    return 1;
  }
  return 0;
}


}; /* namespace ux */


#endif  /* UX_ATTRS_H */
