/* UniEXP - Universo Experimental
 * Copyright (C) 1999,2002,2003,2004,2006,2007 Silvio Almeida
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#ifndef UX_ART_0_H
#define UX_ART_0_H


#include <iostream>
#include <string>

#include <uniexp/Attrs.h>



/** dispositivos de estrutura e articulação.
 *  o namespace UXelms implementa mecanismos de construção
 *  e articulação dos nós de uma estrutura em árvore.
 *
 *  as árvores do UniEXP permitem nós com vários pais e
 *  caminhos cíclicos. Naturalmente varreduras devem
 *  ser executadas apenas em segmentos acíclicos.
 */

namespace ux {


class Art_0;

/** _fn_varr.
    implementação concreta de unary_function
    que opera com base no parâmetro Art_0*.
 */
class _fn_varr : public TPID, public std::binary_function<Art_0*, Attrs*, void> {
  friend std::istream& operator>>(std::istream& is, _fn_varr& fn);
  friend std::ostream& operator<<(std::ostream& os, const _fn_varr& fn);
protected:
  size_t _count;
public:
  _fn_varr();
  _fn_varr(const _fn_varr& src);
  _fn_varr(const std::string& tp, const std::string& id);
  virtual ~_fn_varr();

  virtual Clonable* factory() const { return new _fn_varr(); }
  virtual Clonable* clone() const { return new _fn_varr(*this); }
  virtual std::istream& input(std::istream& is) { return is>>*this; }
  virtual std::ostream& output(std::ostream& os) const { return os<<*this; }

  virtual void init();
  virtual void operator() (Art_0* art, Attrs* attrs = 0);
};

/** ramificações, nós, pontos de articulação.
 *  
 *  a classe Art é a interface abstrata para os
 *  diferentes tipos de árvores do UniEXP.
 *
 *  nesta interface são abordadas as funções de inserção
 *  e remoção de ítens, enquanto que os
 *  mecanismos de navegação e varredura localizam-se nas
 *  classes derivadas; qualquer árvore deve implementar,
 *  ainda que a custo elevado, localização por nome.
 */
class Art_0 : public TPID {
  friend std::istream& operator>>(std::istream& is, Art_0& art);
  friend std::ostream& operator<<(std::ostream& os, const Art_0& art);
protected:
  const Art_0* _base;
  AllocC<Attrs> _attrs;
  /** adota e retorna filho recém adicionado às articulações */
  virtual Art_0* adot(Art_0* art);
  /** deserda e detona filho recém desarticulado */
  virtual void desh(Art_0* art);
  virtual void clear() { } // pure virtual?
public:
  /** constructor default */
  Art_0();
  /** copy constructor for TPID */
  Art_0(const Art_0& src);
  /** default constructor for TPID */
  Art_0(const std::string& tp, const std::string& id);
  /** constructor por Attrs */
  Art_0(const Attrs& attrs);
  /** constructor por autoref em Attrs */
  Art_0(AllocC<Attrs>* alloc);
  virtual ~Art_0();
  const Art_0& operator=(const Art_0& art);

  /** sobrescrito em subclasses */
  virtual Clonable* factory() const { return new Art_0(); }
  /** sobrescrito em subclasses */
  virtual Clonable* clone() const { return new Art_0(*this); }
  virtual std::istream& input(std::istream& is) { return is>>*this; }
  virtual std::ostream& output(std::ostream& os) const { return os<<*this; }

  /** pai: const Art_0* */
  virtual const Art_0* base() const;
  /** raiz: const Art_0* */
  virtual const Art_0* root() const;
  /** acesso aos atributos */
  virtual Attrs* getAttrs() { return _attrs.ptr(); }
  /** procura Art_0* somente em subelms: posição em subelms */
  virtual Art_0* art(size_t pos) { return 0; } // pure virtual?
  virtual const Art_0* art(size_t pos) const { return 0; } // pure virtual?
  /** procura Art_0* somente em subelms: nome do Art_0 */
  virtual Art_0* art(const std::string& artName) { return 0; } // pure virtual?
  virtual const Art_0* art(const std::string& artName) const { return 0; } // pure virtual?
  /** procura Art_0* somente em subelms: nome do Attrs associado */
  virtual Art_0* art_e(const std::string& attrsName) { return 0; } // pure virtual?
  virtual const Art_0* art_e(const std::string& attrsName) const { return 0; } // pure virtual?
  /** procura Art_0* em this e recursivamente em subelms: nome do Art_0 */
  virtual Art_0* sub(const std::string& artName) { return 0; } // pure virtual?
  virtual const Art_0* sub(const std::string& artName) const { return 0; } // pure virtual?
  /** procura Art_0* em this e recursivamente em subelms: nome do Attrs associado */
  virtual Art_0* sub_e(const std::string& attrsName) { return 0; } // pure virtual?
  virtual const Art_0* sub_e(const std::string& attrsName) const { return 0; } // pure virtual?
  /** procura Art_0* em base, this e subelms recursivamente pelo caminho */
  virtual Art_0* sub_c(const std::string& path);
  virtual const Art_0* sub_c(const std::string& path) const;

  virtual bool empty() const { return true; }
  virtual size_t size() const { return 0; }
  /** determina profundidade absoluta, ou seja, até que _ba==0 */
  virtual int prof() const;
  /** determina profundidade até que _ba==r ou _ba==0 */
  virtual int prof(Art_0* ref) const;
  /** retorna caminho desde a origem */
  virtual std::string path() const;

  virtual void ad_ini(Art_0*) { }
  virtual void ad_fim(Art_0*) { }
  virtual bool ad_pos(Art_0*, int pos) { return 0; } /**< insere em p */
  virtual bool ad_pre(Art_0*, Art_0* ref) { return 0; } /**< insere _r */
  virtual bool ad_pos(Art_0*, Art_0* ref) { return 0; } /**< insere r_ */
  virtual void rm_pri() { }
  virtual void rm_ult() { }
  virtual bool rm_pos(int pos) { return 0; }
  virtual bool rm_art(Art_0* art) { return 0; }

  /** varredura com functor */
  virtual void xx(_fn_varr* fn, Attrs* attrs = 0) { }
  /** varredura reversa com functor */
  virtual void zx(_fn_varr* fn, Attrs* attrs = 0) { }
  /** varredura com pre e post functors */
  virtual void pp(_fn_varr* preFn, _fn_varr* postFn, Attrs* attrs = 0) { }
  /** varredura com pre e post functors entre cada filho */
  virtual void cc(_fn_varr* preChildFn, _fn_varr* postChildFn, Attrs* attrs = 0) { }
  /** varredura pp + cc */
  virtual void ppcc(_fn_varr* preFn, _fn_varr* postFn, _fn_varr* preChildFn, _fn_varr* postChildFn, Attrs* attrs = 0) { }
  
  /** obtém atributo local, ou T() */
  template<typename T> T localAttr(const std::string& attrName) const;
  /** obtém ponteiro para atributo local, ou zero */
  template<typename T> T* localAttrPtr(const std::string& attrName);
  /** obtém atributo ancestral, ou T() */
  template<typename T> T attr(const std::string& attrName) const;
  /** obtém ponteiro para atributo ancestral, ou zero */
  template<typename T> T* attrPtr(const std::string& attrName);
};


/** bizarro, erro de sintaxe com
    _attrs.ptr()->getAttributes<T>(attrName),
    é um bug do gcc corrigido na versão 3.4.x.
*/
template<typename T> T Art_0::localAttr(const std::string& attrName) const {
  return _attrs.ptr()->template getAttribute<T>(attrName);
}

template<typename T> T* Art_0::localAttrPtr(const std::string& attrName) {
  return _attrs.ptr()->template getAttributePtr<T>(attrName);
}

template<typename T> T Art_0::attr(const std::string& attrName) const {
  T* ptr = _attrs.ptr()->template getAttributePtr<T>(attrName);
  return ptr ? *ptr : _base ? _base->attr<T>(attrName) : T();
}

template<typename T> T* Art_0::attrPtr(const std::string& attrName) {
  T* ptr = _attrs.ptr()->template getAttributePtr<T>(attrName);
  return ptr ? *ptr : _base ? _base->attrPtr<T>(attrName) : 0;
}


}; /* namespace ux */

#endif  /* UX_ART_0_H */

