/* UniEXP - Universo Experimental
 * Copyright (C) 1999,2002,2003,2004,2006,2007 Silvio Almeida
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#ifndef UX_ALLOC_H
#define UX_ALLOC_H


#include <iostream>
#include <list>
#include <typeinfo>

#include <uniexp/Auxl.h>


namespace ux {


/** alocador externo-local automático.
 *  isto não é std::allocator, trata-se de uma classe  básica
 *  que implementa um mecanismo para alocação dinâmica e para
 *  referências automáticas
 */
template<typename T>
class Alloc {
  template <typename TT> friend std::istream& operator>> (std::istream&, Alloc<TT>&);
  template <typename TT> friend std::ostream& operator<< (std::ostream&, const Alloc<TT>&);
protected:
  typedef std::list<Alloc<T>*> _l_A;
  typedef typename _l_A::iterator _il_A;
  typedef typename _l_A::const_iterator _cil_A;
  /** ponteiro; indica alocação local */
  T* _ptr;
  /** base; mecanismos de ponteiros */
  Alloc<T>* _base;
  /** referências; mecanismo de destruição automática */
  _l_A _refs;
  void pushRef(Alloc<T>* alloc);
  void eraseRef(Alloc<T>* a);
public:
  /** contrutor default - cópia direta */
  Alloc(const T* ptr = 0);
  /** construindo por cópia direta */
  Alloc(const T& val);
  /** construindo por cópia */
  Alloc(const Alloc<T>& src);
  /** construindo para referência automática */
  Alloc(Alloc<T>* alloc);
  ~Alloc();
  Alloc& operator=(const T* ptr);
  Alloc& operator=(const Alloc<T>& alloc);
  Alloc& operator=(Alloc<T>* alloc);

  bool operator==(T* cmp) const { return ptr() == cmp; }

  Alloc<T>* alloc();
  const Alloc<T>* alloc() const;
  inline T* ptr();
  inline const T* ptr() const;
  inline operator T();
  inline operator T() const;
  size_t nRefs() const;
};



template<typename T>
  Alloc<T>::Alloc(const T* ptr) {
  _ptr = ptr ? new T(*ptr) : new T();
  _base = 0;
}

template<typename T>
Alloc<T>::Alloc(const T& val)
  : _base(0) {
    _ptr = new T(val);
}

template<typename T>
Alloc<T>::Alloc(const Alloc<T>& src) {
  try {
    _ptr = new T(*(src.ptr()));
    _base = src._base;
  } catch (...) {
    std::cerr<<"ALERTA: tipo não implementa factory para Alloc<"<<src<<">"<<std::endl;
  }
}

template<typename T>
Alloc<T>::Alloc(Alloc<T>* alloc)
  : _ptr(0), _base(alloc) {
  if (alloc)
    alloc->pushRef(this);
}

template<typename T>
Alloc<T>::~Alloc() {
  if (! _refs.empty()) {
    _il_A i = _refs.begin();
    _il_A pos = i++;
    _il_A end = _refs.end();
    /** transmite comando de âncora ao primeiro referral */
    if (_ptr) {
      (*pos)->_ptr = _ptr;
    }
    if (_base != 0) {
      (*pos)->_base = _base;
    }
    /** ancora demais referrals no primeiro */
    for (; i!=end; i++) {
      (*i)->_base = (*pos);
      (*pos)->pushRef(*i);
    }
  } else {
    if (_ptr) {
      delete _ptr;
      _ptr = 0;
    } else {
      if (_base) {
	_base->eraseRef(this);
      }
    }
  }
}

template<typename T>
Alloc<T>& Alloc<T>::operator=(const T* ptr) {
  if (_ptr && _ptr != ptr) {
    delete _ptr;
  }
  _ptr = ptr ? new T(*ptr) : new T();
  return *this;
}

template<typename T>
Alloc<T>& Alloc<T>::operator=(const Alloc<T>& src) {
  if (this != &src) {
    if (_ptr)
      delete _ptr;
    if (src.ptr()) {
      _ptr = new T(*(src.ptr()));
    } else {
      _ptr = new T();
    }
    _base=0;
  }
  return *this;
}

template<typename T>
Alloc<T>& Alloc<T>::operator=(Alloc<T>* src)
{
  if (src && this != src) {
    if (_ptr) {
      delete _ptr;
      _ptr = 0;
    }
    _base = src;
  }
  return *this;
}


template<typename T>
Alloc<T>* Alloc<T>::alloc()
{
  return this;
}

template<typename T>
const Alloc<T>* Alloc<T>::alloc() const
{
  return this;
}

template<typename T>
T* Alloc<T>::ptr()
{
  return _ptr ? _ptr : _base? _base->ptr() : 0;
}

template<typename T>
const T* Alloc<T>::ptr() const
{
  return _ptr ? const_cast<const T*>(_ptr) : _base ? _base->ptr() : 0;
}

template<typename T>
Alloc<T>::operator T()
{
  return _ptr ? *_ptr : (_base && _base->ptr()) ? *(_base->ptr()) : 0;
}

template<typename T>
Alloc<T>::operator T() const
{
  return const_cast<const T>(this->operator T());
}

template<typename T>
size_t Alloc<T>::nRefs() const
{
  return _refs.size();
}

template<typename T>
void Alloc<T>::pushRef(Alloc<T>* alloc)
{
  _il_A i = find(_refs.begin(), _refs.end(), alloc);
  if (i == _refs.end()) {
    _refs.push_back(alloc);
  }
}

template<typename T>
void Alloc<T>::eraseRef(Alloc<T>* alloc)
{
  _il_A i = find(_refs.begin(), _refs.end(), alloc);
  if (i != _refs.end()) {
    _refs.erase(i);
  }
}


template<typename T>
std::istream& operator>>(std::istream& is, Alloc<T>& alloc) {
  return is;
}

template<typename T>
std::ostream& operator<<(std::ostream& os, const Alloc<T>& alloc) {
  return os<<"Alloc<"<<*(alloc.ptr())<<">";
}


/** AllocC implementa constructors via clone() e factory()
 */
template<typename T>
class AllocC : public Alloc<T> {
public:
  /** contrutor default */
  AllocC(const T* ptr = 0);
  /** construindo por cópia */
  AllocC(const AllocC<T>& src);
  /** construindo por cópia direta */
  AllocC(const T& val);
  /** construindo para referência automática */
  AllocC(AllocC<T>* allocc);
  AllocC<T>& operator=(const T* ptr);
  AllocC<T>& operator=(const AllocC<T>& src);
};


template<typename T>
AllocC<T>::AllocC(const T* ptr) {
  if (&(T::clone) != 0) {
    Alloc<T>::_ptr = ptr ? static_cast<T*>(ptr->clone()) : static_cast<T*>(ptr->factory());
    Alloc<T>::_base = 0;
  }
}

template<typename T>
AllocC<T>::AllocC(const AllocC<T>& src) {
  if (&(T::clone) != 0) {
    Alloc<T>::_ptr =0;
    Alloc<T>::_base = 0;
    try {
      Alloc<T>::_ptr = static_cast<T*>(src.ptr()->clone());
    } catch (...) {
      std::cerr<<"ALERTA: tipo não implementa clone() --- Alloc< "<<src<<" >"<<std::endl;
    }
  }
}

template<typename T>
AllocC<T>::AllocC(const T& val) {
  if (&(T::clone) != 0) {
    Alloc<T>::_base = 0;
    Alloc<T>::_ptr = static_cast<T*>(val.clone());
  }
}

template<typename T>
AllocC<T>::AllocC(AllocC<T>* allocc) {
  if (&(T::clone) != 0 && allocc) {
    Alloc<T>::_ptr = 0;
    Alloc<T>::_base = allocc;
    allocc->pushRef(this);
  }
}

template<typename T>
AllocC<T>& AllocC<T>::operator=(const T* ptr) {
  if (Alloc<T>::_ptr && Alloc<T>::_ptr != ptr) {
    delete Alloc<T>::_ptr;
  }
  Alloc<T>::_ptr = ptr ? static_cast<T*>(ptr->clone()) : static_cast<T*>(ptr->factory());
  return *this;
}

template<typename T>
AllocC<T>& AllocC<T>::operator=(const AllocC<T>& src) {
  if (this != &src) {
    if (Alloc<T>::_ptr)
      delete Alloc<T>::_ptr;
    if (src.ptr()) {
      Alloc<T>::_ptr = static_cast<T*>(src.ptr()->clone());
    } else {
      Alloc<T>::_ptr = static_cast<T*>(src.ptr()->factory());
    }
    Alloc<T>::_base=0;
  }
  return *this;
}

}; /* namespace ux */

#endif  /* UX_UXALLOC_H */


