/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   connection.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: the connection object is a low-level wrapper over
 *              ClanLib's network API, it's the equivalent of a socket
 */


/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "connection.h"
#include "debug.h"

/*---------------------------------------------------------------------------
 variants
 ---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * creates a connection on a specific channel
 */
U61_Connection::U61_Connection(CL_NetSession *s,CL_NetComputer c,
			       int ch,bool give_rights)
{
  session=s;
  channel=ch;
  opened=true;

  U61_LOG_DEBUG("Opening connection on channel "<<ch<<"");

  group.add(c);

  if (give_rights)
    {
      /*
       * for now I decided to authorize anyone to communicate on the channel
       * since I'm not so concerned about u61's security 8-)
       */
      try 
	{
	  session->set_access(ch,group,
			   ACCESS_CHANNEL_READ|ACCESS_CHANNEL_WRITE);
	  U61_LOG_DEBUG("Connection access granted");
	}
      catch (CL_Error e)
	{
	  U61_LOG_WARNING("Access error : "<<e.message.c_str()<<"");
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * deletes a connection
 */
U61_Connection::~U61_Connection()
{
  close();
}

/*--------------------------------------------------------------------------*/
/*
 * sends a packet on the connection.
 * returns false if the sending operation failed
 */
bool U61_Connection::send(U61_Packet *p)
{
  int size;
  bool result=true;

  if (opened)
    {
      size=p->size;
      if (size>U61_PACKET_MAX_DATA)
	{
	  size=U61_PACKET_MAX_DATA;
	}

      CL_NetMessage message(p->data,size);

      try
	{
	  session->send(channel,group,message,true);
	}
      catch (CL_Error e)
	{
	  result=false;
	  U61_LOG_WARNING("Send error : "<<e.message.c_str());
	}
    }

  return result;
}
   
/*--------------------------------------------------------------------------*/
/*
 * sends a string on the connection.
 * returns false if the sending operation failed
 * The string will be truncated at the "max" length if it's too long
 */
bool U61_Connection::send(char *str,int max)
{
  int size;
  bool result=true;

  if (opened)
    {
      size=strlen(str);
      if (size>max)
	{
	  size=max;
	}

      CL_NetMessage message(str,size);

      try
	{
	  session->send(channel,group,message,true);
	}
      catch (CL_Error e)
	{
	  result=false;
	  U61_LOG_WARNING("Send error : "<<e.message.c_str());
	}
    }

  return result;
}
   
/*--------------------------------------------------------------------------*/
/*
 * sends an int on the connection.
 * returns false if the sending operation failed
 */
bool U61_Connection::send(int i)
{
  bool result=true;
  unsigned char buffer[4];

  if (opened)
    {
      buffer[0]=0xFF & i;
      buffer[1]=0xFF & i>>8;
      buffer[2]=0xFF & i>>16;
      buffer[3]=i>>24;
      CL_NetMessage message(buffer,4);

      try
	{
	  session->send(channel,group,message,true);
	}
      catch (CL_Error e)
	{
	  result=false;
	  U61_LOG_WARNING("Send error : "<<e.message.c_str());
	}
    }

  return result;
}
   
/*--------------------------------------------------------------------------*/
/*
 * sends an event on the connection.
 * returns false if the sending operation failed
 */
bool U61_Connection::send(U61_Event evt)
{
  bool result=true;
  unsigned char buffer[8]; 

  if (opened)
    {
      evt.serialize(buffer);
      CL_NetMessage message(buffer,8);

      try
	{
	  session->send(channel,group,message,true);
	}
      catch (CL_Error e)
	{
	  result=false;
	  U61_LOG_WARNING("Send error : "<<e.message.c_str());
	}
    }

  return result;
}
   
/*--------------------------------------------------------------------------*/
/*
 * returns true if there's a packet available on this connection
 */
bool U61_Connection::peek()
{
  bool result=false;
  if (opened)
    {  
      result=session->peek(channel);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the next packet available on this connection
 */
bool U61_Connection::recv(U61_Packet *p)
{
  CL_NetMessage message;
  int size;
  bool result=true;

  if (opened)
    {
      try 
	{
	  message=session->receive(channel,-1);

	  size=message.data.size();
	  if (size>U61_PACKET_MAX_DATA)
	    {
	      size=U61_PACKET_MAX_DATA;
	    }

	  p->size=size;
	  memcpy(p->data,message.data.data(),size);
	}
      catch (CL_Error e)
	{
	  result=false;
	  U61_LOG_WARNING("Receive error : "<<e.message.c_str()); 
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the next string available on this connection
 * The string is written in a buffer, which should be large enough
 * to receive all the data... Beware, the available size must be
 * of "max+1" since a trailing 0 must be added
 * However, the data will be truncated at the size given be the "max" arg.
 */
bool U61_Connection::recv(char *str,int max)
{
  CL_NetMessage message;
  int size;
  bool result=true;

  if (opened)
    {
      try 
	{
	  message=session->receive(channel,-1);

	  size=message.data.size();
	  if (size>max)
	    {
	      size=max;
	    }

	  memcpy(str,message.data.data(),size);
	  str[size]=0;
	}
      catch (CL_Error e)
	{
	  result=false;
	  U61_LOG_WARNING("Receive error : "<<e.message.c_str());
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the next int available on this connection
 */
bool U61_Connection::recv(int *i)
{
  CL_NetMessage message;
  bool result=true;
  int size;
  unsigned char *buffer;

  if (opened)
    {
      try 
	{
	  message=session->receive(channel,-1);

	  size=message.data.size();
	  if (size!=sizeof(*i))
	    {
	      *i=0;
	      result=false;
	    }
	  else
	    {
	      buffer=(unsigned char *) message.data.data();
	      (*i)=((unsigned int) buffer[3])<<24 
		| ((unsigned int) buffer[2])<<16 
		| ((unsigned int) buffer[1])<<8 
		| ((unsigned int) buffer[0]);
	    }  
	}
      catch (CL_Error e)
	{
	  result=false;
	  U61_LOG_WARNING("Receive error : "<<e.message.c_str());
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the next event available on this connection
 */
bool U61_Connection::recv(U61_Event *evt)
{
  CL_NetMessage message;
  bool result=true;
  int size;

  if (opened)
    {
      try 
	{
	  message=session->receive(channel,-1);

	  size=message.data.size();
	  if (size!=8)
	    {
	      result=false;
	      U61_LOG_WARNING("Event size is incorrect");
	    }
	  else
	    {
	      evt->unserialize((unsigned char *) message.data.data());
	      U61_LOG_DEBUG("Event "<<*evt<<" received");
	    }  
	}
      catch (CL_Error e)
	{
	  result=false;
	  U61_LOG_WARNING("Receive error : "<<e.message.c_str());  
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * closes the connection
 */
void U61_Connection::close()
{
  char buffer[2];

  /*
   * We flush the incoming queue
   */
  while (peek())
    {
      recv(buffer,1);
    }

  group.disconnect();

  opened=false;
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if the computer is associated to this connection
 * Used to kill players
 */
bool U61_Connection::is_associated(CL_NetComputer *c)
{
  bool result=false;

  if (group.get_computers().front()==*c)
    {
      result=true;
    }

  return result;
}









































