/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   config.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: the config object holds all the values which are stored
 *              on the HD between two game sessions. it would for instance
 *              store the player names and key settings
 *              it basically provides functions to load and save all the
 *              game parameters, and makes these parameters usable by any
 *              other class.
 *              it also provides some function which help handling game
 *              settings in general
 */


/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "config.h"
#include "platform.h"
#include "script.h"
#include "debug.h"
#include "sound.h"
#include "music.h"
#include "global.h"

#include <ClanLib/lua.h>

/*---------------------------------------------------------------------------
 constants
 ---------------------------------------------------------------------------*/

#define U61_CONFIG_DUMMY_VALUE 1001

#define U61_CONFIG_ID_DUMMY "cfg_dummy"
#define U61_CONFIG_ID_PLAYER "cfg_player_%d_"

#define U61_CONFIG_ID_LAUNCH_COUNT "cfg_launch_count"
#define U61_CONFIG_ID_DATA_DIR "cfg_data_dir"
#define U61_CONFIG_ID_SCRIPT_DIR "cfg_script_dir"
#define U61_CONFIG_ID_DATA_FILE "cfg_data_file"
#define U61_CONFIG_ID_SCRIPT_FILE "cfg_script_file"

#define U61_CONFIG_ID_FULLSCREEN "cfg_fullscreen"

#define U61_CONFIG_ID_SERVER_ID "cfg_server_id"
#define U61_CONFIG_ID_NETWORK_PORT "cfg_network_port"
#define U61_CONFIG_ID_NETWORK_CHANNEL "cfg_network_channel"

#define U61_CONFIG_ID_INITIAL_SPEED "cfg_initial_speed"
#define U61_CONFIG_ID_ACCELERATION "cfg_acceleration"
#define U61_CONFIG_ID_CURSE_DELAY "cfg_curse_delay"

#define U61_CONFIG_ID_SOUND_VOLUME "cfg_sound_volume"
#define U61_CONFIG_ID_MUSIC_VOLUME "cfg_music_volume"

#define U61_CONFIG_ID_MAX_FPS "cfg_max_fps"

#define U61_CONFIG_ID_NETWORK_ANTICIPATION "cfg_network_anticipation"

#define U61_CONFIG_ID_FIRST_REPEAT_MOVE "first_repeat_move"
#define U61_CONFIG_ID_STANDARD_REPEAT_MOVE "standard_repeat_move"
#define U61_CONFIG_ID_FIRST_REPEAT_ROTATE "first_repeat_rotate"
#define U61_CONFIG_ID_STANDARD_REPEAT_ROTATE "standard_repeat_rotate"
#define U61_CONFIG_ID_FIRST_REPEAT_OTHER "first_repeat_other"
#define U61_CONFIG_ID_STANDARD_REPEAT_OTHER "standard_repeat_other"

#define U61_CONFIG_ID_ANTICIPATION "anticipation"
#define U61_CONFIG_ID_ACTIVABLE "playing"
#define U61_CONFIG_ID_NAME "name"

#define U61_CONFIG_ID_KEY_START_STOP "key_start_stop"
#define U61_CONFIG_ID_KEY_MOVE_LEFT "key_move_left"
#define U61_CONFIG_ID_KEY_MOVE_RIGHT "key_move_right"
#define U61_CONFIG_ID_KEY_ROTATE_LEFT "key_rotate_left"
#define U61_CONFIG_ID_KEY_ROTATE_RIGHT "key_rotate_right"
#define U61_CONFIG_ID_KEY_MOVE_DOWN "key_move_down"
#define U61_CONFIG_ID_KEY_DROP "key_drop"
#define U61_CONFIG_ID_KEY_NEXT_VICTIM "key_next_victim"
#define U61_CONFIG_ID_KEY_PREV_VICTIM "key_prev_victim"
#define U61_CONFIG_ID_KEY_USE_ANTIDOTE "key_use_antidote"

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * creation with default values
 */ 
U61_Config::U61_Config()
{

}

/*--------------------------------------------------------------------------*/
/*
 * destruction
 */
U61_Config::~U61_Config()
{

}

/*--------------------------------------------------------------------------*/
/* 
 * initialization with values from the config file
 */ 
bool U61_Config::read()
{
  bool ok=false;
  char *filename;
  
  U61_Global::data.chdir_to_initial_dir();

  U61_LOG_DEBUG("read config");
  filename=U61_Platform::get_config_file();
  U61_LOG_MESSAGE("Loading config from \""<<filename<<"\".");

  U61_Script::do_file(filename);

  if (read_int(U61_CONFIG_ID_DUMMY,0)==U61_CONFIG_DUMMY_VALUE)
    {
      ok=true;
    }

  reset();
  
  U61_Global::data.chdir_to_data_dir();

  return ok;
}
   
/*--------------------------------------------------------------------------*/
/* 
 * initialization with lua config file values
 */ 
void U61_Config::reset()
{
  int pl;
  char default_data_dir[U61_CONST_STRING_SIZE];
  char default_script_dir[U61_CONST_STRING_SIZE];
  char default_data_file[U61_CONST_STRING_SIZE];
  char default_script_file[U61_CONST_STRING_SIZE];

  strcpy(default_data_dir,U61_Platform::get_builtin_theme_path());
  strcpy(default_script_dir,U61_Platform::get_builtin_script_path());
  strcpy(default_data_file,default_data_dir);
  strcat(default_data_file,"default.dat");
  strcpy(default_script_file,default_script_dir);
  strcat(default_script_file,"classic.lua");

  launch_count=read_int(U61_CONFIG_ID_LAUNCH_COUNT,0);
  read_str(data_dir,U61_CONFIG_ID_DATA_DIR,default_data_dir);
  read_str(script_dir,U61_CONFIG_ID_SCRIPT_DIR,default_script_dir);
  read_str(data_file,U61_CONFIG_ID_DATA_FILE,default_data_file);
  read_str(script_file,U61_CONFIG_ID_SCRIPT_FILE,default_script_file);

  fullscreen=read_bool(U61_CONFIG_ID_FULLSCREEN,false);

  read_str(server_id,U61_CONFIG_ID_SERVER_ID,"127.0.0.1");
  network_port=read_int(U61_CONFIG_ID_NETWORK_PORT,8061);
  network_channel=read_int(U61_CONFIG_ID_NETWORK_CHANNEL,1);
        
  initial_speed=read_int(U61_CONFIG_ID_INITIAL_SPEED,5);
  acceleration=read_int(U61_CONFIG_ID_ACCELERATION,5);
  curse_delay=read_int(U61_CONFIG_ID_CURSE_DELAY,20);

  sound_volume=read_int(U61_CONFIG_ID_SOUND_VOLUME,U61_SOUND_VOLUME_MAX/2);
  music_volume=read_int(U61_CONFIG_ID_MUSIC_VOLUME,U61_MUSIC_VOLUME_MAX/2);

  max_fps=read_int(U61_CONFIG_ID_MAX_FPS,18);

  network_anticipation=read_bool(U61_CONFIG_ID_NETWORK_ANTICIPATION,false);

  for (pl=0;pl<U61_CONFIG_NB_PLAYERS;++pl)
    {
      player[pl].first_repeat_move=
	read_player_int(pl,U61_CONFIG_ID_FIRST_REPEAT_MOVE,10);
      player[pl].standard_repeat_move=
	read_player_int(pl,U61_CONFIG_ID_STANDARD_REPEAT_MOVE,5);
      player[pl].first_repeat_rotate=
	read_player_int(pl,U61_CONFIG_ID_FIRST_REPEAT_ROTATE,20);
      player[pl].standard_repeat_rotate=
	read_player_int(pl,U61_CONFIG_ID_STANDARD_REPEAT_ROTATE,10);
      player[pl].first_repeat_other=
	read_player_int(pl,U61_CONFIG_ID_FIRST_REPEAT_OTHER,40);
      player[pl].standard_repeat_other=
	read_player_int(pl,U61_CONFIG_ID_STANDARD_REPEAT_OTHER,20);
      player[pl].anticipation=
	read_player_bool(pl,U61_CONFIG_ID_ANTICIPATION,false);
    }

  player[0].activable=read_player_bool(0,U61_CONFIG_ID_ACTIVABLE,true);
  read_player_str(player[0].name,0,U61_CONFIG_ID_NAME,"ufoot");
  player[0].key_start_stop=read_player_int(0,U61_CONFIG_ID_KEY_START_STOP, CL_KEY_F5);
  player[0].key_move_left=read_player_int(0,U61_CONFIG_ID_KEY_MOVE_LEFT, CL_KEY_LEFT);
  player[0].key_move_right=read_player_int(0,U61_CONFIG_ID_KEY_MOVE_RIGHT, CL_KEY_RIGHT);
  player[0].key_rotate_left=read_player_int(0,U61_CONFIG_ID_KEY_ROTATE_LEFT, CL_KEY_DOWN);
  player[0].key_rotate_right=read_player_int(0,U61_CONFIG_ID_KEY_ROTATE_RIGHT, CL_KEY_UP);
  player[0].key_move_down=read_player_int(0,U61_CONFIG_ID_KEY_MOVE_DOWN, CL_NUM_KEYS-32+'d');
  player[0].key_drop=read_player_int(0,U61_CONFIG_ID_KEY_DROP, CL_NUM_KEYS-32+' ');
  player[0].key_next_victim=read_player_int(0,U61_CONFIG_ID_KEY_NEXT_VICTIM, CL_NUM_KEYS-32+'n');
  player[0].key_prev_victim=read_player_int(0,U61_CONFIG_ID_KEY_PREV_VICTIM, CL_NUM_KEYS-32+'p');
  player[0].key_use_antidote=read_player_int(0,U61_CONFIG_ID_KEY_USE_ANTIDOTE, CL_NUM_KEYS-32+'a');

  player[1].activable=read_player_bool(1,U61_CONFIG_ID_ACTIVABLE,true);
  read_player_str(player[1].name,1,U61_CONFIG_ID_NAME,"etoile");
  player[1].key_start_stop=read_player_int(1,U61_CONFIG_ID_KEY_START_STOP, CL_KEY_F6);
  player[1].key_move_left=read_player_int(1,U61_CONFIG_ID_KEY_MOVE_LEFT, CL_KEY_LEFT);
  player[1].key_move_right=read_player_int(1,U61_CONFIG_ID_KEY_MOVE_RIGHT, CL_KEY_RIGHT);
  player[1].key_rotate_left=read_player_int(1,U61_CONFIG_ID_KEY_ROTATE_LEFT, CL_KEY_DOWN);
  player[1].key_rotate_right=read_player_int(1,U61_CONFIG_ID_KEY_ROTATE_RIGHT, CL_KEY_UP);
  player[1].key_move_down=read_player_int(1,U61_CONFIG_ID_KEY_MOVE_DOWN, CL_NUM_KEYS-32+'d');
  player[1].key_drop=read_player_int(1,U61_CONFIG_ID_KEY_DROP, CL_NUM_KEYS-32+' ');
  player[1].key_next_victim=read_player_int(1,U61_CONFIG_ID_KEY_NEXT_VICTIM, CL_NUM_KEYS-32+'n');
  player[1].key_prev_victim=read_player_int(1,U61_CONFIG_ID_KEY_PREV_VICTIM, CL_NUM_KEYS-32+'p');
  player[1].key_use_antidote=read_player_int(1,U61_CONFIG_ID_KEY_USE_ANTIDOTE, CL_NUM_KEYS-32+'a');

  player[2].activable=read_player_bool(2,U61_CONFIG_ID_ACTIVABLE,true);
  read_player_str(player[2].name,2,U61_CONFIG_ID_NAME,"popol");
  player[2].key_start_stop=read_player_int(2,U61_CONFIG_ID_KEY_START_STOP, CL_KEY_F7);
  player[2].key_move_left=read_player_int(2,U61_CONFIG_ID_KEY_MOVE_LEFT, CL_KEY_LEFT);
  player[2].key_move_right=read_player_int(2,U61_CONFIG_ID_KEY_MOVE_RIGHT, CL_KEY_RIGHT);
  player[2].key_rotate_left=read_player_int(2,U61_CONFIG_ID_KEY_ROTATE_LEFT, CL_KEY_DOWN);
  player[2].key_rotate_right=read_player_int(2,U61_CONFIG_ID_KEY_ROTATE_RIGHT, CL_KEY_UP);
  player[2].key_move_down=read_player_int(2,U61_CONFIG_ID_KEY_MOVE_DOWN, CL_NUM_KEYS-32+'d');
  player[2].key_drop=read_player_int(2,U61_CONFIG_ID_KEY_DROP, CL_NUM_KEYS-32+' ');
  player[2].key_next_victim=read_player_int(2,U61_CONFIG_ID_KEY_NEXT_VICTIM, CL_NUM_KEYS-32+'n');
  player[2].key_prev_victim=read_player_int(2,U61_CONFIG_ID_KEY_PREV_VICTIM, CL_NUM_KEYS-32+'p');
  player[2].key_use_antidote=read_player_int(2,U61_CONFIG_ID_KEY_USE_ANTIDOTE, CL_NUM_KEYS-32+'a');

  player[3].activable=read_player_bool(3,U61_CONFIG_ID_ACTIVABLE,true);
  read_player_str(player[3].name,3,U61_CONFIG_ID_NAME,"arthur");
  player[3].key_start_stop=read_player_int(3,U61_CONFIG_ID_KEY_START_STOP, CL_KEY_F8);
  player[3].key_move_left=read_player_int(3,U61_CONFIG_ID_KEY_MOVE_LEFT, CL_KEY_LEFT);
  player[3].key_move_right=read_player_int(3,U61_CONFIG_ID_KEY_MOVE_RIGHT, CL_KEY_RIGHT);
  player[3].key_rotate_left=read_player_int(3,U61_CONFIG_ID_KEY_ROTATE_LEFT, CL_KEY_DOWN);
  player[3].key_rotate_right=read_player_int(3,U61_CONFIG_ID_KEY_ROTATE_RIGHT, CL_KEY_UP);
  player[3].key_move_down=read_player_int(3,U61_CONFIG_ID_KEY_MOVE_DOWN, CL_NUM_KEYS-32+'d');
  player[3].key_drop=read_player_int(3,U61_CONFIG_ID_KEY_DROP, CL_NUM_KEYS-32+' ');
  player[3].key_next_victim=read_player_int(3,U61_CONFIG_ID_KEY_NEXT_VICTIM, CL_NUM_KEYS-32+'n');
  player[3].key_prev_victim=read_player_int(3,U61_CONFIG_ID_KEY_PREV_VICTIM, CL_NUM_KEYS-32+'p');
  player[3].key_use_antidote=read_player_int(3,U61_CONFIG_ID_KEY_USE_ANTIDOTE, CL_NUM_KEYS-32+'a');
}

/*--------------------------------------------------------------------------*/
/* 
 * saves the values to the config file
 */ 
bool U61_Config::write()
{
  bool ok=false;
  char *filename;
  FILE *file;
  int pl;
  
  U61_Global::data.chdir_to_initial_dir();

  U61_LOG_DEBUG("write config");
  if (U61_Platform::create_user_dir())
    {
      filename=U61_Platform::get_config_file();
      U61_LOG_MESSAGE("Writing config to \""<<filename<<"\".");
      file=fopen(filename,"w");

      if (file!=NULL)
	{
	  ok=true;

	  ok=ok && write_int(file,U61_CONFIG_ID_DUMMY, U61_CONFIG_DUMMY_VALUE);
	  ok=ok && write_int(file,U61_CONFIG_ID_LAUNCH_COUNT, launch_count+1);

	  ok=ok && write_str(file,U61_CONFIG_ID_DATA_DIR, data_dir);
	  ok=ok && write_str(file,U61_CONFIG_ID_SCRIPT_DIR, script_dir);
	  ok=ok && write_str(file,U61_CONFIG_ID_DATA_FILE, data_file);
	  ok=ok && write_str(file,U61_CONFIG_ID_SCRIPT_FILE, script_file);

	  ok=ok && write_int(file,U61_CONFIG_ID_FULLSCREEN, fullscreen);

	  ok=ok && write_str(file,U61_CONFIG_ID_SERVER_ID, server_id);
	  ok=ok && write_int(file,U61_CONFIG_ID_NETWORK_PORT, network_port);
	  ok=ok && write_int(file,U61_CONFIG_ID_NETWORK_CHANNEL, network_channel);

	  ok=ok && write_int(file,U61_CONFIG_ID_INITIAL_SPEED, initial_speed);
	  ok=ok && write_int(file,U61_CONFIG_ID_ACCELERATION, acceleration);
	  ok=ok && write_int(file,U61_CONFIG_ID_CURSE_DELAY, curse_delay);

	  ok=ok && write_int(file,U61_CONFIG_ID_SOUND_VOLUME, sound_volume);
	  ok=ok && write_int(file,U61_CONFIG_ID_MUSIC_VOLUME, music_volume);

	  ok=ok && write_int(file,U61_CONFIG_ID_MAX_FPS, max_fps);

	  ok=ok && write_int(file,U61_CONFIG_ID_NETWORK_ANTICIPATION, network_anticipation);

	  for (pl=0;pl<U61_CONFIG_NB_PLAYERS;++pl)
	    {
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_FIRST_REPEAT_MOVE,
					player[pl].first_repeat_move);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_STANDARD_REPEAT_MOVE,
					player[pl].standard_repeat_move);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_FIRST_REPEAT_ROTATE,
					player[pl].first_repeat_rotate);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_STANDARD_REPEAT_ROTATE,
					player[pl].standard_repeat_rotate);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_FIRST_REPEAT_OTHER,
					player[pl].first_repeat_other);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_STANDARD_REPEAT_OTHER,
					player[pl].standard_repeat_other);
 
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_ANTICIPATION,
					player[pl].anticipation);

	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_ACTIVABLE,
					player[pl].activable);
	      ok=ok && write_player_str(pl,file,U61_CONFIG_ID_NAME,
					player[pl].name);

	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_KEY_START_STOP,
					player[pl].key_start_stop);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_KEY_MOVE_LEFT,
					player[pl].key_move_left);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_KEY_MOVE_RIGHT,
					player[pl].key_move_right);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_KEY_ROTATE_LEFT,
					player[pl].key_rotate_left);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_KEY_ROTATE_RIGHT,
					player[pl].key_rotate_right);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_KEY_MOVE_DOWN,
					player[pl].key_move_down);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_KEY_DROP,
					player[pl].key_drop);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_KEY_NEXT_VICTIM,
					player[pl].key_next_victim);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_KEY_PREV_VICTIM,
					player[pl].key_prev_victim);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_KEY_USE_ANTIDOTE,
					player[pl].key_use_antidote);
	    } 

	  fclose(file);
	}
    }
  
  U61_Global::data.chdir_to_data_dir();

  return ok;
}

/*--------------------------------------------------------------------------*/
/*
 * loads the user scripts that will be used by the script module
 */
bool U61_Config::load_script()
{
  bool ok=false;

  ok=U61_Script::do_file(script_file);
    
  return ok;
}

/*--------------------------------------------------------------------------*/
/*
 * loads a string entry from the config file
 */
void U61_Config::read_str(char *buf, char *name,char *def)
{
  strcpy(buf,U61_Script::get_global_str(name,def));
}

/*--------------------------------------------------------------------------*/
/*
 * loads a number entry from the config file
 */
int U61_Config::read_int(char *name,int def)
{
  return U61_Script::get_global_int(name,def);
}

/*--------------------------------------------------------------------------*/
/*
 * loads a boolean entry from the config file
 */
bool U61_Config::read_bool(char *name,bool def)
{
  return U61_Script::get_global_bool(name,def);
}

/*--------------------------------------------------------------------------*/
/*
 * saves a string entry into the config file
 */
bool U61_Config::write_str(FILE *file,char *name,char *value)
{
  char buffer[U61_CONST_STRING_SIZE];
  char buffer_value[U61_CONST_STRING_SIZE];
  int pos;
  bool ok;

  /*
   * Now we have to trap escape caracters such as \ if we
   * want the system to be bullet proof. It is required for
   * windows (\ in every path), but it can also be usefull on
   * UNIX, your never know...
   */

  pos=0;
  while (*value!=0)
    {
      switch (*value)
	{
	case '\\':
	  buffer_value[pos]='\\';
	  pos++;
	  /*
	   * no break; instruction, since we want the caracter to
	   * be added after the \
	   */
	default:
	  buffer_value[pos]=*value;
	  pos++;
	}
      value++;
    }
  buffer_value[pos]=0;

  strcpy(buffer,name);
  strcat(buffer," = \"");
  strcat(buffer,buffer_value);
  strcat(buffer,"\"\n");

  ok=(fwrite(buffer,strlen(buffer),1,file)>0);

  return ok;
} 

/*--------------------------------------------------------------------------*/
/*
 * saves a numeric entry into the config file
 */
bool U61_Config::write_int(FILE *file,char *name,int value)
{
  char buffer[U61_CONST_STRING_SIZE];
  char buffer2[20];
  bool ok;

  sprintf(buffer2,"%d",value);

  strcpy(buffer,name);
  strcat(buffer," = ");
  strcat(buffer,buffer2);
  strcat(buffer,"\n");

  ok=(fwrite(buffer,strlen(buffer),1,file)>0);

  return ok;
} 

/*--------------------------------------------------------------------------*/
/*
 * loads a string entry from the config file, for a specific player
 */
void U61_Config::read_player_str(char *buf,int pl,char *name,char *def)
{
  char buffer[U61_CONST_STRING_SIZE];

  sprintf(buffer,U61_CONFIG_ID_PLAYER,pl);
  strcat(buffer,name);

  read_str(buf,buffer,def);
}

/*--------------------------------------------------------------------------*/
/*
 * loads a number entry from the config file, for a specific player
 */
int U61_Config::read_player_int(int pl,char *name,int def)
{
  char buffer[U61_CONST_STRING_SIZE];

  sprintf(buffer,U61_CONFIG_ID_PLAYER,pl);
  strcat(buffer,name);

  return read_int(buffer,def);
}

/*--------------------------------------------------------------------------*/
/*
 * loads a boolean entry from the config file, for a specific player
 */
bool U61_Config::read_player_bool(int pl,char *name,bool def)
{
  char buffer[U61_CONST_STRING_SIZE];

  sprintf(buffer,U61_CONFIG_ID_PLAYER,pl);
  strcat(buffer,name);

  return read_bool(buffer,def);
}

/*--------------------------------------------------------------------------*/
/*
 * saves a string entry into the config file, for a specific player
 */
bool U61_Config::write_player_str(int pl,FILE *file,char *name,char *value)
{
  char buffer[U61_CONST_STRING_SIZE];

  sprintf(buffer,U61_CONFIG_ID_PLAYER,pl);
  strcat(buffer,name);

  return write_str(file,buffer,value);
} 

/*--------------------------------------------------------------------------*/
/*
 * saves a numeric entry into the config file, for a specific player
 */
bool U61_Config::write_player_int(int pl,FILE *file,char *name,int value)
{
  char buffer[U61_CONST_STRING_SIZE];

  sprintf(buffer,U61_CONFIG_ID_PLAYER,pl);
  strcat(buffer,name);

  return write_int(file,buffer,value);
} 



