/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   platformwindows.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: contains windows-specific code.
 *              for instance, the directory used for data storing are
 *              different under linux or windows
 */


/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include <stdlib.h>
#include <time.h>
#include <process.h>
#include <ClanLib/mikmod.h>
#include <ClanLib/lua.h>

#include "platform.h"
#include "const.h"
#include "debug.h"

/*---------------------------------------------------------------------------
 variants
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * This file is used to dump the scores and keep a trace of what's happening.
 * On Windows it is located in .\score.txt
 */
static FILE *score_file=NULL;

/*--------------------------------------------------------------------------*/
/*
 * The file is used to log the debug messages, it is very important under 
 * Windows where there's no decent logging utility (except the ClanLib debug
 * window but I like a file better...)
 */
static FILE *debug_file=NULL;

/*--------------------------------------------------------------------------*/
/*
 * This file is used to dump errors and various in-game messages.
 * This file does not exists on Linux since syslog and console output are
 * used instead. As usual, Windows provides almost nothing to trace errors,
 * so we have to do everything ourselves...
 */
static FILE *trace_file=NULL;

/*--------------------------------------------------------------------------*/
/*
 * information structure for use by the directory scanning routines
 * I got this code from Allegro by Shaw Hargreaves, and it help
 * me a lot! Allegro is just another game programming library, it
 * should be findable on http://alleg.sourceforge.net/
 */
typedef struct FFIND_INFO {
   struct _finddata_t info;
   long handle;
   int attrib;
   int last;
} FFIND_INFO;
static void *find_info=NULL;

/* emulate the FA_* flags for platforms that don't already have them */
#ifndef FA_RDONLY
   #define FA_RDONLY       1
   #define FA_HIDDEN       2
   #define FA_SYSTEM       4
   #define FA_LABEL        8
   #define FA_DIREC        16
   #define FA_ARCH         32
#endif

/* life would be so easy if compilers would all use the same names! */
#ifndef S_IRUSR
#define S_IRUSR   S_IREAD
#define S_IWUSR   S_IWRITE
#endif

/*--------------------------------------------------------------------------*/
/*
 * The _al_blabla functions all come from Allegro. Again, I think
 * this copy/paste operation saved me some long hours of boring
 * coding 8-)
 */

static void *_al_findfirst(char *name, int attrib);
int _al_findnext(void *dta, char *nameret, int *aret);
void _al_findclose(void *dta);

/* _al_findfirst:
 *  Initiates a directory search.
 */
void *_al_findfirst(char *name, int attrib)
{
   FFIND_INFO *info;
//   int a;

   info = (FFIND_INFO *) malloc(sizeof(FFIND_INFO));

   if (info!=NULL)
   {
   info->attrib = attrib;
   info->last=0;

   if ((info->handle = _findfirst(name, &info->info)) < 0) {
      free(info);
      return NULL;
   }
/*
   a = info->info.attrib & (FA_HIDDEN | FA_SYSTEM | FA_LABEL | FA_DIREC);

   if ((a & attrib) != a) {
      if (_al_findnext(info, nameret, aret) != 0) {
	 _findclose(info->handle);
	 free(info);
	 return NULL;
      }
      else
	 return info;
   }

//   do_uconvert(info->info.name, U_ASCII, nameret, U_CURRENT, -1);
	strcpy(nameret,info->info.name);

   if (aret)
      *aret = info->info.attrib;
*/   }

   return info;
}



/* _al_findnext:
 *  Retrives the next file from a directory search.
 */
int _al_findnext(void *dta, char *nameret, int *aret)
{
   FFIND_INFO *info = (FFIND_INFO *) dta;
   int a;

   if (info->last!=0)
   {
	   return -1;
   }

   a = info->info.attrib & (FA_HIDDEN | FA_SYSTEM | FA_LABEL | FA_DIREC);
    while ((a & info->attrib) != a) {
      if (_findnext(info->handle, &info->info) != 0) {
	 return -1;
      }

      a = info->info.attrib & (FA_HIDDEN | FA_SYSTEM | FA_LABEL | FA_DIREC);

   }

//   do_uconvert(info->info.name, U_ASCII, nameret, U_CURRENT, -1);
	strcpy(nameret,info->info.name);

   if (aret)
      *aret = info->info.attrib;

	if (_findnext(info->handle, &info->info) != 0)
	{
		info->last=1;
	}

   return 0;
}

/* _al_findclose:
 *  Cleans up after a directory search.
 */
void _al_findclose(void *dta)
{
   FFIND_INFO *info = (FFIND_INFO *) dta;

   _findclose(info->handle);
   free(info);
}




/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * returns the path where user data are stored, needs to be writeable
 */
char *U61_Platform::get_user_path()
{
  static char buffer[U61_CONST_STRING_SIZE]; 

  strcpy(buffer,".\\");

  return buffer;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the path where builtin data is stored, does not need to be writeable
 */
char *U61_Platform::get_data_path()
{
  static char buffer[U61_CONST_STRING_SIZE]; 

  strcpy(buffer,".\\");	

  return buffer;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the path for builtin themes 
 */
char *U61_Platform::get_builtin_theme_path()
{
  static char buffer[U61_CONST_STRING_SIZE];

  strcpy(buffer,get_data_path());
  strcat(buffer,"builtin-theme\\");

  return buffer;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the path for builtin scripts
 */
char *U61_Platform::get_builtin_script_path()
{
  static char buffer[U61_CONST_STRING_SIZE];

  strcpy(buffer,get_data_path());
  strcat(buffer,"builtin-script\\");

  return buffer;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the path for user themes 
 */
char *U61_Platform::get_user_theme_path()
{
  static char buffer[U61_CONST_STRING_SIZE];

  strcpy(buffer,get_user_path());
  strcat(buffer,"user-theme\\");

  return buffer;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the path for user scripts
 */
char *U61_Platform::get_user_script_path()
{
  static char buffer[U61_CONST_STRING_SIZE];

  strcpy(buffer,get_user_path());
  strcat(buffer,"user-script\\");

  return buffer;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the name of the config file (/home/ufoot/.u61/config for instance)
 */
char *U61_Platform::get_config_file()
{
  static char buffer[U61_CONST_STRING_SIZE]; 

  strcpy (buffer,get_user_path());
  strcat(buffer,"config.txt");
 
  return buffer;
}

/*--------------------------------------------------------------------------*/
/*
 * transforms "C:\\GAMES\\U61\\builtin-theme\\classic.lua" into "classic"
 */
void U61_Platform::strip_remote_script_path(char *path)
{
  char buffer[U61_CONST_STRING_SIZE];
  char *pos;

  /*
   * We remove the path (directories part)
   */
  pos=strrchr(path,'\\');
  if (pos!=NULL)
    {
      strcpy(buffer,pos+1);
      strcpy(path,buffer);
    }

  /*
   * We remove the extension
   */
  pos=strrchr(path,'.');
  if (pos!=NULL)
    {
      pos[0]=0;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * transforms  "classic" into "C:\\GAMES\\U61\\builtin-theme\\classic.lua"
 */
void U61_Platform::unstrip_remote_script_path(char *path)
{
  char buffer[U61_CONST_STRING_SIZE];

  strncpy(buffer,get_user_script_path(),sizeof(buffer)-1);
  strcat(buffer,path);
  strcat(buffer,".lua");

  strncpy(path,buffer,sizeof(buffer)-1);
}

/*--------------------------------------------------------------------------*/
/*
 * opens a directory for file scanning
 */
bool U61_Platform::open_dir(char *dir, char *ext)
{
  bool ok=false;
  char buffer[U61_CONST_STRING_SIZE];

  sprintf(buffer,"%s*%s",dir,ext);
  find_info=_al_findfirst(buffer,0);
  if (find_info!=NULL)
  {
	ok=true;
  }

  return ok;
}

/*--------------------------------------------------------------------------*/
/*
 * retrieves the next file in the directory
 */
char *U61_Platform::next_file()
{
  static char buffer[U61_CONST_STRING_SIZE];
  char *result=NULL;
  int attrib;

  if (_al_findnext(find_info,buffer,&attrib)==0)
  {
	result=buffer;
  }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * closes the directory
 */
bool U61_Platform::close_dir()
{
  bool ok=false;

  if (find_info!=NULL)
  {
	_al_findclose(find_info);
	find_info=NULL;
  }
  ok=true;

  return ok;
}

/*--------------------------------------------------------------------------*/
/*
 * Creates the directory where user files will be stored
 * returns true if the directory exists or has been succesfully created
 */
bool U61_Platform::create_user_dir()
{
    bool result=false;

    /*
	 * for now the user files are simply stored in ".\" so the
	 * directory dos not need to be created. Later, U61 might use
	 * c:\windows\profile\user\u61 but it's not the case yet...
	 */
	result=true;

	return true;
}


/*--------------------------------------------------------------------------*/
/*
 * Quits U61 and launches "file"
 * Used when the player changes the current theme
 */
void U61_Platform::exec(char *file)
{
  char buffer[U61_CONST_STRING_SIZE];
//  int len;

  U61_LOG_MESSAGE("Restarting...");
  U61_Debug::deinit();

  CL_Lua::close();
  CL_SetupMikMod::deinit();
  CL_SetupCore::deinit_network();    
  CL_SetupCore::deinit_sound();
  CL_SetupCore::deinit_display();
  CL_SetupCore::deinit();

  /*
   * We try to launch the executable described
   * by argv[0]. This succeeds in case the exe is .\U61.exe
   * or something with a full path. execl shouldn't
   * returns if it succeeds
   */
  execl(file,file,NULL);

  /*
   * Now if we get there, it means there's an error.
   * Since we have uninitialized ClanLib, we *must*
   * quit, or there's a core dump arround the corner...
   */
  U61_Debug::init();
  U61_LOG_ERROR("Unable to execute "<<file<<" or "<<buffer);
  U61_Debug::deinit();

  exit(1);
}

/*--------------------------------------------------------------------------*/
/*
 * Opens the score file, on UNIX, it is ~/.u61/score.txt
 */
void U61_Platform::open_score_file()
{
  char buffer[U61_CONST_STRING_SIZE];

  if (score_file==NULL)
    {
      strcpy (buffer,get_user_path());
      strcat(buffer,"score.txt");

      score_file=fopen(buffer,"a");
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Closes the score file, on UNIX, it is ~/.u61/score.txt
 */
void U61_Platform::close_score_file()
{
  if (score_file!=NULL)
    {
      fclose(score_file);
    }

  score_file=NULL;
}

/*--------------------------------------------------------------------------*/
/*
 * Logs an error message with syslog and in the debug log if needed
 */
void U61_Platform::dump_score(char *text)
{
  time_t curtime;
  struct tm *loctime;
  char buffer_date[U61_CONST_STRING_SIZE];

  if (score_file!=NULL)
    {     
      /*
       * Get the current time.
       */
      curtime = time (NULL);
     
      /*
       * Convert it to local time representation.
       */
      loctime = localtime (&curtime);
     
      sprintf(buffer_date,asctime(loctime));
      /*
       * We get rid of the trailing '\n'
       */
      if (strlen(buffer_date)>=24)
	{
	  buffer_date[24]=0;
	}
      fprintf(score_file,"%-24s ; %s\n",buffer_date,text);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Opens all the log files, on UNIX, this concerns only the debug log
 * file, since the rest of the log is handled by syslog
 */
void U61_Platform::open_log_files()
{
  char buffer[U61_CONST_STRING_SIZE];

  if (trace_file==NULL)
    {
      strcpy (buffer,get_user_path());
      strcat(buffer,"trace.txt");

      trace_file=fopen(buffer,"w");
    }

#ifdef DEBUG
  if (debug_file==NULL)
    {
      strcpy (buffer,get_user_path());
      strcat(buffer,"debug.txt");

      debug_file=fopen(buffer,"w");
    }
#endif
}

/*--------------------------------------------------------------------------*/
/*
 * Closes all the log files, on UNIX, this concerns only the debug log
 * file, since the rest of the log is handled by syslog
 */
void U61_Platform::close_log_files()
{
  if (trace_file!=NULL)
    {
      fclose(trace_file);
    }
  trace_file=NULL;


#ifdef DEBUG
  if (debug_file!=NULL)
    {
      fclose(debug_file);
    }
#endif

  debug_file=NULL;
}

/*--------------------------------------------------------------------------*/
/*
 * Logs an error message with syslog and in the debug log if needed
 */
void U61_Platform::log_to_error_file(char *text)
{
  if (trace_file!=NULL)
    {
      fprintf(trace_file,"ERROR! %s\n",text);
    }
#ifdef DEBUG
  if (debug_file!=NULL)
    {
      fprintf(debug_file,"*;error  ;%25s;%-5d;%s\n","unknown",0,text);
    }
#endif
}

/*--------------------------------------------------------------------------*/
/*
 * Logs a warning message with syslog and in the debug log if needed
 */
void U61_Platform::log_to_warning_file(char *text)
{
  if (trace_file!=NULL)
    {
      fprintf(trace_file,"WARNING! %s\n",text);
    }
#ifdef DEBUG
  if (debug_file!=NULL)
    {
      fprintf(debug_file,"*;warning;%25s;%-5d;%s\n","unknown",0,text);
    }
#endif
}

/*--------------------------------------------------------------------------*/
/*
 * Logs a standard message with syslog and in the debug log if needed
 */
void U61_Platform::log_to_message_file(char *text)
{
  if (trace_file!=NULL)
    {
      fprintf(trace_file,"%s\n",text);
    }
#ifdef DEBUG
  if (debug_file!=NULL)
    {
      fprintf(debug_file," ;message;%25s;%-5d;%s\n","unknown",0,text);
    }
#endif
}

/*--------------------------------------------------------------------------*/
/*
 * Logs a debug message with syslog and in the debug log if needed
 */
void U61_Platform::log_to_debug_file(char *text,char *file,int line)
{
#ifdef DEBUG
  if (debug_file!=NULL)
    {
      fprintf(debug_file," ;debug  ;%25s;%-5d;%s\n",file,line,text);
    }
#endif
}

