/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   clientprotocol.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: the client sends informations about him and retrieves
 *              information from the server, including the rules and
 *              the state of all the other players
 */


/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "clientprotocol.h"
#include "global.h"
#include "time.h"
#include "platform.h"
#include "debug.h"

/*---------------------------------------------------------------------------
 constants
 ---------------------------------------------------------------------------*/

#define U61_CLIENTPROTOCOL_SEARCHING                10
#define U61_CLIENTPROTOCOL_CONNECTED_TO_SERVER      11
#define U61_CLIENTPROTOCOL_JOIN_DELAY_ELAPSED       12
#define U61_CLIENTPROTOCOL_CHANNEL_FOUND            13
#define U61_CLIENTPROTOCOL_SCRIPT_NAME_RECEIVED     30
#define U61_CLIENTPROTOCOL_SCRIPT_RECEIVED          31
#define U61_CLIENTPROTOCOL_ID0_RECEIVED             40
#define U61_CLIENTPROTOCOL_ID1_RECEIVED             41
#define U61_CLIENTPROTOCOL_INITIAL_SPEED_RECEIVED   42
#define U61_CLIENTPROTOCOL_ACCELERATION_RECEIVED    43
#define U61_CLIENTPROTOCOL_CURSE_DELAY_RECEIVED     44
#define U61_CLIENTPROTOCOL_TIME_RECEIVED            45
#define U61_CLIENTPROTOCOL_PLAYING                  60   
#define U61_CLIENTPROTOCOL_CLOSED                   61

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * creates a client protocol
 */
U61_ClientProtocol::U61_ClientProtocol(U61_ClientDispatcher *d, 
				       int p,int ch,
                                       char *id)
  : U61_Protocol(p,ch)
{
  dispatcher=d;
  server_id=id;
  connection=NULL;
}

/*--------------------------------------------------------------------------*/
/*
 * deletes the protocol object
 */
U61_ClientProtocol::~U61_ClientProtocol()
{
  if (connection!=NULL)
    {
      delete connection;
      connection=NULL;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Checks if the server is still there...
 */
void U61_ClientProtocol::check_connection()
{
  int i;
  U61_Player *p;

  if (stage!=U61_CLIENTPROTOCOL_CLOSED 
      && game!=NULL 
      && game->receive_game_closed())
    {
      stage=U61_CLIENTPROTOCOL_CLOSED;
      U61_LOG_DEBUG("stage=CLOSED");
      connection->close();

      /*
       * We kill all the remote players and let the game keep going locally
       */
      for (i=0;i<U61_DISPATCHER_MAX_PLAYER_ID;++i)
	{
	  p=U61_Global::game.find_player_by_id(i);
          if (p!=NULL && p->is_available() && !p->is_local())
	    {
	      p->kill();
	    }
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * sends the next packet. sometimes there's not really a packet send but
 * the point is to identify the sequence of messages to send to the server
 */
int U61_ClientProtocol::send_next_packet()
{
  int result=U61_PROTOCOL_LATER;

  switch(stage)
    {
    case U61_PROTOCOL_BEGIN:
      if ((result=start_searching())==U61_PROTOCOL_DONE)
	{
	  stage=U61_CLIENTPROTOCOL_SEARCHING;
	  U61_LOG_DEBUG("stage=SEARCHING");
	}
      break;
    case U61_CLIENTPROTOCOL_CONNECTED_TO_SERVER:
      if ((result=join_delay())==U61_PROTOCOL_DONE)
	{
	  stage=U61_CLIENTPROTOCOL_JOIN_DELAY_ELAPSED;
	  U61_LOG_DEBUG("stage=JOIN_DELAY_ELAPSED");
	}
      break;
    case U61_CLIENTPROTOCOL_CHANNEL_FOUND:
      if ((result=send_id(connection))==U61_PROTOCOL_DONE)
	{
	  stage=U61_PROTOCOL_ID_SENT;
	  U61_LOG_DEBUG("stage=ID_SENT");
	}
      break;
    case U61_PROTOCOL_ID_RECEIVED:
      if ((result=send_version(connection))==U61_PROTOCOL_DONE)
	{
	  stage=U61_PROTOCOL_VERSION_SENT;
	  U61_LOG_DEBUG("stage=VERSION_SENT");
	}
      break;
    case U61_CLIENTPROTOCOL_TIME_RECEIVED:
      if ((result=send_ready(connection))==U61_PROTOCOL_DONE)
	{
	  stage=U61_PROTOCOL_READY_SENT;
	  U61_LOG_DEBUG("stage=READY_SENT");
	}
      break;
    case U61_PROTOCOL_READY_RECEIVED:
      start_game();
      dispatcher->open(connection);
      
      stage=U61_CLIENTPROTOCOL_PLAYING;
      U61_LOG_DEBUG("stage=PLAYING");
      
      break;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * receives the next packet. sometimes there's not really a packet send but
 * the point is to identify the sequence of messages to receive from the
 * server
 */
int U61_ClientProtocol::recv_next_packet()
{
  int result=U61_PROTOCOL_LATER;

  switch(stage)
    {
    case U61_CLIENTPROTOCOL_SEARCHING:
      if ((result=connect_to_server())==U61_PROTOCOL_DONE)
	{
	  stage=U61_CLIENTPROTOCOL_CONNECTED_TO_SERVER;
	  U61_LOG_DEBUG("stage=CONNECTED_TO_SERVER");
	}
      break;
    case U61_CLIENTPROTOCOL_JOIN_DELAY_ELAPSED:
      if ((result=find_channel())==U61_PROTOCOL_DONE)
	{
	  stage=U61_CLIENTPROTOCOL_CHANNEL_FOUND;
	  U61_LOG_DEBUG("stage=CHANNEL_FOUND");
	}
      break;
    case U61_PROTOCOL_ID_SENT:
      if ((result=recv_id(connection))==U61_PROTOCOL_DONE)
	{  
	  stage=U61_PROTOCOL_ID_RECEIVED;
	  U61_LOG_DEBUG("stage=ID_RECEIVED");
	}
      break;
    case U61_PROTOCOL_VERSION_SENT:
      if ((result=recv_version(connection))==U61_PROTOCOL_DONE)
	{
	  stage=U61_PROTOCOL_VERSION_RECEIVED;
	  U61_LOG_DEBUG("stage=VERSION_RECEIVED");
	}
      break;
    case U61_PROTOCOL_VERSION_RECEIVED:
      if ((result=recv_script_name())==U61_PROTOCOL_DONE)
	{
	  stage=U61_CLIENTPROTOCOL_SCRIPT_NAME_RECEIVED;
	  U61_LOG_DEBUG("stage=SCRIPT_NAME_RECEIVED");
	}
      break;
    case U61_CLIENTPROTOCOL_SCRIPT_NAME_RECEIVED:
      if ((result=recv_script())==U61_PROTOCOL_DONE)
	{
	  stage=U61_CLIENTPROTOCOL_SCRIPT_RECEIVED;
	  U61_LOG_DEBUG("stage=SCRIPT_RECEIVED");
	}
      break;
    case U61_CLIENTPROTOCOL_SCRIPT_RECEIVED:
      if ((result=recv_id0())==U61_PROTOCOL_DONE)
	{
	  stage=U61_CLIENTPROTOCOL_ID0_RECEIVED;
	  U61_LOG_DEBUG("stage=ID0_RECEIVED");
	}
      break;
    case U61_CLIENTPROTOCOL_ID0_RECEIVED:
      if ((result=recv_id1())==U61_PROTOCOL_DONE)
	{
	  stage=U61_CLIENTPROTOCOL_ID1_RECEIVED;
	  U61_LOG_DEBUG("stage=ID1_RECEIVED");
	}
      break;
    case U61_CLIENTPROTOCOL_ID1_RECEIVED:
      if ((result=recv_initial_speed())==U61_PROTOCOL_DONE)
	{
	  stage=U61_CLIENTPROTOCOL_INITIAL_SPEED_RECEIVED;
	  U61_LOG_DEBUG("stage=INITIAL_SPEED_RECEIVED");
	}
      break;
    case U61_CLIENTPROTOCOL_INITIAL_SPEED_RECEIVED:
      if ((result=recv_acceleration())==U61_PROTOCOL_DONE)
	{
	  stage=U61_CLIENTPROTOCOL_ACCELERATION_RECEIVED;
	  U61_LOG_DEBUG("stage=ACCELERATION_RECEIVED");
	}
      break;
    case U61_CLIENTPROTOCOL_ACCELERATION_RECEIVED:
      if ((result=recv_curse_delay())==U61_PROTOCOL_DONE)
	{
	  stage=U61_CLIENTPROTOCOL_CURSE_DELAY_RECEIVED;
	  U61_LOG_DEBUG("stage=CURSE_DELAY_RECEIVED");
	}
      break;
    case U61_CLIENTPROTOCOL_CURSE_DELAY_RECEIVED:
      if ((result=recv_time())==U61_PROTOCOL_DONE)
	{
	  stage=U61_CLIENTPROTOCOL_TIME_RECEIVED;
	  U61_LOG_DEBUG("stage=TIME_RECEIVED");
	}
      break;
    case U61_PROTOCOL_READY_SENT:
      if ((result=recv_ready(connection))==U61_PROTOCOL_DONE)
	{
	  stage=U61_PROTOCOL_READY_RECEIVED;
	  U61_LOG_DEBUG("stage=READY_RECEIVED");
	}
      break;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * enter the server search state
 */
int U61_ClientProtocol::start_searching()
{
  int result=U61_PROTOCOL_LATER;

  try 
    {
      CL_Network::find_game_at(U61_PROTOCOL_GAME_ID,server_id,port);
      result=U61_PROTOCOL_DONE;
    }
  catch (CL_Error e)
    {
      U61_LOG_WARNING("Find error : "<<e.message.c_str()<<"");
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * tries to find out if a game has been found on the server
 * and obtain a connection from this server
 */
int U61_ClientProtocol::connect_to_server()
{
  int result=U61_PROTOCOL_LATER;

  try 
    {
      if (CL_Network::peek_game_found())
	{
	  game=CL_Network::receive_game_found(U61_PROTOCOL_TIMEOUT);
	  if (game!=NULL)
	    {
              /*
               * OK, a game has been found
               */
              join_delay_start=U61_Time::for_effect();
	      result=U61_PROTOCOL_DONE;
	    }
	}
    }
  catch (CL_Error e)
    {
      U61_LOG_WARNING("Connect error : "<<e.message.c_str()<<"");
    }
  U61_LOG_DEBUG("find server");

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * This function just waits one second before trying to send stuff
 * to the server, this is for ClanLib to find the time to give
 * network access on the right channel
 */
int U61_ClientProtocol::join_delay()
{
  int result=U61_PROTOCOL_LATER;

  if (U61_Time::for_effect()>join_delay_start+U61_CLIENTPROTOCOL_JOIN_DELAY)
    {
      channel_to_test=0;
      result=U61_PROTOCOL_DONE;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * So now we search a free channel in order to open a connection object
 */
int U61_ClientProtocol::find_channel()
{
  int result=U61_PROTOCOL_LATER;
  const CL_NetComputer *server;

  if (game->is_writable(channel_to_test) 
      && game->is_readable(channel_to_test))
    {
      server=game->get_server();
      connection=new U61_Connection(game,(CL_NetComputer *) server,
                                    channel_to_test,false);
      result=U61_PROTOCOL_DONE;
    }
  else
    {
      channel_to_test++;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the script name, this value will be used to create
 * a filename for the server remote script on the client
 */
int U61_ClientProtocol::recv_script_name()
{
  int result=U61_PROTOCOL_LATER;

  if (connection->peek())
    {
      if (connection->recv(U61_Global::remote_script.name_buffer,
			   sizeof(U61_Global::remote_script.name_buffer)-1))
	{
	  U61_Platform::strip_remote_script_path
	    (U61_Global::remote_script.name_buffer);
	  result=U61_PROTOCOL_DONE;
          U61_LOG_DEBUG("script_name=\""<<U61_Global::remote_script.name_buffer<<"\"");
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the script itself, this can be a long packet,
 * I have no idea wether ClanLib handles this correctly yet (I mean that
 * I fear this transmission could "block" the game for a while)
 */
int U61_ClientProtocol::recv_script()
{
  int result=U61_PROTOCOL_LATER;

  if (connection->peek())
    {
      if (connection->recv(U61_Global::remote_script.code_buffer,
			   sizeof(U61_Global::remote_script.code_buffer)-1))
	{
	  result=U61_PROTOCOL_DONE;
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the ID that will be given to the first local player
 */
int U61_ClientProtocol::recv_id0()
{
  int result=U61_PROTOCOL_LATER;

  if (connection->peek())
    {
      if (connection->recv(&id0))
	{
	  result=U61_PROTOCOL_DONE;
	  U61_LOG_DEBUG("id0="<<id0);
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the ID that will be given to the second local player
 */
int U61_ClientProtocol::recv_id1()
{
  int result=U61_PROTOCOL_LATER;

  if (connection->peek())
    {
      if (connection->recv(&id1))
	{
	  result=U61_PROTOCOL_DONE;
	  U61_LOG_DEBUG("id1="<<id1);
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the initial speed that will be used to initialize the local maps
 */
int U61_ClientProtocol::recv_initial_speed()
{
  int result=U61_PROTOCOL_LATER;

  if (connection->peek())
    {
      if (connection->recv(&initial_speed))
	{
	  result=U61_PROTOCOL_DONE;
	  U61_LOG_DEBUG("initial_speed="<<initial_speed);
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the acceleration that will be used to initialize the local maps
 */
int U61_ClientProtocol::recv_acceleration()
{
  int result=U61_PROTOCOL_LATER;

  if (connection->peek())
    {
      if (connection->recv(&acceleration))
	{
	  result=U61_PROTOCOL_DONE;
	  U61_LOG_DEBUG("acceleration="<<acceleration);
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the curse delay that will be used to initialize the local maps
 */
int U61_ClientProtocol::recv_curse_delay()
{
  int result=U61_PROTOCOL_LATER;

  if (connection->peek())
    {
      if (connection->recv(&curse_delay))
	{
	  result=U61_PROTOCOL_DONE;
	  U61_LOG_DEBUG("curse_delay="<<curse_delay);
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the time that will be used to initialize the local maps
 */
int U61_ClientProtocol::recv_time()
{
  int result=U61_PROTOCOL_LATER;

  if (connection->peek())
    {
      if (connection->recv(&time))
	{
	  result=U61_PROTOCOL_DONE;
	  U61_LOG_DEBUG("time="<<time);
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Starts the game (at last!)
 */
void U61_ClientProtocol::start_game()
{
  U61_Global::game.do_network_join(U61_Global::remote_script.code_buffer,
				   id0,id1,
				   initial_speed,acceleration,curse_delay,
				   time);
}








