/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   dispatcher.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: the dispatcher is responsible for sending & receiving
 *              messages from the network, and send them to the right person
 *              it's the main "complex" interface with the low level
 *              network API
 */


/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "dispatcher.h"
#include "time.h"
#include "debug.h"

/*---------------------------------------------------------------------------
 variants
 ---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * creates a dispatcher
 */
U61_Dispatcher::U61_Dispatcher()
{
  reset();
}

/*--------------------------------------------------------------------------*/
/*
 * deletes a dispatcher
 */
U61_Dispatcher::~U61_Dispatcher()
{
}

/*--------------------------------------------------------------------------*/
/*
 * Empties all the queues
 */
void U61_Dispatcher::reset()
{
  int i;

  outgoing_queue.clear();
  for (i=0;i<MAX_PLAYER_ID;++i)
    {
      incoming_queue[i].clear();
      busy_flag[i]=false;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if there's been a message for this player when
 * the dispatcher was processed last time
 */
bool U61_Dispatcher::is_player_busy(int player_id)
{
  return busy_flag[player_id];
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if there are no more incoming events for a given player
 */
bool U61_Dispatcher::empty_in(int player_id)
{
  return incoming_queue[player_id].empty();
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if there are no more outgoing messages
 */
bool U61_Dispatcher::empty_out()
{
  return outgoing_queue.empty();
}

/*--------------------------------------------------------------------------*/
/*
 * Puts an event in the outgoing queue
 */
void U61_Dispatcher::put_out(U61_Event event)
{
  U61_LOG_DEBUG("Event "<<event<<" in dispatcher");
  outgoing_queue.push_back(event);
}

/*--------------------------------------------------------------------------*/
/*
 * Gets an event for a given player
 */
U61_Event U61_Dispatcher::get_in(int player_id)
{
  U61_Event event;

  event=incoming_queue[player_id].front();
  incoming_queue[player_id].pop_front();

  return event;
}

/*--------------------------------------------------------------------------*/
/*
 * Gets an event from the outgoing queue
 */
U61_Event U61_Dispatcher::get_out()
{
  U61_Event event;

  event=outgoing_queue.front();
  outgoing_queue.pop_front();

  return event;
}

/*--------------------------------------------------------------------------*/
/*
 * Puts an event in the incoming queue
 */
void U61_Dispatcher::put_in(U61_Event event, int player_id)
{
  incoming_queue[player_id].push_back(event);
}

/*--------------------------------------------------------------------------*/
/*
 * Distributes all the events
 */
bool U61_Dispatcher::process()
{
  int i;
  U61_Event event;
  bool ok=true;
  int count=0;

  for (i=0;i<MAX_PLAYER_ID;++i)
    {
      busy_flag[i]=false;
    }

  while (peek() && count<MAX_AT_ONCE)
    {
      if (!recv(&event))
	{
	  ok=false;
	}
      else
	{
          if (event.code==U61_Event::START_STOP)
	    {
               busy_flag[event.target]=true;
	    }
	  if (event.time>U61_Time::for_event())
	    {
	      U61_LOG_DEBUG("Speeding up");
	      U61_Time::speed_up();
	    }
          put_in(event,event.target);
	}
      count++;
    }
 
  if (count>MAX_AT_ONCE/2)
    {
      U61_LOG_WARNING("Dispatcher is filled up with events");
    }

  while (!empty_out())
    {
      event=get_out();

      /*
       * REQUEST_CURSE events are copied into the local queue
       * since the target can be on a the same machine
       * than the sender
       */
      if (event.code==U61_Event::REQUEST_CURSE)
      {
        put_in(event,event.target);
      }

      if (!send(event))
	{
	  ok=false;
	}
    }

  return ok;
}


/*--------------------------------------------------------------------------*/
/*
 * sends an event to no one 
 */
bool U61_Dispatcher::send(U61_Event evt)
{
  bool ok=true;

  return ok;
}

/*--------------------------------------------------------------------------*/
/*
 * returns always false by default 
 */
bool U61_Dispatcher::recv(U61_Event *evt)
{
  bool ok=false;

  return ok;
} 

/*--------------------------------------------------------------------------*/
/*
 * returns always false by default
 */
bool  U61_Dispatcher::peek()
{
  bool ok=false;

  return ok;
} 

