/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   localsource.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: a local source is the event source used for local players
 *              it polls informations
 *              - from the keyboard
 *              - from the map (for new block management for instance)
 *              it also generates dummy events regularly so that there's
 *              not too wide a gap between the "last" and "sure" members
 *              of the map class
 */

/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "localsource.h"
#include "time.h"

/*---------------------------------------------------------------------------
 globals
 ---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * creation of a local event source
 */ 
U61_LocalSource::U61_LocalSource(U61_Input *i,U61_Map *m,
				 U61_Dispatcher *d,
                                 int player_id, char *player_name)
{
  dispatcher=d;
  player=player_id;
  strcpy(name,player_name);
  input=i;
  input->reset();
  map=m;
  dummy_counter=0;
}

/*--------------------------------------------------------------------------*/
/* 
 * destruction of a local event source
 */ 
U61_LocalSource::~U61_LocalSource()
{

}

/*--------------------------------------------------------------------------*/
/*
 * polls all the events, and sorts them by time
 */
void U61_LocalSource::poll()
{
  poll_input();
  poll_map();
  poll_dispatcher();
  poll_dummy();
}

/*--------------------------------------------------------------------------*/
/*
 * send a start event, even if the key has not been pressed
 */
void U61_LocalSource::send_start_stop(int time)
{
  U61_Event evt;

  evt.time=time;
  evt.author=player;
  evt.target=player;
  evt.code=U61_Event::START_STOP;
  put(evt); 
  send_name(time);
}

/*--------------------------------------------------------------------------*/
/*
 * polls the action events, linked to a user action (key press etc...)
 */
void U61_LocalSource::poll_input()
{
  U61_Event evt;

  while (!input->empty())
    {
      evt=input->get();
      evt.author=player;
      evt.target=player;
      put(evt);
      if (evt.code==U61_Event::START_STOP)
	{
	  send_name(evt.time);
	}
    }    
}

/*--------------------------------------------------------------------------*/
/*
 * polls the events that are automatically generated as time goes on
 */
void U61_LocalSource::poll_map()
{
  U61_Event evt;
 
  map->poll_auto_events();    
  while (map->exists_auto_event())
    {
      evt=map->get_auto_event();
      evt.time=U61_Time::for_event();
      evt.author=player;
      evt.target=player;
      put(evt);
      /*
       * We re-send the name each time the player looses since
       * each time the player looses there's a reset of the history
       * of the dispatcher. This is to avoid the sending of a
       * zillion events if a player has been playing for hours 8-)
       */
      if (evt.code==U61_Event::LOOSE)
	{
	  send_name(evt.time);
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * polls the dispatcher events, linked to a foreign player (curses etc...)
 */
void U61_LocalSource::poll_dispatcher()
{
  U61_Event evt;

  while (!dispatcher->empty_in(player))
    {
      evt=dispatcher->get_in(player);
      //cout<<"Local player receiving event "<<evt<<" from dispatcher\n";
      evt.time=U61_Time::for_event();
      put(evt);
    }    
}

/*--------------------------------------------------------------------------*/
/*
 * fires dummy event which do nothing so that during local play, auto
 * events are correctly handled (ie one has to fire events now and then
 * so that the "sure" tab is sometimes updated :)
 */
void U61_LocalSource::poll_dummy()
{
  U61_Event evt;

  dummy_counter++;
  if (dummy_counter>=DUMMY_DELAY)
    {
      dummy_counter=0; 
      evt.code=U61_Event::NOTHING;
      evt.time=U61_Time::for_event();
      put(evt);
    } 
}

/*--------------------------------------------------------------------------*/
/*
 * send some events which will set up the player's name
 */
void U61_LocalSource::send_name(int time)
{
  U61_Event evt;
  char c;
  char *str;

  //cout<<"Sending name "<<name<<"\n";

  str=name;
  
  evt.time=time;
  evt.author=player;
  evt.target=player;
  evt.code=U61_Event::NAME_LETTER;

  while ((c=*(str++))!=0)
    {
      evt.par=c;
      put(evt);
    }
}

