/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   data.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: the data object holds all the data of the game
 *              this includes the graphics, the sounds and anything
 *              which is included in the datafile
 *              it provides an easy and standard way for all other
 *              classes to access the game resources
 */

/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include <ClanLib/mikmod.h>
#include <string.h>

#include "data.h"
#include "const.h"
#include "debug.h"
#include "platform.h"

/*---------------------------------------------------------------------------
 variants
 ---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * creates an uninitialized data object
 */
U61_Data::U61_Data()
{

}

/*--------------------------------------------------------------------------*/
/*
 * destroys a data object
 */
U61_Data::~U61_Data()
{
}

/*--------------------------------------------------------------------------*/
/*
 * unloads the content of the last loaded datafile
 */
void U61_Data::unload()
{
    /*
     * now we unload everything the manager loaded
     */
    manager->unload_all_resources();

    /*
     * now the manager is freed
     */
    delete manager;
    manager=NULL;
}

/*--------------------------------------------------------------------------*/
/*
 * loads the data from a disk .dat file
 * returns true if load is successful, false otherwise
 * this is the first part of loading, here we only load things that
 * do not require to have clanlib graphical mode set for instance
 */
bool U61_Data::load1(char *name)
{
    bool success=true;
    char full_name[U61_STRING_SIZE];
    
    /*
     * adds the install path before the name of the data file
     * under unix, INSTALL_DATA_PATH is usually /usr/local/share/u61/
     * but under windows it would simply be ./
     */
    strcpy(full_name,U61_Platform::get_data_path());
    strcat(full_name,name);

    /*
     * creation of the resource manager, which will be
     * used to load all the sections
     */
    //cout<<"Loading data from \""<<full_name<<"\".\n";
    manager=CL_ResourceManager::create(full_name,true);

    if (manager!=NULL)
    {
        if (success)
        {
            //cout<<"Loading properties\n";
            success=success && load_properties();
        }
    }
    else
    {
        success=false;
    }

    return success;
}

/*--------------------------------------------------------------------------*/
/*
 * this function performs the second part of loading of data
 * here, most of the time, the graphic mode has to be set
 */
bool U61_Data::load2()
{
    bool success=true;
 
    if (manager!=NULL)
    {
        /*
         * the following code loads all the different sections
         * if one section fails, the load process is aborted
         */
        if (success)
        {
            //cout<<"Loading fonts\n";
	    success=success && load_fonts();
        }
        
        if (success)
        {
            //cout<<"Loading sounds\n";
	    success=success && load_sounds();
        }
        
        if (success)
        {
            //cout<<"Loading musics\n";
	    success=success && load_musics();
        }

        if (success)
        {
            //cout<<"Loading squares\n";
	    success=success && load_squares();
        }

        if (success)
        {
            //cout<<"Loading maps\n";
	    success=success && load_maps();
        }

        if (success)
        {
            //cout<<"Loading background\n";
	    success=success && load_back();
        }

        if (success)
        {
            //cout<<"Loading info\n";
	    success=success && load_info();
        }
    }
    else
    {
        success=false;
    }

    return success;
}

/*--------------------------------------------------------------------------*/
/*
 * loads global properties 
 */
bool U61_Data::load_properties()
{
    bool success=true;

    screen_w=CL_Res_Integer::load(ID_PROPERTY_SCREEN_W,manager);
    screen_h=CL_Res_Integer::load(ID_PROPERTY_SCREEN_H,manager);

    offset_map_x[SIZE_BIG]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_MAP_X_BIG,manager);
    offset_map_y[SIZE_BIG]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_MAP_Y_BIG,manager);
    offset_name_x[SIZE_BIG]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_NAME_X_BIG,manager);
    offset_name_y[SIZE_BIG]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_NAME_Y_BIG,manager);
    offset_score_x[SIZE_BIG]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_SCORE_X_BIG,manager);
    offset_score_y[SIZE_BIG]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_SCORE_Y_BIG,manager);
    offset_target_x[SIZE_BIG]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_TARGET_X_BIG,manager);
    offset_target_y[SIZE_BIG]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_TARGET_Y_BIG,manager);
    offset_preview_x[SIZE_BIG]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_PREVIEW_X_BIG,manager);
    offset_preview_y[SIZE_BIG]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_PREVIEW_Y_BIG,manager);
    offset_misc_x[SIZE_BIG]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_MISC_X_BIG,manager);
    offset_misc_y[SIZE_BIG]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_MISC_Y_BIG,manager);

    offset_map_x[SIZE_SMALL]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_MAP_X_SMALL,manager);
    offset_map_y[SIZE_SMALL]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_MAP_Y_SMALL,manager);
    offset_name_x[SIZE_SMALL]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_NAME_X_SMALL,manager);
    offset_name_y[SIZE_SMALL]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_NAME_Y_SMALL,manager);
    offset_score_x[SIZE_SMALL]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_SCORE_X_SMALL,manager);
    offset_score_y[SIZE_SMALL]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_SCORE_Y_SMALL,manager);
    offset_target_x[SIZE_SMALL]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_TARGET_X_SMALL,manager);
    offset_target_y[SIZE_SMALL]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_TARGET_Y_SMALL,manager);
    offset_preview_x[SIZE_SMALL]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_PREVIEW_X_SMALL,manager);
    offset_preview_y[SIZE_SMALL]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_PREVIEW_Y_SMALL,manager);
    offset_misc_x[SIZE_SMALL]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_MISC_X_SMALL,manager);
    offset_misc_y[SIZE_SMALL]=
        CL_Res_Integer::load(ID_PROPERTY_OFFSET_MISC_Y_SMALL,manager);

    return success;    
}

/*--------------------------------------------------------------------------*/
/*
 * loads the fonts
 */
bool U61_Data::load_fonts()
{
    bool success=true;

    font_menu=CL_Font::load(ID_FONT_MENU,manager);
    success=success && (font_menu!=NULL);
   
    if (!success)
    {
        cout<<"Font loading failed\n";
    }

    font_info[SIZE_BIG]=CL_Font::load(ID_FONT_INFO_BIG,manager);
    success=success && (font_info[SIZE_BIG]!=NULL);
   
    if (!success)
    {
        cout<<"Font loading failed\n";
    }

    font_info[SIZE_SMALL]=CL_Font::load(ID_FONT_INFO_SMALL,manager);
    success=success && (font_info[SIZE_SMALL]!=NULL);
   
    if (!success)
    {
        cout<<"Font loading failed\n";
    }

    return success;
}

/*--------------------------------------------------------------------------*/
/*
 * loads the sounds
 */
bool U61_Data::load_sounds()
{
    bool success=true;

    sound_game_start=CL_SoundBuffer::load(ID_SOUND_GAME_START,manager);
    success=success && (sound_game_start!=NULL);

    sound_menu_move=CL_SoundBuffer::load(ID_SOUND_MENU_MOVE,manager);
    success=success && (sound_menu_move!=NULL);

    sound_menu_validate=CL_SoundBuffer::load(ID_SOUND_MENU_VALIDATE,manager);
    success=success && (sound_menu_validate!=NULL);

    sound_block_touch=CL_SoundBuffer::load(ID_SOUND_BLOCK_TOUCH,manager);
    success=success && (sound_block_touch!=NULL);

    sound_block_pattern=CL_SoundBuffer::load(ID_SOUND_BLOCK_PATTERN,manager);
    success=success && (sound_block_pattern!=NULL);

    sound_player_start=CL_SoundBuffer::load(ID_SOUND_PLAYER_START,manager);
    success=success && (sound_player_start!=NULL);

    sound_curse_receive=CL_SoundBuffer::load(ID_SOUND_CURSE_RECEIVE,manager);
    success=success && (sound_curse_receive!=NULL);

    if (!success)
    {
        cout<<"Sound loading failed\n";
    }

    return success;
}

/*--------------------------------------------------------------------------*/
/*
 * loads the musics
 */
bool U61_Data::load_musics()
{
    bool success=true;

//    mikmod_test=CL_Streamed_MikModSample::create("data/music/country.mod",NULL,true);
//    success=success && (mikmod_test!=NULL);
   
    if (!success)
    {
        cout<<"Music loading failed\n";
    }

    return success;
}

/*--------------------------------------------------------------------------*/
/*
 * loads the squares
 */
bool U61_Data::load_squares()
{
    bool success=true;
    int type;
    char id[U61_STRING_SIZE];
//    CL_Surface *temp;

    for (type=0; type<NB_SQUARE_TYPE; ++type)
    {
        strcpy(id,ID_SQUARE_SMALL);
        id[strlen(id)-1]='0'+type;
        square[type][SIZE_SMALL]=CL_Surface::load(id,manager);
        success=success && (square[type][SIZE_SMALL]!=NULL);

        strcpy(id,ID_SQUARE_BIG);
        id[strlen(id)-1]='0'+type;
        square[type][SIZE_BIG]=CL_Surface::load(id,manager);
        success=success && (square[type][SIZE_BIG]!=NULL);
    }

    strcpy(id,ID_SQUARE_SMALL);
    id[strlen(id)-1]=0;
    strcat(id,ID_SQUARE_CURSE_BG);
    curse_bg[SIZE_SMALL]=CL_Surface::load(id,manager);
    success=success && (curse_bg[SIZE_SMALL]!=NULL);

    strcpy(id,ID_SQUARE_BIG);
    id[strlen(id)-1]=0;
    strcat(id,ID_SQUARE_CURSE_BG);
    curse_bg[SIZE_BIG]=CL_Surface::load(id,manager);
    success=success && (curse_bg[SIZE_BIG]!=NULL);

    strcpy(id,ID_SQUARE_SMALL);
    id[strlen(id)-1]=0;
    strcat(id,ID_SQUARE_CURSE_FG);
    curse_fg[SIZE_SMALL]=CL_Surface::load(id,manager);
    success=success && (curse_fg[SIZE_SMALL]!=NULL);

    strcpy(id,ID_SQUARE_BIG);
    id[strlen(id)-1]=0;
    strcat(id,ID_SQUARE_CURSE_FG);
    curse_fg[SIZE_BIG]=CL_Surface::load(id,manager);
    success=success && (curse_fg[SIZE_BIG]!=NULL);

    strcpy(id,ID_SQUARE_SMALL);
    id[strlen(id)-1]=0;
    strcat(id,ID_SQUARE_PREVISION);
    prevision[SIZE_SMALL]=CL_Surface::load(id,manager);
    success=success && (prevision[SIZE_SMALL]!=NULL);

    strcpy(id,ID_SQUARE_BIG);
    id[strlen(id)-1]=0;
    strcat(id,ID_SQUARE_PREVISION);
    prevision[SIZE_BIG]=CL_Surface::load(id,manager);
    success=success && (prevision[SIZE_BIG]!=NULL);

    strcpy(id,ID_SQUARE_SMALL);
    id[strlen(id)-1]=0;
    strcat(id,ID_SQUARE_EXPLOSION);
    explosion[SIZE_SMALL]=CL_Surface::load(id,manager);
    success=success && (explosion[SIZE_SMALL]!=NULL);

    strcpy(id,ID_SQUARE_BIG);
    id[strlen(id)-1]=0;
    strcat(id,ID_SQUARE_EXPLOSION);
    explosion[SIZE_BIG]=CL_Surface::load(id,manager);
    success=success && (explosion[SIZE_BIG]!=NULL);

    if (!success)
    {
        cout<<"Square loading failed\n";
    }
    else
    {
        square_w[SIZE_SMALL]=square[0][SIZE_SMALL]->get_width();
        square_h[SIZE_SMALL]=square[0][SIZE_SMALL]->get_height();
        square_w[SIZE_BIG]=square[0][SIZE_BIG]->get_width();
        square_h[SIZE_BIG]=square[0][SIZE_BIG]->get_height();
    }

    return success;
}

/*--------------------------------------------------------------------------*/
/*
 * loads the map images
 */
bool U61_Data::load_maps()
{
    bool success=true;
    int level;
    int level_max;
    char id[U61_STRING_SIZE];

    level_max=CL_Res_Integer::load(ID_MAP_NUMBER,manager,1);
    nb_map=0;
    for (level=0; level<level_max; ++level)
    {
        strcpy(id,ID_MAP_SMALL);
        id[strlen(id)-1]='0'+level;
        map[level][SIZE_SMALL]=CL_Surface::load(id,manager);
        success=success && (map[level][SIZE_SMALL]!=NULL);

        strcpy(id,ID_MAP_BIG);
        id[strlen(id)-1]='0'+level;
        map[level][SIZE_BIG]=CL_Surface::load(id,manager);
        success=success && (map[level][SIZE_BIG]!=NULL);
    }


    if (!success)
    {
        cout<<"Map loading failed\n";
    }
    else
    {
        map_w[SIZE_SMALL]=map[0][SIZE_SMALL]->get_width();
        map_h[SIZE_SMALL]=map[0][SIZE_SMALL]->get_height();
        map_w[SIZE_BIG]=map[0][SIZE_BIG]->get_width();
        map_h[SIZE_BIG]=map[0][SIZE_BIG]->get_height();
    }

    return success;
}

/*--------------------------------------------------------------------------*/
/*
 * loads the background image 
 */
bool U61_Data::load_back()
{
    bool success=true;

    back=CL_Surface::load(ID_BACK_MAIN,manager);
    success=success && (back!=NULL);

    if (!success)
    {
        cout<<"Back loading failed\n";
    }

    return success;
}

/*--------------------------------------------------------------------------*/
/*
 * loads the info textures (where scores are displayed)
 */
bool U61_Data::load_info()
{
    bool success=true;

    info[SIZE_BIG]=CL_Surface::load(ID_INFO_BIG,manager);
    success=success && (info[SIZE_BIG]!=NULL);
    info[SIZE_SMALL]=CL_Surface::load(ID_INFO_SMALL,manager);
    success=success && (info[SIZE_SMALL]!=NULL);

    if (!success)
    {
        cout<<"Info loading failed\n";
    }
    else
    {
        info_w[SIZE_SMALL]=info[SIZE_SMALL]->get_width();
        info_h[SIZE_SMALL]=info[SIZE_SMALL]->get_height();
        info_w[SIZE_BIG]=info[SIZE_BIG]->get_width();
        info_h[SIZE_BIG]=info[SIZE_BIG]->get_height();
    }

    return success;
}

