/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   game.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: basically contains the main game loop, the object
 *              game is created by the application object, and then
 *              controls everything. I tried to keep this file as small
 *              as possible but it's not so easy...
 *              however, if you want to see how the game works, this
 *              is *the* file to check. 
 *              it loads, the config, starts the gfx mode, loads data
 *              then launches the menu etc...
 */

/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/
  
#include "block.h"
#include "game.h"
#include "time.h"
#include "mainmenu.h"
#include "sound.h"
#include "music.h"
#include "square.h"
#include "script.h"
#include "global.h"
#include "platform.h"

/*---------------------------------------------------------------------------
 globals
 ---------------------------------------------------------------------------*/
 

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * initialization of the game
 */ 
U61_Game::U61_Game()
{
//    cout<< "U61 game object created\n";
}

/*--------------------------------------------------------------------------*/
/* 
 * de-initialization of the game
 */ 
U61_Game::~U61_Game()
{
//    cout << "Display : " <<  calls_to_display << " Logic : " << calls_to_logic << " Menu : " << calls_to_menu << " Poll : " << calls_to_poll << " Poll CL : " << calls_to_poll_clanlib << " Poll U61 : " << calls_to_poll_u61 << "\n";
}

/*--------------------------------------------------------------------------*/
/*
 * performs initialization of the game
 * this contains functions that are required for the game to work but
 * are likely to fail is the game is not correctly installed or
 * parametered
 */
bool U61_Game::init()
{
    bool success=true;
    
    calls_to_poll=0;
    calls_to_poll_clanlib=0;
    calls_to_poll_u61=0;
    calls_to_display=0;
    calls_to_menu=0;
    calls_to_logic=0;

    U61_Global::config.read();
    
    U61_Script::init();

    if (success)
    {
        success=success && U61_Global::data.load1
               (U61_Global::config.data_file);
    }

    if (success)
    {
        cout<<"Trying video mode\n";
        CL_Display::set_videomode(U61_Global::data.screen_w,
                                  U61_Global::data.screen_h,16,false);
    }

    if (success)
    {
        success=success && U61_Global::data.load2();
    }  
    if (success)
    {
        success=success && U61_Global::config.load_script();
    }  

    //cout<<"Setting video mode\n";
    //CL_Display::set_videomode(640,480,16,false);

    return success;
}

/*--------------------------------------------------------------------------*/
/*
 * closes the game sand exits "cleanly" ;)
 */
void U61_Game::deinit()
{

    U61_Global::config.write();
    U61_Global::data.unload();
}

/*--------------------------------------------------------------------------*/
/*
 * the run function calls the logic and display functions
 * the display and logic functions may never be running at the same time,
 * therefore, there's no need to lock memory 
 */
void U61_Game::run()
{
    int done_for_this_display;
    U61_Menu *exit_menu;

    state=STATE_GO;
    
    bind_keys();
    reset_players();

    /*
     * we set all the time parameters to default values
     * these may be changed afterwards for real play but they
     * still need to be set for the game not to crash...
     */
    U61_Time::reset();
    last_clanlib_time_polled=CL_System::get_time();
    last_u61_time_polled=U61_Time::for_event();

    /*
     * the game starts on the main menu, which has the exit menu as
     * a parent, logical, eh?
     *
     */
    exit_menu=new U61_ExitMenu();
    current_menu=new U61_MainMenu(exit_menu);

    /*
     * Plays a startup sound
     */
    U61_Sound::play_game_start();
    U61_Music::test();

    /*
     * main game loop
     */
    while (state==STATE_GO) 
    {
        /*
         * we perform calls to the logic function, but we don't
         * call it more than "max_logic_per_display" times
         * this way, the player will always have its screen
         * updated, even if his computer is not fast enough
         */
        done_for_this_display=0;
        while (U61_Time::is_logic_needed() 
               && done_for_this_display < MAX_LOGIC_PER_DISPLAY)
        {
            poll();
            logic();
            U61_Time::acknowledge_logic();
            done_for_this_display++;
        }

        /*
         * if "max_logic_per_display" calls to logic have been
         * done, it means the computer is slow, therefore, we
         * force the computer to slow down the game, since it's
         * not powerfull enough to run it at full speed
         */
        if (done_for_this_display>=MAX_LOGIC_PER_DISPLAY)
        {
          /*  while (U61_Time::is_logic_needed())
            {
                U61_Time::slow_down();
            }*/
        }

        /*
         * call the display, menu and flip fonctions once per loop
         */
        poll();
        display();
        poll();
        menu();
        poll();
        CL_Display::flip_display(false);
    };
}

/*--------------------------------------------------------------------------*/
/*
 * stops the game, ie exits the main game loop, 
 */
void U61_Game::stop()
{
    state=STATE_STOP;
}

/*--------------------------------------------------------------------------*/
/*
 * polls the keyboard, and calls any function that has to be call
 * very often, and that is fast to execute
 */
void U61_Game::poll()
{
    int i;
    int clanlib_time;
    int u61_time;

    clanlib_time=CL_System::get_time();
    if (last_clanlib_time_polled<clanlib_time)
    {
        /*
         * required by ClanLib, polls the keyboard and does 
         * some system low level stuff
         */
        CL_System::keep_alive();
        last_clanlib_time_polled=clanlib_time;
        calls_to_poll_clanlib++;

        u61_time=U61_Time::for_event();
        /*
         * this test disables useless polling
         * polling is done at a maximum rate of once per game cycle
         */
        if (last_u61_time_polled<u61_time)
        {
            last_u61_time_polled=u61_time;   

            menu_input.poll(u61_time);
            for (i=0;i<U61_Layout::NB_PLAYERS_LOCAL;++i)
            {
                player_input[i].poll(u61_time);
            }
            //cout<<"polled "<<ticks_now<<"\n";

            calls_to_poll_u61++;     
        }
    }
    calls_to_poll++;
}
 
/*--------------------------------------------------------------------------*/
/* 
 * logic function of the game
 * the logic function is called at a fixed time rate, no matter how long
 * it takes to perform the display. this way, the game will run at the
 * same speed on any kind of machine.
 */ 
void U61_Game::logic()
{
    pump_player_events();

    //U61_Script::test();
    calls_to_logic++;
}

/*--------------------------------------------------------------------------*/
/* 
 * display function of the game
 * this function performs an entire update of the game display
 * it is called as often as possible, depending on how fast the machine is
 */ 
void U61_Game::display()
{
    CL_Display::clear_display();

    U61_Square::update_cycle();
/*
    for (i=0;i<U61_Layout::NB_PLAYERS_TOTAL;++i)
    {
        if (player[i].is_available())
        {
            player[i].draw(0,0);
        }
    }
 */   
    layout.draw();
    if (current_menu!=NULL)
    {
        current_menu->draw();
    }

    calls_to_display++;  
}

/*--------------------------------------------------------------------------*/
/* 
 * controls the menu. when the game is running it does nothing but waiting
 * for a keypress
 */
void U61_Game::menu()
{
    U61_Menu *next_menu;
    U61_Event key;

    next_menu=current_menu;

    if (!menu_input.empty())
    {
        key=menu_input.get();
        next_menu=current_menu->action(key.code);
    }

    if (next_menu==NULL)
    {
        stop();
    }
    else
    {
        if (next_menu==current_menu->get_parent())
        {
            delete current_menu;
        }

        current_menu=next_menu;
    }

    calls_to_menu++;
}

/*--------------------------------------------------------------------------*/
/*
 * starts a local game, with only one player, without asking questions
 */
void U61_Game::start_quick()
{
    cout<<"quick start\n";
    time_reset();
    reset_players();
    player[0].init_local(&(player_input[0]),0);
    layout.init_alone(player);
}

/*--------------------------------------------------------------------------*/
/*
 * starts a local game, with only one player
 */
void U61_Game::start_local_alone()
{
    cout<<"local alone game started\n";
    time_reset();
    reset_players();
    player[0].init_local(&(player_input[0]),0);
    layout.init_alone(player);
}

/*--------------------------------------------------------------------------*/
/*
 * starts a local game, with up to 4 players
 */
void U61_Game::start_local_multiplayer()
{
    cout<<"local multiplayer game started\n";
    time_reset();
    reset_players();
    layout.reset();
}

/*--------------------------------------------------------------------------*/
/*
 * creates a new network game 
 */
void U61_Game::start_network_new()
{
    cout<<"new network game started\n";
    time_reset();
    reset_players();
    layout.reset();
}

/*--------------------------------------------------------------------------*/
/*
 * joins an existing network game
 */
void U61_Game::start_network_join()
{
    cout<<"joining network game\n";
    time_reset();
    reset_players();
    layout.reset();
}

/*--------------------------------------------------------------------------*/
/*
 * resets the time
 */
void U61_Game::time_reset(int time)
{
    int i;

    U61_Time::reset(time);
    last_u61_time_polled=time;
    menu_input.reset();
    for (i=0;i<U61_Layout::NB_PLAYERS_LOCAL;++i)
    {
        player_input[i].reset();
    }    
}

/*--------------------------------------------------------------------------*/
/*
 * bind the different keys defined in the config object to the input
 * objects. has to be called each time a key setting is changed
 */
void U61_Game::bind_keys()
{
    int i;

    for (i=0;i<U61_Layout::NB_PLAYERS_LOCAL && i<U61_Config::NB_PLAYERS;++i)
    {
        player_input[i].init(&(U61_Global::config.player[i]));
    }
} 

/*--------------------------------------------------------------------------*/
/*
 * resets the players, after this, no player will be active
 */
void U61_Game::reset_players()
{
    int i;

    for (i=0;i<U61_Layout::NB_PLAYERS_TOTAL;++i)
    {
        player[i].reset();
        player[i].set_id(i);
    }
} 

/*--------------------------------------------------------------------------*/
/*
 * pumps all the events associated to players
 * ie it checks incoming events and generates events to be transfered
 */
void U61_Game::pump_player_events()
{
    int i;

    for (i=0;i<U61_Layout::NB_PLAYERS_TOTAL;++i)
    {
        if (player[i].is_available())
        {
            player[i].pump_events();
        }
    }
} 

