/* txt2xhtml - convert plain text to XHTML document
   Copyright (C) 2004  Krzysztof Jurewicz

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   You can contact the author by sending email to nerf@poczta.fm */

#include <stdio.h>
#if STDC_HEADERS
# include <string.h>
#else
# include <strings.h>
#endif
#include <getopt.h>
#include <ctype.h>
#include <stdlib.h>
#include <stddef.h>
#include <errno.h>
#include <error.h>
#include <libintl.h>
#define _(String) gettext(String)
#define N_(String)

#include "config.h"

/* replace signs by entities that represent them in XHTML */
char*
replace (const int sign, int* previous_space, const unsigned long int tab_width,
	 unsigned long int* line_length, const int replace_tags_signs)
{
  if (sign == '\"')
    return "&quot;";
  if (sign == '&')
    return "&amp;";
  if (sign == '<' && replace_tags_signs)
    return "&lt;";
  if (sign == '>' && replace_tags_signs)
    return "&gt;";
  if (sign == '\'')
    return "&apos;";
  if (sign == ' ')
    return "&nbsp;";
  else
    *previous_space = 0;
  if (sign=='\t')
      {
	unsigned long int quantity = *line_length % tab_width;
	char* tabs;
	if (*previous_space || *line_length == 0)
	  {
	    quantity++;
	    tabs = (char*) xmalloc ((unsigned long) (tab_width - quantity) * 6 + 2);
	    strcpy (tabs, " ");
	    *line_length++;
	  }
	else
	  {
	    tabs = (char*) xmalloc ((unsigned long) (tab_width - quantity) * 6 + 1);
	    *previous_space = 1;
	    strcpy (tabs, "");
	  }
	while (quantity != tab_width)
	  {
	    strcat (tabs, "&nbsp;");
	    quantity++;
	    *line_length++;
	  }
	return tabs;
      }
  char temp[] = " ";
  temp[0] = sign;
  char temp2[1];
  strcpy (temp2, temp);
  return temp2;
}

/* copy value between two arrays of chars if it is valid hexadecimal (but without '#') color value */
void
set_color (char* target[], const char color[])
{
  if (strlen (color) == 3 || strlen (color) == 6)
    {
      int i = 0;
      while (color[i] != NULL)
	{
	  if (!isxdigit (color[i]))
	    error (1, 0, _("%s: not valid color value"), color);
	  i++;
	}
      strcpy (target, color);
      return;
    }
  else
    error (1, 0, _("%s: not valid color value"), color);
}

/* allocate memory and check for an error */
void*
xmalloc (size_t size)
{
  register void* value = malloc (size);
  if (value == NULL)
    error (1, 0, _("virtual memory exhausted"));
  else
    return value;
}

/* reallocate memory and check for an error */
void*
xrealloc (const void* ptr, size_t size)
{
  register void* value = realloc (ptr, size);
  if (value == NULL)
    error (1, 0, _("virtual memory exhausted"));
  else
    return value;
}

void
xfree (void* ptr)
{
  free (ptr);
  ptr = 0;
}

/* open a file and check for an error */
FILE*
xfopen64 (const char* file, const char* opentype)
{
  FILE* temp = fopen64 (file, opentype);
  if (temp == NULL)
    error (1, 0, _("%s: no such file"), file);
  else
    return temp;
}

/* copy characters between two strings */
void
ptr_strcpy (char* target[], const char* input[])
{
  *target = (char*) xmalloc ((unsigned) strlen (*input) + 1);
  strcpy (*target, *input);
}

/* convert a string to unsigned long int */
unsigned long int
u_long_int (const char* integer[])
{
  char* tailptr;
  errno = 0;
  unsigned long int temp = strtoul (*integer, &tailptr, 0);
  if (strcmp (tailptr, "") != 0 || errno)
    {
      if (errno)
	error (1, errno, "%s", *integer);
      else
	error (1, 0, _("%s: not valid integer value"), *integer);
    }
  else
    return temp;
}

int main (int argc, char* argv[])
{
  /* Launch gettext */
  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);
  
  /* Were command-line options specified? */
  static int help_flag = 0;
  static int version_flag = 0;

  /* Has user specified some styles? */
  static int background_color_flag = 0;
  static int color_flag = 0;
  static int a_link_color_flag = 0;
  static int a_visited_color_flag = 0;
  static int a_hover_color_flag = 0;
  static int a_active_color_flag = 0;

  /* Shall we convert url's and www addresses? */
  static int url_flag = 1;
  static int www_flag = 1;

  /* Shall we replace tags '<' and '>' signs? */
  static int replace_tags_signs = 1;

  /* Shall we output complete XHTML document or only content of the <body> tag? */
  static int complete_document = 1;

  /* Shall we output the <div> tag? */
  static int div_tag = 1;

  /* Shall we output XHTML code to stdout insted of file? */
  static int stdout_output = 0;

  /* Is there more than one input file? */
  static int multiple_files = 0;

  /* Shall we generate output filenames automatically? */
  static int auto_filenames = 0;

  /* Have we defined any styles? */
  static int styles_defined = 0;
  static int body_styles = 0;
  static int a_link_styles = 0;
  static int a_visited_styles = 0;
  static int a_hover_styles = 0;
  static int a_active_styles = 0;

  /* page title */
  char* title = 0;

  /* document language, "en" by default */
  static char lang[] = "en";

  /* encoding */
  char* encoding = 0;

  /* default encoding */
  static char* default_encoding = "utf-8";

  /* background color, white by default */
  static char background_color[6] = "fff";

  /* text color, black by default */
  static char color[6] = "000";

  /* link colors */
  static char a_link_color[6] = "00e";
  static char a_visited_color[6] = "60c";
  static char a_hover_color[6] = "00e";
  static char a_active_color[6] = "e00";

  /* external stylesheet */
  char* external_stylesheet = 0;

  /* imported stylesheet */
  char* stylesheet = 0;

  /* style specified through the command line */
  char* style = 0;

  /* output filename */
  char* output_filename = 0;

  /* default output filename */
  static char* default_output_filename = "a.xhtml";

  /* extension for output filenames */
  char* filename_extension = 0;

  /* default filename extension */
  static char* default_filename_extension = ".xhtml";

  /* tab width */
  unsigned long int tab_width = 8;

  /* data used while converting to XHTML is below */

  /* current letter */
  static int ch;

  /* Was previous character a space? */
  static int previous_space = 0;

  /* string used to store current letter */
  static char temp[8] = "";
  
  /* current word */
  char* word;

  /* length of current word */
  unsigned long int length = 1;

  /* is current word an url? */
  static int is_url = 0;

  /* does current word end with a dot (set as 1 only if the word is a url)? */
  static int ends_dot = 0;

  /* is current word a www address without information about protocol (without "http://")? */
  static int is_www = 0;

  /* length of current line */
  static unsigned long int line_length = 0;

  /* input and output streams */
  static FILE* input = NULL;
  static FILE* output = NULL;

  /* Let's check command line parameters */
  static struct option long_options[]=
    {
      {"help", no_argument, &help_flag, 1},
      {"version", no_argument, &version_flag, 1},
      {"background-color", optional_argument, 0, 'b'},
      {"color", optional_argument, 0, 'c'},
      {"a-link-color", optional_argument, 0, 'l'},
      {"a-hover-color", optional_argument, 0, 'h'},
      {"a-active-color", optional_argument, 0, 'a'},
      {"a-visited-color", optional_argument, 0, 'v'},
      {"title", required_argument, 0, 't'},
      {"disable-www-links", no_argument, &www_flag, 0},
      {"disable-links", no_argument, &url_flag, 0},
      {"body-only", no_argument, &complete_document, 0},
      {"without-div", no_argument, &div_tag, 0},
      {"stylesheet", required_argument, 0, 's'},
      {"external-stylesheet", required_argument, 0, 'x'},
      {"style", required_argument, 0, 'S'},
      {"lang", required_argument, 0, 'L'},
      {"to-stdout", no_argument, &stdout_output, 1},
      {"filename-extension", required_argument, 0, 'f'},
      {"auto-filenames", no_argument, &auto_filenames, 1},
      {"output", required_argument, 0, 'o'},
      {"tab-width", required_argument, 0, 'T'},
      {"encoding", required_argument, 0, 'e'},
      {0, 0, 0, 0}
    };

  static int option_index = 0;

  static int o;

  while (1)
      {
	o = getopt_long (argc, argv, "o:t:", long_options, &option_index);
	if (o == -1)
	  break;
	switch (o)
	  {
	  case 'b':
	    styles_defined = 1;
	    body_styles = 1;
	    background_color_flag = 1;
	    if (optarg != NULL)
	      set_color (&background_color, optarg);
	    break;

	  case 'c':
	    styles_defined = 1;
	    body_styles = 1;
	    color_flag = 1;
	    if (optarg != NULL)
	      set_color (&color, optarg);
	    break;

	  case 'l':
	    styles_defined = 1;
	    a_link_styles = 1;
	    a_link_color_flag = 1;
	    if (optarg != NULL)
	      set_color (&a_link_color, optarg);
	    break;

	  case 'a':
	    styles_defined = 1;
	    a_active_styles = 1;
	    a_active_color_flag = 1;
	    if (optarg != NULL)
	      set_color (&a_active_color, optarg);
	    break;

	  case 'h':
	    styles_defined = 1;
	    a_hover_styles = 1;
	    a_hover_color_flag = 1;
	    if (optarg != NULL)
	      set_color (&a_hover_color, optarg);
	    break;

	  case 'v':
	    styles_defined = 1;
	    a_visited_styles = 1;
	    a_visited_color_flag = 1;
	    if (optarg != NULL)
	      set_color (&a_visited_color, optarg);
	    break;

	  case 't':
	    ptr_strcpy (&title, &optarg);
	    break;

	  case 's':
	    styles_defined = 1;
	    ptr_strcpy (&stylesheet, &optarg);
	    break;

	  case 'x':
	    ptr_strcpy (&external_stylesheet, &optarg);
	    break;

	  case 'S':
	    ptr_strcpy (&style, &optarg);
	    break;

	  case 'L':
	    if (strlen (optarg) == 2)
	      strcpy (lang, optarg);
	    else
		error (1, 0, _("%s: not valid language value"), optarg);
	    break;

	  case 'f':
	    ptr_strcpy (&filename_extension, &optarg);

	  case 'o':
	    ptr_strcpy (&output_filename, &optarg);
	    break;

	  case 'T':
	    tab_width = u_long_int (&optarg);
	    break;

	  case 'e':
	    ptr_strcpy (&encoding, &optarg);
	    break;
	  }
      }

  if (argc - 1 - optind > 0)
    {
      multiple_files = 1;
      auto_filenames = 1;
    }

  /* Option '--version' has higher priority than '--help'. */
  if (help_flag || version_flag)
    {
      if (version_flag)
	printf ("%s %s \n\
Copyright (C) 2004 Krzysztof Jurewicz\n\
txt2xhtml comes with ABSOLUTELY NO WARRANTY, not even with implied\n\
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE\n\
You can redistribute and/or modify txt2xhtml\n\
under the terms of the GNU General Public License\n\
For more information about these matters, see the file named COPYING\n", PACKAGE, VERSION);
      else
	{
	  printf (_("Usage: %s [options] files...\n\n"), argv[0]);
	  puts (_("Options:"));
	  puts (_("  -o, --output=<file>\t\tset output file as <file>"));
	  puts (_("  --help\t\t\tdisplay this help"));
	  puts (_("  --version\t\t\tdisplay information about version"));
	  puts (_("  --stylesheet=<file>\t\timport stylesheet <file> to document"));
	  puts (_("  --external-stylesheet=<file>\tset external stylesheet as <file>"));
	  puts (_("  --style=<styles>\t\tset local stylesheet as <styles>"));
	  puts (_("  --background-color=<color>\tset background color as #<color>"));
	  puts (_("  --color=<color>\t\tset text color as #<color>"));
	  puts (_("  --a-link-color=<color>\tset a:link color as #<color>"));
	  puts (_("  --a-visited-color=<color>\tset a:visited color as #<color>"));
	  puts (_("  --a-hover-color=<color>\tset a:hover color as #<color>"));
	  puts (_("  --a-active-color=<color>\tset a:active color as #<color>"));
	  puts (_("  -t, --title=<title>\t\tset page title as <title>"));
	  puts (_("  --disable-www-links\t\tdisable converting to links url's\n\
\t\t\t\tstarting with 'www.'"));
	  puts (_("  --disable-links\t\tdisable converting to links all url's"));
	  puts (_("  --body-only\t\t\toutput only the <body> tag content"));
	  puts (_("  --without-div\t\t\tdo not output the <div> tag when the\n\
\t\t\t\t'--body-only' option is used"));
	  puts (_("  --lang=<language>\t\tset page language as <language>"));
	  puts (_("  --encoding=<encoding>\t\tset page encoding as <encoding>"));
	  puts (_("  --tab-width=<width>\t\tset tab width as <width>"));
	  puts (_("  --filename-extension=<ext>\tset automatically created output\n\
\t\t\t\tfilenames extensions as <ext>"));
	  puts (_("  --auto-filenames\t\tautomaticaly create output filenames"));
	  puts (_("  --to-stdout\t\t\tplace output in the standard output"));
	  printf (_("\nReport bugs to <%s>\n"), PACKAGE_BUGREPORT);
	}
      return 0;
    }

  while (optind < argc)
    {
      word = (char*) xmalloc ((unsigned) length);
      strcpy (word, "");

      /* setting filename extenmsion if user didn't specify his own */
      if (!filename_extension)
	ptr_strcpy (&filename_extension, &default_filename_extension);

      /* setting output filename if user didn't specify his own */
      if (!output_filename && !auto_filenames)
	ptr_strcpy (&output_filename, &default_output_filename);
      if (auto_filenames)
	{
	  output_filename = (char*) xmalloc ((unsigned) strlen (argv[optind]) + strlen (filename_extension) + 1);
	  strcpy (output_filename, argv[optind]);
	  strcat (output_filename, filename_extension);
	}

      xfree (filename_extension);

      /* setting title if user didn't specify his own */
      if (!title)
	ptr_strcpy (&title, &argv[optind]);

      /* setting encoding if user didn't specify his own */
      if (!encoding)
	ptr_strcpy (&encoding, &default_encoding);

      /* opens output file */
      if (stdout_output)
	output = stdout;
      else
	output = fopen(output_filename, "w");
      xfree (output_filename);

      if (complete_document)
	{
	  /* inserts first part of document */
	  fprintf (output, "<?xml version=\"1.0\" encoding=\"%s\"?>\n\
<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.1//EN\" \"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd\">\n\
<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"%s\">\n<head>\n", encoding, lang);
	  xfree (encoding);

	  /* inserts title */
	  fprintf (output, "<title>%s</title>\n", title);
	  xfree (title);

	  /* inserts information about generator */
	  fprintf (output, "<meta name=\"generator\" content=\"%s %s\"/>\n", PACKAGE, VERSION);

	  /* inserts external stylesheet */
	  if (external_stylesheet)
	    {
	      fprintf (output, "<link rel=\"stylesheet\" type=\"text/css\" href=\"%s\"/>\n", external_stylesheet);
	      xfree (external_stylesheet);
	    }

	  /* Do we need to specify any styles? */
	  if (styles_defined)
	    {
	      fputs ("<style type=\"text/css\">\n", output);
	      if (stylesheet)
		{
		  int c;
		  input = xfopen64 (stylesheet, "r");
		  xfree (stylesheet);
		  while(1)
		    {
		      c = getc (input);
		      if (c == EOF)
			break;
		      fprintf (output, "%c", c);
		    }
		  fputs ("\n\n", output);
		  fclose(input);
		}
	      if (body_styles)
		{
		  fputs ("html\n{\n", output);
		  if (background_color_flag)
		    {
		      fprintf (output, "\tbackground-color: #%s;\n", background_color);
		    }
		  if (color_flag)
		    {
		      fprintf (output, "\tcolor: #%s;\n", color);
		    }
		  fputs ("}\n\n", output);
		}
	      if (a_link_styles)
		{
		  fputs ("a:link\n{\n", output);
		  if (a_link_color_flag)
		    fprintf (output, "\tcolor: #%s;\n", a_link_color);
		  fputs ("}\n\n", output);
		}
	      if (a_visited_styles)
		{
		  fputs ("a:visited\n{\n", output);
		  if (a_link_color_flag)
		    fprintf (output, "\tcolor: #%s;\n", a_visited_color);
		  fputs ("}\n\n", output);
		}
	      if (a_hover_styles)
		{
		  fputs ("a:hover\n{\n", output);
		  if (a_hover_color_flag)
		    fprintf (output, "\tcolor: #%s;\n", a_hover_color);
		  fputs ("}\n\n", output);
		}
	      if (a_active_styles)
		{
		  fputs ("a:active\n{\n", output);
		  if (a_active_color_flag)
		    fprintf (output, "\tcolor: #%s;\n", a_active_color);
		  fputs ("}\n\n", output);
		}
	      fputs ("</style>\n", output);
	    }
	  fputs ("</head>\n<body", output);
	  if (style)
	    {
	      fprintf (output, " style=\"%s\"", style);
	      xfree (style);
	    }
	  fputs (">\n", output);
	}

      /* If we output a complete document, a <div> tag must be present. */
      if (div_tag || complete_document)
	fputs ("<div>\n", output);

      /* opens input file */
      input = xfopen64(argv[optind], "r");

      /* Let's convert it to XHTML and save it! */
      int i = 0;
      while (i<2)
	{
	  ch = getc(input);
	  if(ch == EOF)
	    i++;
	  if (ch == '\n' || (ch == ' ' && !previous_space) || i == 1)
	    {
	      if (!is_url)
		{
		  fprintf (output, "%s", word);
		}
	      else
		{
		  if (word[length-1] == '.')
		    {
		      word[length-1] = NULL;
		      ends_dot = 1;
		    }
		  fprintf (output, "<a href=\"");
		  if (is_www)
		    fprintf (output, "http://");
		  fprintf (output, "%s", word);
		  fprintf (output, "\">%s</a>", word);
		  if (ends_dot)
		    fprintf (output, ".");
		  is_url = 0;
		  ends_dot = 0;
		  is_www = 0;
		}
	      length = 1;
	      word = (char*) xrealloc (word, (unsigned) length);
	      strcpy (word, "");
	      if (ch == '\n')
		{
		  fprintf (output, "<br/>\n");
		  line_length = 0;
		}
	      if (ch == ' ')
		{
		  putc(' ', output);
		  previous_space = 1;
		}
	    }
	  else
	    {
	      strcpy (temp, replace (ch, &previous_space, tab_width, &line_length, replace_tags_signs));
	      line_length++;
	      length += strlen (temp);
	      word = (char*) xrealloc (word, (unsigned) length);
	      strcat (word, temp);
	      strcpy (temp, "");
	    }
	  if ((strcmp (word, "http://") == 0 || strcmp (word, "ftp://") == 0 || strcmp (word, "news://") == 0) && url_flag)
	    is_url = 1;
	  if (strcmp (word, "www.") == 0 && www_flag)
	    {
	      is_www = 1;
	      is_url = 1;
	    }
	  if (ch != ' ')
	    previous_space = 0;
	}

      /* input file isn't needed anymore */
      fclose (input);

      if (div_tag || complete_document)
	fputs ("\n</div>", output);

      if (complete_document)
	fputs ("\n</body>\n</html>", output);

      /* Finished! */
      fclose (output);
      optind++;
      xfree (word);
    }
  return 0;
}
