/* song.h
 *
 * Copyright 2002 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef _SONG_H
#define _SONG_H

#include "mmd.h"
#include "playseq.h"
#include "block.h"
#include "instrument.h"
#include "midi.h"

struct song {
  /* Name of the song */
  char *name;
  /* Tempo, ticks per line */
  unsigned char tempo, ticksperline;
  /* Number of sections */
  unsigned short numsections;
  /* Section array */
  unsigned short *sections;
  /* Number of blocks */
  unsigned short numplayseqs;
  /* Playing sequence array */
  struct playseq **playseqs;
  /* Number of blocks */
  unsigned short numblocks;
  /* Block array */
  struct block **blocks;
  /* Number of instruments */
  unsigned char numinstruments;
  /* Instrument array */
  struct instrument **instruments;
  /* Maximum number of tracks */
  unsigned short maxtracks;
  /* Track volume array */
  unsigned char *trackvolumes;
  /* Master volume */
  unsigned char mastervolume;
  /* Number of System Exclusive messages */
  unsigned short numsysexes;
  /* System Exclusive messages */
  struct sysex **sysexes;
};

/* Allocates a new song structure and initializes it to default values */
struct song *song_alloc(void);
/* Frees a song structure and its contents */
void song_free(struct song *);
/* Inserts a new block in the block array in the given position */
void song_insert_block(struct song *, int);
/* Deletes a block from the given position of the block array */
void song_delete_block(struct song *, int);
/* Inserts a new playseq in the playseq array in the given position */
void song_insert_playseq(struct song *, int);
/* Deletes a playseq from the given position of the playseq array */
void song_delete_playseq(struct song *, int);
/* Inserts a new section in the section array in the given position */
void song_insert_section(struct song *, int);
/* Deletes a section from the given position of the section array */
void song_delete_section(struct song *, int);
/* Inserts a new SysEx in the SysEx array in the given position */
void song_insert_sysex(struct song *, int);
/* Deletes a SysEx from the given position of the SysEx array */
void song_delete_sysex(struct song *, int);
/* Sets a section in the given position to point somewhere */
void song_set_section(struct song *, int, int);
/* Sets the number of ticks per line for a song */
void song_set_tpl(struct song *, unsigned char);
/* Sets the tempo of a song */
void song_set_tempo(struct song *, unsigned char);
/* If the maximum number of tracks has changed recreate the track volumes */
int song_check_maxtracks(struct song *);
/* Make sure the instrument exists; add instruments if necessary */
void song_check_instrument(struct song *, int);
/* Converts an MMD2 module to a song */
struct song *song_convert_MMD2(struct MMD2 *);
/* Converts MMD effect values at given location */
void song_convert_effects(unsigned char *);
/* Loads a song from an XML file */
struct song *song_load(char *);
/* Parses a song element in an XML file */
struct song *song_parse(xmlDocPtr, xmlNsPtr, xmlNodePtr);
/* Saves a song to an XML file */
void song_save(struct song *, char *);

#endif
