/* song.c
 *
 * Copyright 2002 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <libxml/tree.h>
#include "song.h"
#include "editor.h"
#include "player.h"

char *xmlencode(unsigned char *, int);

extern struct editor editor;

/* Allocates a new song structure and initializes it to default values */
struct song *song_alloc(void) {
  struct song *song;
  int i;

  song=(struct song *)calloc(1, sizeof(struct song));
  song->name=(char *)strdup("Untitled");
  song->tempo=130;
  song->ticksperline=6;
  song->numsections=1;
  song->sections=(unsigned short *)calloc(1, sizeof(unsigned short));
  song->numplayseqs=1;
  song->playseqs=(struct playseq **)calloc(1, sizeof(struct playseq *));
  song->playseqs[0]=playseq_alloc();
  song->numblocks=1;
  song->blocks=(struct block **)calloc(1, sizeof(struct block *));
  song->blocks[0]=block_alloc(4, 64, 1);
  song->numinstruments=1;
  song->instruments=(struct instrument **)calloc(1, sizeof(struct instrument *));
  song->instruments[0]=instrument_alloc();
  song->maxtracks=4;
  song->trackvolumes=(char *)calloc(4, sizeof(char));
  song->mastervolume=127;
  for(i=0; i<song->maxtracks; i++)
    song->trackvolumes[i]=127;

  return(song);
}

/* Frees a song structure and its contents */
void song_free(struct song *song) {
  if(song) {
    if(song->name)
      free(song->name);
    
    if(song->numsections>0)
      free(song->sections);
    
    if(song->numplayseqs>0) {
      int i;
      
      for(i=0; i<song->numplayseqs; i++)
	playseq_free(song->playseqs[i]);
      
      free(song->playseqs);
    }
    
    if(song->numblocks>0) {
      int i;
      
      for(i=0; i<song->numblocks; i++)
	block_free(song->blocks[i]);
      
      free(song->blocks);
    }
    
    if(song->numinstruments>0) {
      int i;
      
      for(i=0; i<song->numinstruments; i++)
	if(song->instruments[i]!=NULL)
	  instrument_free(song->instruments[i]);
      
      free(song->instruments);
    }
    
    if(song->maxtracks>0)
      free(song->trackvolumes);
    
    free(song);
  }
}

/* Inserts a new block in the block array in the given position */
void song_insert_block(struct song *song, int pos) {
  int i;

  /* Check block existence */
  if(pos<0)
    pos=0;
  else if(pos>song->numblocks)
    pos=song->numblocks;

  song->numblocks++;
  /* Reallocate the block array */
  song->blocks=(struct block **)realloc(song->blocks, song->numblocks*sizeof(struct block *));

  /* Move rest of the blocks onwards */
  for(i=song->numblocks-1; i>pos; i--)
    song->blocks[i]=song->blocks[i-1];

  /* Insert a new block similar to the current block */
  song->blocks[pos]=block_alloc(song->blocks[editor.block]->tracks,
				song->blocks[editor.block]->length,
				song->blocks[editor.block]->effectpages);
  
  /* Update playing sequences */
  for(i=0; i<song->numplayseqs; i++) {
    int j;

    for(j=0; j<song->playseqs[i]->length; j++)
      if(song->playseqs[i]->blocknumbers[j]>=pos)
	song->playseqs[i]->blocknumbers[j]++;
  }
}

/* Deletes a block from the given position of the block array */
void song_delete_block(struct song *song, int pos) {
  int i;

  /* Check block existence */
  if(pos<0)
    pos=0;
  else if(pos>=song->numblocks)
    pos=song->numblocks-1;

  /* Don't delete the last block */
  if(song->numblocks>1) {
    /* Free the block in question */
    block_free(song->blocks[pos]);
    song->numblocks--;
    
    /* Move rest of the blocks backwards */
    for(i=pos; i<song->numblocks; i++)
      song->blocks[i]=song->blocks[i+1];
    
    /* Reallocate the block array */
    song->blocks=(struct block **)realloc(song->blocks, song->numblocks*sizeof(struct block *));
    
    /* Update playing sequences */
    for(i=0; i<song->numplayseqs; i++) {
      int j;
      
      for(j=0; j<song->playseqs[i]->length; j++)
	if(song->playseqs[i]->blocknumbers[j]>pos)
	  song->playseqs[i]->blocknumbers[j]--;
    }
  }
}

/* Inserts a new playseq in the playseq array in the given position */
void song_insert_playseq(struct song *song, int pos) {
  int i;

  /* Check playseq existence */
  if(pos<0)
    pos=0;
  else if(pos>song->numplayseqs)
    pos=song->numplayseqs;

  song->numplayseqs++;
  /* Reallocate the playseqs array */
  song->playseqs=(struct playseq **)realloc(song->playseqs, song->numplayseqs*sizeof(struct playseq *));

  /* Move rest of the playseqs onwards */
  for(i=song->numplayseqs-1; i>pos; i--)
    song->playseqs[i]=song->playseqs[i-1];

  /* Insert a new playing sequence */
  song->playseqs[pos]=playseq_alloc();
  
  /* Update sections */
  for(i=0; i<song->numsections; i++) {
    if(song->sections[i]>=pos)
	song->sections[i]++;
  }
}

/* Deletes a playseq from the given position of the playseq array */
void song_delete_playseq(struct song *song, int pos) {
  int i;
  
  /* Check block existence */
  if(pos<0)
    pos=0;
  else if(pos>=song->numplayseqs)
    pos=song->numplayseqs-1;

  /* Don't delete the last playseq */
  if(song->numplayseqs>1) {
    /* Free the playseq to be deleted */
    playseq_free(song->playseqs[pos]);
    song->numplayseqs--;
    
    /* Move rest of the playseqs backwards */
    for(i=pos; i<song->numplayseqs; i++)
      song->playseqs[i]=song->playseqs[i+1];
    
    /* Reallocate the playseqs array */
    song->playseqs=(struct playseq **)realloc(song->playseqs, song->numplayseqs*sizeof(struct playseq *));

    player_refresh_playseq_and_block();
  }
}


/* Inserts a new section in the section array in the given position */
void song_insert_section(struct song *song, int pos) {
  int i;

  /* Check block existence */
  if(pos<0)
    pos=0;
  else if(pos>song->numsections)
    pos=song->numsections;

  song->numsections++;
  /* Reallocate the sections array */
  song->sections=(unsigned short *)realloc(song->sections, song->numsections*sizeof(unsigned short));

  /* Move rest of the sections onwards */
  for(i=song->numsections-1; i>pos; i--)
    song->sections[i]=song->sections[i-1];
}

/* Deletes a section from the given position of the section array */
void song_delete_section(struct song *song, int pos) {
  int i;
  
  /* Check block existence */
  if(pos<0)
    pos=0;
  else if(pos>=song->numsections)
    pos=song->numsections-1;

  /* Don't delete the last section */
  if(song->numsections>1) {
    song->numsections--;
    
    /* Move rest of the sections backwards */
    for(i=pos; i<song->numsections; i++)
      song->sections[i]=song->sections[i+1];
    
    /* Reallocate the sections array */
    song->sections=(unsigned short *)realloc(song->sections, song->numsections*sizeof(unsigned short));

    player_refresh_playseq_and_block();
  }
}

/* Inserts a new sysex in the sysex array in the given position */
void song_insert_sysex(struct song *song, int pos) {
  int i;

  /* Check sysex existence */
  if(pos<0)
    pos=0;
  else if(pos>song->numsysexes)
    pos=song->numsysexes;

  song->numsysexes++;
  /* Reallocate the sysex array */
  song->sysexes=(struct sysex **)realloc(song->sysexes, song->numsysexes*sizeof(struct sysex *));

  /* Move rest of the sysexes onwards */
  for(i=song->numsysexes-1; i>pos; i--)
    song->sysexes[i]=song->sysexes[i-1];

  /* Insert a new sysex similar to the current sysex */
  song->sysexes[pos]=(struct sysex *)calloc(1, sizeof(struct sysex));
}

/* Deletes a sysex from the given position of the sysex array */
void song_delete_sysex(struct song *song, int pos) {
  int i;

  /* Check sysex existence */
  if(pos<0)
    pos=0;
  else if(pos>=song->numsysexes)
    pos=song->numsysexes-1;

  /* Don't delete inexisting sysexes */
  if(song->numsysexes>0) {
    /* Free the sysex in question */
    if(song->sysexes[pos]->data!=NULL)
      free(song->sysexes[pos]->data);
    if(song->sysexes[pos]->name!=NULL)
      free(song->sysexes[pos]->name);
    free(song->sysexes[pos]);
    song->numsysexes--;
    
    /* Move rest of the sysexes backwards */
    for(i=pos; i<song->numsysexes; i++)
      song->sysexes[i]=song->sysexes[i+1];
    
    /* Reallocate the sysex array */
    song->sysexes=(struct sysex **)realloc(song->sysexes, song->numsysexes*sizeof(struct sysex *));
  }
}

/* Sets a section in the given position to point somewhere */
void song_set_section(struct song *song, int pos, int playseq) {
  if(playseq>=0 && playseq<song->numplayseqs)
    song->sections[pos]=playseq;
}

/* Sets the number of ticks per line for a song */
void song_set_tpl(struct song *song, unsigned char ticksperline) {
  song->ticksperline=ticksperline;
}

/* Sets the tempo of a song */
void song_set_tempo(struct song *song, unsigned char tempo) {
  song->tempo=tempo;
}

/* If the maximum number of tracks has changed recreate the track volumes */
int song_check_maxtracks(struct song *song) {
  int i, oldmax=song->maxtracks, max=0;

  /* Check the maximum number of tracks; */
  for(i=0; i<song->numblocks; i++)
    if(song->blocks[i]->tracks>max)
      max=song->blocks[i]->tracks;
  
  /* If the maximum number of tracks as changed... */
  if(oldmax!=max) {
    /* Reallocate volume array */
    song->trackvolumes=(unsigned char *)realloc(song->trackvolumes,
						max*sizeof(unsigned char));

    /* Set new tracks to volume 127 */
    for(i=song->maxtracks; i<max; i++)
      song->trackvolumes[i]=127;

    song->maxtracks=max;
    return 1;
  } else
    return 0;
}

/* Make sure the instrument exists; add instruments if necessary */
void song_check_instrument(struct song *song, int instrument) {
  if(instrument>=song->numinstruments || song->instruments[instrument]==NULL) {
    if(instrument>=song->numinstruments) {
      int i;

      song->instruments=realloc(song->instruments,
				(instrument+1)*sizeof(struct instrument *));
      for(i=song->numinstruments; i<(instrument+1); i++)
	song->instruments[i]=NULL;

      song->numinstruments=instrument+1;
    }
    song->instruments[instrument]=instrument_alloc();
  }
}

/* Converts an MMD2 module to a song */
struct song *song_convert_MMD2(struct MMD2 *mmd) {
  struct song *song;
  int i, j, k, l;
  unsigned char effectconversion[256];

  if(mmd==NULL)
    return NULL;

  /* Create a table for converting MMD effects; values need special handling */
  for(i=0; i<256; i++)
    effectconversion[i]=i;
  effectconversion[0]=EFFECT_MIDI_COMMAND_VALUE;
  effectconversion[1]=EFFECT_PITCH_WHEEL;
  effectconversion[2]=EFFECT_PITCH_WHEEL;
  effectconversion[3]=EFFECT_PITCH_WHEEL;
  effectconversion[4]=EFFECT_NOT_DEFINED;
  effectconversion[5]=EFFECT_NOT_DEFINED;
  effectconversion[6]=EFFECT_NOT_DEFINED;
  effectconversion[7]=EFFECT_NOT_DEFINED;
  effectconversion[8]=EFFECT_NOT_DEFINED;
  effectconversion[9]=EFFECT_TPL;
  effectconversion[10]=EFFECT_VELOCITY;
  effectconversion[11]=EFFECT_PLAYSEQ_POSITION;
  effectconversion[12]=EFFECT_VELOCITY;
  effectconversion[13]=EFFECT_CHANNEL_PRESSURE;
  effectconversion[14]=EFFECT_NOT_DEFINED;
  effectconversion[15]=EFFECT_TEMPO;
  effectconversion[16]=EFFECT_SEND_SYSEX;
  effectconversion[17]=EFFECT_NOT_DEFINED;
  effectconversion[18]=EFFECT_NOT_DEFINED;
  effectconversion[19]=EFFECT_NOT_DEFINED;
  effectconversion[20]=EFFECT_NOT_DEFINED;
  effectconversion[21]=EFFECT_NOT_DEFINED;
  effectconversion[22]=EFFECT_NOT_DEFINED;
  effectconversion[23]=EFFECT_NOT_DEFINED;
  effectconversion[24]=EFFECT_NOT_DEFINED;
  effectconversion[25]=EFFECT_NOT_DEFINED;
  effectconversion[26]=EFFECT_NOT_DEFINED;
  effectconversion[27]=EFFECT_NOT_DEFINED;
  effectconversion[28]=EFFECT_NOT_DEFINED;
  effectconversion[29]=EFFECT_NOT_DEFINED;
  effectconversion[30]=EFFECT_NOT_DEFINED;
  effectconversion[31]=EFFECT_NOTE_DELAY;

  song=(struct song *)calloc(1, sizeof(struct song));
  if(mmd->expdata && mmd->expdata->songname)
    song->name=(char *)strdup(mmd->expdata->songname);
  else
    song->name="Untitled";
  if(mmd->song) {
    /* Tempo */
    song->tempo=mmd->song->deftempo;
    song->ticksperline=mmd->song->tempo2;
    /* Sections */
    song->numsections=mmd->song->songlen;
    song->sections=(unsigned short *)calloc(song->numsections,
					    sizeof(unsigned short));
    for(i=0; i<song->numsections; i++) {
      /* FIX: Where to get these? */
    }
    /* Playing sequences */
    song->numplayseqs=mmd->song->numpseqs;
    song->playseqs=(struct playseq **)calloc(song->numplayseqs,
					     sizeof(struct playseq *));
    for(i=0; i<song->numplayseqs; i++) {
      struct playseq *playseq=(struct playseq *)calloc(1,
						       sizeof(struct playseq));
      struct PlaySeq *PlaySeq=mmd->song->playseqtable[i];
      song->playseqs[i]=playseq;
      if(PlaySeq->name)
	playseq->name=(char *)strdup(PlaySeq->name);
      playseq->length=PlaySeq->length;
      playseq->blocknumbers=(unsigned short *)calloc(playseq->length,
						     sizeof(unsigned short));
      for(j=0; j<song->playseqs[i]->length; j++)
	playseq->blocknumbers[j]=PlaySeq->seq[j];
    }
    /* Blocks */
    song->numblocks=mmd->song->numblocks;
    song->blocks=(struct block **)calloc(song->numblocks,
					 sizeof(struct block *));
    for(i=0; i<song->numblocks; i++) {
      struct block *block;
      struct MMD1Block *MMD1Block=mmd->blockarr[i];
      unsigned char *notes=(unsigned char *)MMD1Block+sizeof(struct MMD1Block);
      if(MMD1Block->info->pagetable)
	block=block_alloc(MMD1Block->numtracks, MMD1Block->lines+1,
			  MMD1Block->info->pagetable->num_pages+1);
      else
	block=block_alloc(MMD1Block->numtracks, MMD1Block->lines+1, 1);

      song->blocks[i]=block;
      for(j=0; j<block->length; j++)
	for(k=0; k<block->tracks; k++) {
	  block->notes[(j*block->tracks+k)*2]=notes[(j*block->tracks+k)*4];
	  block->notes[(j*block->tracks+k)*2+1]=notes[(j*block->tracks+k)*4+1];
	  block->effects[(j*block->tracks+k)*2]=effectconversion[notes[(j*block->tracks+k)*4+2]];
	  block->effects[(j*block->tracks+k)*2+1]=notes[(j*block->tracks+k)*4+3];
	  /* Effect values may need special handling */
	  switch(block->effects[(j*block->tracks+k)*2]) {
	  case EFFECT_VELOCITY:
	    /* Fix volume range to 0..127 */
	    if(block->effects[(j*block->tracks+k)*2+1]==64)
	      block->effects[(j*block->tracks+k)*2+1]=127;
	    else
	      block->effects[(j*block->tracks+k)*2+1]*=2;
	    break;
	  case EFFECT_NOTE_DELAY:
	    /* Command 1F in MMD is both note delay and repeat */
	    /* FIX: now only note delay (upper nibble) is supported */
	    block->effects[(j*block->tracks+k)*2+1]=block->effects[(j*block->tracks+k)*2+1]>>4;
	    break;
	  }
	}

      if(MMD1Block->info->pagetable) {
	for(j=0; j<MMD1Block->info->pagetable->num_pages; j++) {
	  for(k=0; k<(MMD1Block->lines+1); k++)
	    for(l=0; l<MMD1Block->numtracks; l++) {
	      block->effects[(j+1)*2*block->tracks*block->length+(k*block->tracks+l)*2]=effectconversion[MMD1Block->info->pagetable->page[j][(k*MMD1Block->numtracks+l)*2]];
	      block->effects[(j+1)*2*block->tracks*block->length+(k*block->tracks+l)*2+1]=MMD1Block->info->pagetable->page[j][(k*MMD1Block->numtracks+l)*2+1];

	      /* Effect values may need special handling */
	      /* Fix volume range to 0..127 */
	      switch(block->effects[(j+1)*2*block->tracks*block->length+(k*block->tracks+l)*2]) {
	      case EFFECT_VELOCITY:
		if(block->effects[(j+1)*2*block->tracks*block->length+(k*block->tracks+l)*2+1]==64)
		  block->effects[(j+1)*2*block->tracks*block->length+(k*block->tracks+l)*2+1]=127;
		else
		  block->effects[(j+1)*2*block->tracks*block->length+(k*block->tracks+l)*2+1]*=2;
		break;
	      }
	    }
	}
      }
    }
    /* Instruments */
    song->numinstruments=mmd->song->numsamples;
    song->instruments=(struct instrument **)calloc(song->numinstruments,
						   sizeof(struct instrument *));
    for(i=0; i<song->numinstruments; i++) {
      song->instruments[i]=instrument_alloc();
      if(mmd->expdata) {
	song->instruments[i]->name=(char *)strdup(mmd->expdata->iinfo[i].name);
	song->instruments[i]->hold=mmd->expdata->exp_smp[i].hold;
      }
      song->instruments[i]->midichannel=mmd->song->sample[i].midich-1;
      song->instruments[i]->midipreset=mmd->song->sample[i].midipreset;
      song->instruments[i]->transpose=mmd->song->sample[i].strans;
      if(mmd->song->sample[i].svol==64)
	song->instruments[i]->defaultvelocity=127;
      else
	song->instruments[i]->defaultvelocity=mmd->song->sample[i].svol*2;
    }

    /* Track volumes */
    song->maxtracks=mmd->song->numtracks;
    song->trackvolumes=(char *)calloc(song->maxtracks, sizeof(char));
    for(i=0; i<song->maxtracks; i++) {
      if(mmd->song->trackvols[i]<64)
	song->trackvolumes[i]=mmd->song->trackvols[i]*2;
      else
	song->trackvolumes[i]=127;
    }
    if(mmd->song->mastervol<64)
      song->mastervolume=mmd->song->mastervol*2;
    else
      song->mastervolume=127;

    /* SysEX dumps */
    if(mmd->expdata && mmd->expdata->dumps) {
      /* Is this really the only way to do this? I know, this is horrible but
       * this is what the lame MMD spec says */
      struct MMDDump **dumps=(struct MMDDump **)((char *)mmd->expdata->dumps+
						 sizeof(struct MMDDumpData));

      song->numsysexes=mmd->expdata->dumps->numdumps;
      song->sysexes=(struct sysex **)calloc(song->numsysexes,
						   sizeof(struct sysex *));
      for(i=0; i<song->numsysexes; i++) {
	song->sysexes[i]=(struct sysex *)calloc(1, sizeof(struct sysex));
	song->sysexes[i]->length=dumps[i]->length;
	song->sysexes[i]->data=(unsigned char *)calloc(song->sysexes[i]->length, sizeof(unsigned char));
	memcpy(song->sysexes[i]->data, dumps[i]->data,
	       song->sysexes[i]->length);

	if(dumps[i]->ext_len>=20)
	  song->sysexes[i]->name=strdup(dumps[i]->name);
      }
    }
  } else {
    /* Nothing can be done in this case... should never happen though */
    free(song);
    song=song_alloc();
  }

  return song;
}

/* Loads a song from an XML file */
struct song *song_load(char *filename) {
  xmlDocPtr doc;
  xmlNsPtr ns;
  xmlNodePtr cur;
  
  /* Build an XML tree from a the file */
  if(!(doc=xmlParseFile(filename))) {
    /* If it didn't succeed try loading it as an MMD file */
    struct MMD2 *mmd;
    struct song *song;
    mmd=MMD2_load(filename);
    song=song_convert_MMD2(mmd);
    MMD2_free(mmd);
    /* If it didn't succeed either just allocate a new song */
    if(song!=NULL)
      return song;
    else
      return song_alloc();
  }

  /* Go ahead and parse the document */
  cur=xmlDocGetRootElement(doc);
  return song_parse(doc, NULL, cur);
}

/* Parses a song element in an XML file */
struct song *song_parse(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur) {
  struct song *song=NULL;
  
  if((!xmlStrcmp(cur->name, "song")) && (cur->ns==ns)) {
    char *prop;
    /* Allocate song */
    song=(struct song *)calloc(1, sizeof(struct song));
    song->name=xmlGetProp(cur, "name");
    prop=xmlGetProp(cur, "tempo");
    if(prop!=NULL)
      song->tempo=atoi(prop);
    prop=xmlGetProp(cur, "ticksperline");
    if(prop!=NULL)
      song->ticksperline=atoi(prop);
    prop=xmlGetProp(cur, "mastervolume");
    if(prop!=NULL)
      song->mastervolume=atoi(prop);

    cur=cur->xmlChildrenNode;
    while(cur!=NULL) {
      if((!xmlStrcmp(cur->name, "blocks")) && (cur->ns==ns)) {
	xmlNodePtr temp;

	/* Parse and add all block elements */
	temp=cur->xmlChildrenNode;
	while(temp!=NULL) {
	  if(!(xmlIsBlankNode(temp))) {
	    struct block *block=block_parse(doc, ns, temp);
	    if(block!=NULL) {
	      song->numblocks++;
	      song->blocks=realloc(song->blocks,
				   song->numblocks*sizeof(struct block *));
	      song->blocks[song->numblocks-1]=block;
	    }
	  }
	  
	  temp=temp->next;
	}
      } else if((!xmlStrcmp(cur->name, "sections")) && (cur->ns==ns)) {
	xmlNodePtr temp;

	/* Parse and add all section elements */
	temp=cur->xmlChildrenNode;
	while(temp!=NULL) {
	  if(!(xmlIsBlankNode(temp))) {
	    prop=xmlGetProp(temp, "playseq");
	    if(prop!=NULL) {
	      song->numsections++;
	      song->sections=realloc(song->sections, song->numsections*sizeof(unsigned short));
	      song->sections[song->numsections-1]=atoi(prop);
	    }
	  }
	  
	  temp=temp->next;
	}
      } else if((!xmlStrcmp(cur->name, "playseqs")) && (cur->ns==ns)) {
	xmlNodePtr temp;

	/* Parse and add all playseq elements */
	temp=cur->xmlChildrenNode;
	while(temp!=NULL) {
	  if(!(xmlIsBlankNode(temp))) {
	    struct playseq *playseq=playseq_parse(doc, ns, temp);
	    if(playseq!=NULL) {
	      song->numplayseqs++;
	      song->playseqs=realloc(song->playseqs, song->numplayseqs*sizeof(struct playseq *));
	      song->playseqs[song->numplayseqs-1]=playseq;
	    }
	  }
	  
	  temp=temp->next;
	}
      } else if((!xmlStrcmp(cur->name, "instruments")) && (cur->ns==ns)) {
	xmlNodePtr temp;

	/* Parse and add all instrument elements */
	temp=cur->xmlChildrenNode;
	while(temp!=NULL) {
	  if(!(xmlIsBlankNode(temp))) {
	    struct instrument *instrument=instrument_parse(doc, ns, temp);
	    if(instrument!=NULL) {
	      song->numinstruments++;
	      song->instruments=realloc(song->instruments, song->numinstruments*sizeof(struct instrument *));
	      song->instruments[song->numinstruments-1]=instrument;
	    }
	  }
	  
	  temp=temp->next;
	}
      } else if((!xmlStrcmp(cur->name, "trackvolumes")) && (cur->ns==ns)) {
	xmlNodePtr temp;

	/* Parse and add all track volume elements */
	temp=cur->xmlChildrenNode;
	while(temp!=NULL) {
	  if(!(xmlIsBlankNode(temp))) {
	    prop=xmlGetProp(temp, "volume");
	    if(prop!=NULL) {
	      song->maxtracks++;
	      song->trackvolumes=realloc(song->trackvolumes,
				     song->maxtracks*sizeof(unsigned char));
	      song->trackvolumes[song->maxtracks-1]=atoi(prop);
	    }
	  }
	  
	  temp=temp->next;
	}
      } else if((!xmlStrcmp(cur->name, "sysexes")) && (cur->ns==ns)) {
	xmlNodePtr temp;

	/* Parse and add all SysEx elements */
	temp=cur->xmlChildrenNode;
	while(temp!=NULL) {
	  if(!(xmlIsBlankNode(temp))) {
	    struct sysex *sysex=sysex_parse(doc, ns, temp);
	    if(sysex!=NULL) {
	      song->numsysexes++;
	      song->sysexes=realloc(song->sysexes, song->numsysexes*sizeof(struct sysex *));
	      song->sysexes[song->numsysexes-1]=sysex;
	    }
	  }
	  
	  temp=temp->next;
	}
      }
      cur=cur->next;
    }
  } else
    fprintf(stderr, "XML error: expected song, got %s\n", cur->name);

  return song;
}

/* Saves a song to an XML file */
void song_save(struct song *song, char *filename) {
  xmlDocPtr doc;
  xmlNodePtr node, subnode, lb;
  int i;
  char c[10];

  doc=xmlNewDoc("1.0");
  doc->xmlRootNode=xmlNewDocNode(doc, NULL, "song", NULL);
  xmlSetProp(doc->xmlRootNode, "name", song->name);
  snprintf(c, 10, "%d", song->tempo);
  xmlSetProp(doc->xmlRootNode, "tempo", c);
  snprintf(c, 10, "%d", song->ticksperline);
  xmlSetProp(doc->xmlRootNode, "ticksperline", c);
  snprintf(c, 10, "%d", song->mastervolume);
  xmlSetProp(doc->xmlRootNode, "mastervolume", c);
  lb=xmlNewText("\n\n");
  xmlAddChild(doc->xmlRootNode, lb);

  node=xmlNewChild(doc->xmlRootNode, NULL, "blocks", NULL);
  lb=xmlNewText("\n");
  xmlAddChild(node, lb);
  /* Add all blocks */
  for(i=0; i<song->numblocks; i++)
    block_save(song->blocks[i], node);
  lb=xmlNewText("\n\n");
  xmlAddChild(doc->xmlRootNode, lb);

  node=xmlNewChild(doc->xmlRootNode, NULL, "sections", NULL);
  lb=xmlNewText("\n");
  xmlAddChild(node, lb);
  /* Add all sections */
  for(i=0; i<song->numsections; i++) {
    subnode=xmlNewChild(node, NULL, "section", NULL);
    snprintf(c, 10, "%d", song->sections[i]);
    xmlSetProp(subnode, "playseq", c);
    lb=xmlNewText("\n");
    xmlAddChild(node, lb);
  }
  lb=xmlNewText("\n\n");
  xmlAddChild(doc->xmlRootNode, lb);

  node=xmlNewChild(doc->xmlRootNode, NULL, "playseqs", NULL);
  lb=xmlNewText("\n");
  xmlAddChild(node, lb);
  /* Add all playing sequences */
  for(i=0; i<song->numplayseqs; i++)
    playseq_save(song->playseqs[i], node);
  lb=xmlNewText("\n\n");
  xmlAddChild(doc->xmlRootNode, lb);

  node=xmlNewChild(doc->xmlRootNode, NULL, "instruments", NULL);
  lb=xmlNewText("\n");
  xmlAddChild(node, lb);
  /* Add all instruments */
  for(i=0; i<song->numinstruments; i++)
    instrument_save(song->instruments[i], node);
  lb=xmlNewText("\n\n");
  xmlAddChild(doc->xmlRootNode, lb);

  node=xmlNewChild(doc->xmlRootNode, NULL, "trackvolumes", NULL);
  lb=xmlNewText("\n");
  xmlAddChild(node, lb);
  /* Add all track volumes */
  for(i=0; i<song->maxtracks; i++) {
    subnode=xmlNewChild(node, NULL, "track", NULL);
    snprintf(c, 10, "%d", song->trackvolumes[i]);
    xmlSetProp(subnode, "volume", c);
    lb=xmlNewText("\n");
    xmlAddChild(node, lb);
  }
  lb=xmlNewText("\n\n");
  xmlAddChild(doc->xmlRootNode, lb);

  node=xmlNewChild(doc->xmlRootNode, NULL, "sysexes", NULL);
  lb=xmlNewText("\n");
  xmlAddChild(node, lb);
  /* Add all SysEx messages */
  for(i=0; i<song->numsysexes; i++)
    sysex_save(song->sysexes[i], node);
  lb=xmlNewText("\n\n");
  xmlAddChild(doc->xmlRootNode, lb);

  /* Use stdout if no filename is specified */
  if(filename==NULL)
    filename="-";

  xmlSaveFile(filename, doc);
  xmlFreeDoc(doc);
}
