/* editor.c
 *
 * Copyright 2002 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <string.h>
#include <pthread.h>
#include "song.h"
#include "gui.h"
#include "editor.h"
#include "player.h"

struct editor editor;

/* Initializes the editor and either opens an existing song or creates new */
void editor_open(char *filename) {
  int i;

  /* Free all resources if already allocated */
  editor_close();

  /* Make sure everything is reset */
  memset(&editor, 0, sizeof(struct editor));

  /* Open MIDI device */
  editor.midi=fopen("/dev/midi", "r+");
  if(editor.midi==NULL)
    fprintf(stderr, "Could not open /dev/midi. MIDI disabled.\n");

  /* If a filename was supplied open the file */
  if(filename!=NULL)
    editor.song=song_load(filename);
  else
    editor.song=song_alloc();

  /* Initialize notes playing array */
  editor.trackstatus=(char *)calloc(4*editor.song->maxtracks, 1);
  for(i=0; i<editor.song->maxtracks; i++) {
    editor.trackstatus[i*4]=-1;
    editor.trackstatus[i*4+1]=-1;
    editor.trackstatus[i*4+2]=-1;
    editor.trackstatus[i*4+3]=-1;
  }
  editor.space=1;

  /* Player thread is not active */
  editor.threadstatus=-1;

  /* Refresh GUI */
  gui_main_refresh();
  gui_blocklist_refresh();
  gui_playseq_refresh();
  gui_sectionlist_refresh();
  gui_instrument_refresh();
  gui_trackvolumes_refresh();
}

/* Deinitializes the editor */
void editor_close() {
  /* Stop the player thread if active */
  player_stop();

  /* If notes are playing stop them; free the track status array */
  if(editor.trackstatus!=NULL) {
    player_stop_notes();
    free(editor.trackstatus);
    editor.trackstatus=NULL;
  }

  /* Free the song */
  if(editor.song!=NULL) {
    song_free(editor.song);
    editor.song=NULL;
  }

  /* Close MIDI device */
  if(editor.midi!=NULL) {
    fclose(editor.midi);
    editor.midi=NULL;
  }
}
