/* block.c
 *
 * Copyright 2002 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include "block.h"

struct block *block_alloc(int tracks, int length, int effectpages) {
  struct block *block;

  block=(struct block *)calloc(1, sizeof(struct block));
  block->tracks=tracks;
  block->length=length;
  block->notes=(unsigned char *)calloc(2*tracks*length, sizeof(unsigned char));
  block->effectpages=effectpages;
  block->effects=(unsigned char *)calloc(effectpages*2*tracks*length,
					 sizeof(unsigned char));

  return block;
}

void block_free(struct block *block) {
  if(block) {
    if(block->name)
      free(block->name);
    
    if(block->notes)
      free(block->notes);
    
    if(block->effects)
      free(block->effects);
    
    free(block);
  }
}

void block_set_note(struct block *block, unsigned short line,
		    unsigned short track, unsigned char octave,
		    unsigned char note, unsigned char instrument) {
  if(note) {
    block->notes[2*(block->tracks*line+track)]=octave*12+note;
    block->notes[2*(block->tracks*line+track)+1]=instrument;
  } else {
    block->notes[2*(block->tracks*line+track)]=0;
    block->notes[2*(block->tracks*line+track)+1]=0;
  }
}

void block_set_instrument(struct block *block, unsigned short line,
			  unsigned short track, unsigned char instrument) {
  block->notes[2*(block->tracks*line+track)+1]=instrument;
}

void block_set_effect(struct block *block, unsigned short line,
		      unsigned short track, unsigned short effectpage,
		      unsigned char slot, unsigned char data) {
  if(slot&1) {
    block->effects[effectpage*2*block->tracks*block->length+2*(block->tracks*line+track)+slot/2]&=0xf0;
    block->effects[effectpage*2*block->tracks*block->length+2*(block->tracks*line+track)+slot/2]|=data;
  } else {
    block->effects[effectpage*2*block->tracks*block->length+2*(block->tracks*line+track)+slot/2]&=0x0f;
    block->effects[effectpage*2*block->tracks*block->length+2*(block->tracks*line+track)+slot/2]|=(data<<4);
  }
}

void block_set_effect_full(struct block *block, unsigned short line,
			   unsigned short track, unsigned short effectpage,
			   unsigned char effect, unsigned char data) {
  block->effects[effectpage*2*block->tracks*block->length+2*(block->tracks*line+track)]=effect;
  block->effects[effectpage*2*block->tracks*block->length+2*(block->tracks*line+track)+1]=data;
}

void block_set_tracks(struct block *block, unsigned short tracks) {
  if(block) {
    unsigned char *notes=(unsigned char *)calloc(2*tracks*block->length,
						 sizeof(unsigned char));
    unsigned char *effects=(unsigned char *)calloc(block->effectpages*2*tracks*block->length,
						 sizeof(unsigned char));
    int i, j, k;
    unsigned short existingtracks;

    if(tracks<block->tracks)
      existingtracks=tracks;
    else
      existingtracks=block->tracks;

    for(i=0; i<block->length; i++)
      for(j=0; j<existingtracks; j++) {
	notes[(i*tracks+j)*2]=block->notes[(i*block->tracks+j)*2];
	notes[(i*tracks+j)*2+1]=block->notes[(i*block->tracks+j)*2+1];
	for(k=0; k<block->effectpages; k++) {
	  effects[k*2*block->tracks*block->length+(i*tracks+j)*2]=block->effects[k*2*existingtracks*block->length+(i*block->tracks+j)*2];
	  effects[k*2*block->tracks*block->length+(i*tracks+j)*2+1]=block->effects[k*2*existingtracks*block->length+(i*block->tracks+j)*2+1];
	}
      }

    if(block->notes)
      free(block->notes);
    if(block->effects)
      free(block->effects);

    block->notes=notes;
    block->effects=effects;
    block->tracks=tracks;
  }
}

void block_set_length(struct block *block, unsigned short length) {
  if(block) {
    unsigned char *notes=(unsigned char *)calloc(2*block->tracks*length,
						 sizeof(unsigned char));
    unsigned char *effects=(unsigned char *)calloc(block->effectpages*2*block->tracks*length,
						 sizeof(unsigned char));
    int i, j, k;

    unsigned short existinglength;

    if(length<block->length)
      existinglength=length;
    else
      existinglength=block->length;

    for(i=0; i<existinglength; i++)
      for(j=0; j<block->tracks; j++) {
	notes[(i*block->tracks+j)*2]=block->notes[(i*block->tracks+j)*2];
	notes[(i*block->tracks+j)*2+1]=block->notes[(i*block->tracks+j)*2+1];
	for(k=0; k<block->effectpages; k++) {
	  effects[k*2*block->tracks*block->length+(i*block->tracks+j)*2]=block->effects[k*2*block->tracks*existinglength+(i*block->tracks+j)*2];
	  effects[k*2*block->tracks*block->length+(i*block->tracks+j)*2+1]=block->effects[k*2*block->tracks*existinglength+(i*block->tracks+j)*2+1];
	}
      }

    if(block->notes)
      free(block->notes);
    if(block->effects)
      free(block->effects);

    block->notes=notes;
    block->effects=effects;
    block->length=length;
  }
}

void block_set_effectpages(struct block *block, unsigned short effectpages) {
  if(block) {
    unsigned char *effects=(unsigned char *)calloc(effectpages*2*block->tracks*block->length,
						 sizeof(unsigned char));
    unsigned short existingeffectpages;

    if(effectpages<block->effectpages)
      existingeffectpages=effectpages;
    else
      existingeffectpages=block->effectpages;

    memcpy(effects, block->effects,
	   existingeffectpages*2*block->tracks*block->length);

    if(block->effects)
      free(block->effects);

    block->effects=effects;
    block->effectpages=effectpages;
  }
}

/* Parses a block element in an XML file */
struct block *block_parse(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur) {
  struct block *block=NULL;
  xmlNodePtr temp;
  
  if((!xmlStrcmp(cur->name, "block")) && (cur->ns==ns)) {
    char *prop;
    int tracks, length, effectpages;

    /* Get block properties */
    prop=xmlGetProp(cur, "tracks");
    if(prop!=NULL)
      tracks=atoi(prop);
    prop=xmlGetProp(cur, "length");
    if(prop!=NULL)
      length=atoi(prop);
    prop=xmlGetProp(cur, "effectpages");
    if(prop!=NULL)
      effectpages=atoi(prop);

    /* Allocate block */
    block=block_alloc(tracks, length, effectpages);
    block->name=xmlGetProp(cur, "name");

    /* Get block contents */
    cur=cur->xmlChildrenNode;
    while(cur!=NULL) {
      if((!xmlStrcmp(cur->name, "note")) && (cur->ns==ns)) {
	int line, track, note, instrument;

	/* Get note properties */
	prop=xmlGetProp(cur, "line");
	if(prop!=NULL)
	  line=atoi(prop);
	prop=xmlGetProp(cur, "track");
	if(prop!=NULL)
	  track=atoi(prop);
	prop=xmlGetProp(cur, "note");
	if(prop!=NULL)
	  note=atoi(prop);
	prop=xmlGetProp(cur, "instrument");
	if(prop!=NULL)
	  instrument=atoi(prop);

	/* Set the note */
	block_set_note(block, line, track, 0, note, instrument);
      } else if((!xmlStrcmp(cur->name, "effect")) && (cur->ns==ns)) {
	int line, track, effectpage, effect, value;

	/* Get effect properties */
	prop=xmlGetProp(cur, "line");
	if(prop!=NULL)
	  line=atoi(prop);
	prop=xmlGetProp(cur, "track");
	if(prop!=NULL)
	  track=atoi(prop);
	prop=xmlGetProp(cur, "effectpage");
	if(prop!=NULL)
	  effectpage=atoi(prop);
	prop=xmlGetProp(cur, "effect");
	if(prop!=NULL)
	  effect=atoi(prop);
	prop=xmlGetProp(cur, "value");
	if(prop!=NULL)
	  value=atoi(prop);

	/* Set the effect */
	block_set_effect_full(block, line, track, effectpage, effect, value);
      }
      cur=cur->next;
    }
  } else
    fprintf(stderr, "XML error: expected block, got %s\n", cur->name);

  return block;
}

/* Saves a block to an XML document */
void block_save(struct block *block, xmlNodePtr parent) {
  char c[10];
  int i, j, k;
  xmlNodePtr node, subnode, lb;

  node=xmlNewChild(parent, NULL, "block", NULL);
  xmlSetProp(node, "name", block->name);
  snprintf(c, 10, "%d", block->tracks);
  xmlSetProp(node, "tracks", c);
  snprintf(c, 10, "%d", block->length);
  xmlSetProp(node, "length", c);
  snprintf(c, 10, "%d", block->effectpages);
  xmlSetProp(node, "effectpages", c);
  lb=xmlNewText("\n");
  xmlAddChild(node, lb);

  /* Notation data */
  for(i=0; i<block->tracks; i++)
    for(j=0; j<block->length; j++)
      if(block->notes[2*(block->tracks*j+i)]!=0 ||
	 block->notes[2*(block->tracks*j+i)+1]!=0) {
	subnode=xmlNewChild(node, NULL, "note", NULL);
	snprintf(c, 10, "%d", j);
	xmlSetProp(subnode, "line", c);
	snprintf(c, 10, "%d", i);
	xmlSetProp(subnode, "track", c);
	snprintf(c, 10, "%d", block->notes[2*(block->tracks*j+i)]);
	xmlSetProp(subnode, "note", c);
	snprintf(c, 10, "%d", block->notes[2*(block->tracks*j+i)+1]);
	xmlSetProp(subnode, "instrument", c);
	lb=xmlNewText("\n");
	xmlAddChild(node, lb);
      }

  /* Effect data */
  for(i=0; i<block->tracks; i++)
    for(j=0; j<block->length; j++)
      for(k=0; k<block->effectpages; k++)
	if(block->effects[k*2*block->tracks*block->length+2*(block->tracks*j+i)]!=0 ||
	   block->effects[k*2*block->tracks*block->length+2*(block->tracks*j+i)+1]!=0) {
	  subnode=xmlNewChild(node, NULL, "effect", NULL);
	  snprintf(c, 10, "%d", j);
	  xmlSetProp(subnode, "line", c);
	  snprintf(c, 10, "%d", i);
	  xmlSetProp(subnode, "track", c);
	  snprintf(c, 10, "%d", k);
	  xmlSetProp(subnode, "effectpage", c);
	  snprintf(c, 10, "%d", block->effects[k*2*block->tracks*block->length+2*(block->tracks*j+i)]);
	  xmlSetProp(subnode, "effect", c);
	  snprintf(c, 10, "%d", block->effects[k*2*block->tracks*block->length+2*(block->tracks*j+i)+1]);
	  xmlSetProp(subnode, "value", c);
	  lb=xmlNewText("\n");
	  xmlAddChild(node, lb);
	}
  
  lb=xmlNewText("\n");
  xmlAddChild(parent, lb);
}
