/* playseq.c
 *
 * Copyright 2002 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include "playseq.h"

struct playseq *playseq_alloc() {
  struct playseq *playseq;
  
  playseq=(struct playseq *)calloc(1, sizeof(struct playseq));
  playseq->length=1;
  playseq->blocknumbers=(unsigned short *)calloc(1, sizeof(unsigned short));

  return playseq;
}

void playseq_free(struct playseq *playseq) {
  if(playseq) {
    if(playseq->name)
      free(playseq->name);
    
   if(playseq->blocknumbers)
      free(playseq->blocknumbers);
    
    free(playseq);
  }
}

/* Inserts a new block in the block array in the given position */
void playseq_insert(struct playseq *playseq, int pos) {
  int i;
  unsigned short old;

  /* Do not allow inserting beyond the block array size */
  if(pos>playseq->length)
    pos=playseq->length;

  /* Which block number to insert */
  if(pos<playseq->length)
    old=playseq->blocknumbers[pos];
  else
    old=playseq->blocknumbers[playseq->length-1];

  playseq->length++;
  /* Reallocate the block array */
  playseq->blocknumbers=(unsigned short *)realloc(playseq->blocknumbers,
						  playseq->length*sizeof(unsigned short));

  /* Move rest of the sections onwards */
  for(i=playseq->length-1; i>pos; i--)
    playseq->blocknumbers[i]=playseq->blocknumbers[i-1];

  playseq->blocknumbers[pos]=old;
}

/* Deletes a section from the given position of the section array */
void playseq_delete(struct playseq *playseq, int pos) {
  int i;
  
  /* Don't delete the last section */
  if(playseq->length>1) {
    playseq->length--;
    
    /* Move rest of the blocks backwards */
    for(i=pos; i<playseq->length; i++)
      playseq->blocknumbers[i]=playseq->blocknumbers[i+1];
    
    /* Reallocate the block array */
    playseq->blocknumbers=(unsigned short *)realloc(playseq->blocknumbers, playseq->length*sizeof(unsigned short));

    refresh_playseq_and_block();
  }
}

void playseq_set(struct playseq *playseq, unsigned short pos, unsigned short block) {
  playseq->blocknumbers[pos]=block;
}

/* Parses a playseq element in an XML file */
struct playseq *playseq_parse(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur) {
  struct playseq *playseq=NULL;
  xmlNodePtr temp;
  
  if((!xmlStrcmp(cur->name, "playseq")) && (cur->ns==ns)) {
    char *prop;

    /* Allocate playseq */
    playseq=(struct playseq *)calloc(1, sizeof(struct playseq));
    playseq->name=xmlGetProp(cur, "name");

    /* Get playseq contents */
    cur=cur->xmlChildrenNode;
    while(cur!=NULL) {
      if((!xmlStrcmp(cur->name, "playseqentry")) && (cur->ns==ns)) {
	int block;

	/* Get block number */
	prop=xmlGetProp(cur, "block");
	if(prop!=NULL)
	  block=atoi(prop);

	playseq->length++;
	playseq->blocknumbers=realloc(playseq->blocknumbers,
				      playseq->length*sizeof(unsigned short));
	playseq->blocknumbers[playseq->length-1]=block;
      }
      cur=cur->next;
    }
  } else
    fprintf(stderr, "XML error: expected playseq, got %s\n", cur->name);

  return playseq;
}

/* Saves a playing sequence to an XML document */
void playseq_save(struct playseq *playseq, xmlNodePtr parent) {
  char c[10];
  int i;
  xmlNodePtr node, subnode, lb;

  node=xmlNewChild(parent, NULL, "playseq", NULL);
  xmlSetProp(node, "name", playseq->name);
  lb=xmlNewText("\n");
  xmlAddChild(node, lb);
  /* Add all blocks */
  for(i=0; i<playseq->length; i++) {
    subnode=xmlNewChild(node, NULL, "playseqentry", NULL);
    snprintf(c, 10, "%d", playseq->blocknumbers[i]);
    xmlSetProp(subnode, "block", c);
  }
  lb=xmlNewText("\n");
  xmlAddChild(node, lb);
  
  lb=xmlNewText("\n");
  xmlAddChild(parent, lb);
}
