/* player.c
 *
 * Copyright 2002 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gtk/gtk.h>
#include <stdio.h>
#include <signal.h>
#include <sys/time.h>
#include <unistd.h>
#include "song.h"
#include "editor.h"
#include "trackerwidget.h"
#include "midi.h"
#include "player.h"

extern struct editor editor;
extern GtkWidget *gui_main_tracker;

struct itimerval play_timer;

unsigned short tick;

/* Plays the song from the beginning */
void play_song() {
  editor.mode=MODE_PLAY_SONG;
  editor.section=0;
  editor.playseqpos=0;
  refresh_playseq_and_block();
  gettimeofday(&editor.playingstarted, NULL);
  editor.playedsofar.tv_sec=0;
  editor.playedsofar.tv_usec=0;

  play_set_timer(editor.song->tempo);
  setitimer(ITIMER_REAL, &play_timer, NULL);
  signal(SIGALRM, play_handler);

  stop_notes();
}

/* Continues playing the song from the current position */
void continue_song() {
  if(editor.mode!=MODE_PLAY_SONG) {
    editor.mode=MODE_PLAY_SONG;
    
    gettimeofday(&editor.playingstarted, NULL);
    
    play_set_timer(editor.song->tempo);
    setitimer(ITIMER_REAL, &play_timer, NULL);
    signal(SIGALRM, play_handler);
    
    stop_notes();
  }
}

/* Plays the block from the beginning */
void play_block() {
  editor.mode=MODE_PLAY_BLOCK;

  gettimeofday(&editor.playingstarted, NULL);
  editor.playedsofar.tv_sec=0;
  editor.playedsofar.tv_usec=0;

  play_set_timer(editor.song->tempo);
  setitimer(ITIMER_REAL, &play_timer, NULL);
  signal(SIGALRM, play_handler);

  stop_notes();
}

/* Continues playing the block from the current position */
void continue_block() {
  if(editor.mode!=MODE_PLAY_BLOCK) {
    editor.mode=MODE_PLAY_BLOCK;
    
    gettimeofday(&editor.playingstarted, NULL);
    
    play_set_timer(editor.song->tempo);
    setitimer(ITIMER_REAL, &play_timer, NULL);
    signal(SIGALRM, play_handler);
    
    stop_notes();
  }
}

/* Stops the player */
void stop() {
  struct timeval tod;

  if(editor.mode!=MODE_IDLE) {
    editor.mode=MODE_IDLE;
    
    gettimeofday(&tod, NULL);
    editor.playedsofar.tv_sec+=tod.tv_sec-editor.playingstarted.tv_sec;
    editor.playedsofar.tv_usec+=tod.tv_usec-editor.playingstarted.tv_usec;
    if(editor.playedsofar.tv_usec<0) {
      editor.playedsofar.tv_usec+=1000000;
      editor.playedsofar.tv_sec--;
    }
    
    setitimer(ITIMER_REAL, NULL, NULL);
    signal(SIGALRM, SIG_DFL);
  } 
  stop_notes();
}

/* SIGALRT signal handler for triggering the playroutine */
void play_handler(int signum) {
  Tracker *tracker=(Tracker *)gui_main_tracker;
  int i, j;
  struct timeval tod;
  struct block *block=editor.song->blocks[editor.block];
  unsigned char posteffect, postvalue;

  /* Play notes scheduled to be played */
  for(i=0; i<block->tracks; i++) {
    unsigned char volume=127;
    unsigned char notedelay=0;
    /* Handle effects meant to be processed only at first tick */
    for(j=0; j<block->effectpages; j++) {
      unsigned char effect=block->effects[j*block->tracks*block->length*2+(block->tracks*tracker->patpos+i)*2];
      unsigned char value=block->effects[j*block->tracks*block->length*2+(block->tracks*tracker->patpos+i)*2+1];
      switch(effect) {
      case EFFECT_END_BLOCK:
	/* Only on last tick */
	if(tick==editor.song->ticksperline-1) {
	  posteffect=EFFECT_END_BLOCK;
	  postvalue=value;
	}
	break;
      case EFFECT_PLAYSEQ_POSITION:
	/* Only on last tick */
	if(tick==editor.song->ticksperline-1) {
	  posteffect=EFFECT_PLAYSEQ_POSITION;
	  postvalue=value;
	}
	break;
      case EFFECT_STOP:
	/* Only on last tick */
	if(tick==editor.song->ticksperline-1) {
	  stop();
	  gui_info_refresh();
	}
	break;
      case EFFECT_VELOCITY:
	volume=value;
	break;
      case EFFECT_NOTE_DELAY:
	notedelay=value;
	break;
      case EFFECT_TPL:
	song_settpl(editor.song, value);
	gui_info_refresh();
	break;
      case EFFECT_TEMPO:
	song_settempo(editor.song, value);
	gui_info_refresh();
	break;
      }
    }
    
    if(block->notes[(block->tracks*tracker->patpos+i)*2]!=0 &&
       tick==notedelay) {
      unsigned char hold=0;
      unsigned char note=block->notes[(block->tracks*tracker->patpos+i)*2]-1;
      unsigned short instrument=block->notes[(block->tracks*tracker->patpos+i)*2+1]-1;
      
      play_note(instrument, note, volume, i);
      
      if(instrument<editor.song->numinstruments &&
	 editor.song->instruments[instrument]!=NULL)
	hold=editor.song->instruments[block->notes[(block->tracks*tracker->patpos+i)*2+1]-1]->hold;
      
      if(hold==0)
	editor.trackstatus[i*4+3]=-1;
      else
	editor.trackstatus[i*4+3]=hold;
    }
  }

  /* Decrement hold times of notes and stop notes that should be stopped */
  for(i=0; i<editor.song->maxtracks; i++) {
    if(editor.trackstatus[i*4+3]!=-1) {
      editor.trackstatus[i*4+3]--;
      if(editor.trackstatus[i*4+3]==-1) {
	midi_note_off(editor.trackstatus[i*4], editor.trackstatus[i*4+1],
		      127);
	editor.trackstatus[i*4+1]=-1;
      }
    }
  }

  /* Next tick */
  tick++;
  tick%=editor.song->ticksperline;

  switch(posteffect) {
  case EFFECT_END_BLOCK:
    play_playseq_advance();
    refresh_playseq_and_block();
    tracker_set_pattern(tracker, editor.song->blocks[editor.block]);
    tracker_set_patpos(tracker, 0);
    gui_info_refresh();
    gui_blocklist_refresh();
    gui_playseq_refresh();
    break;
  case EFFECT_PLAYSEQ_POSITION:
    editor.playseqpos=postvalue;
    if(editor.playseqpos>=editor.song->playseqs[editor.playseq]->length){
      editor.playseqpos=0;
      play_section_advance();
    }
    refresh_playseq_and_block();
    tracker_set_pattern(tracker, editor.song->blocks[editor.block]);
    tracker_set_patpos(tracker, 0);
    gui_info_refresh();
    gui_blocklist_refresh();
    gui_playseq_refresh();
    break;
  default:
    /* Advance in block if ticksperline ticks have passed */
    if(tick==0) {
      int newpos=tracker->patpos+1;
      if(newpos>=tracker->curpattern->length) {
	newpos=0;
	if(editor.mode==MODE_PLAY_SONG)
	  play_playseq_advance();
	refresh_playseq_and_block();
	tracker_set_pattern(tracker, editor.song->blocks[editor.block]);
	gui_info_refresh();
	gui_blocklist_refresh();
	gui_playseq_refresh();
      }
      tracker_set_patpos(tracker, newpos);
      
      gettimeofday(&tod, NULL);
      gui_timer_refresh((int)(editor.playedsofar.tv_sec+tod.tv_sec-editor.playingstarted.tv_sec));
    }
    break;
  }
}

void play_playseq_advance() {
  editor.playseqpos++;
  if(editor.playseqpos>=editor.song->playseqs[editor.playseq]->length) {
    editor.playseqpos=0;
    play_section_advance();
  }
}

void play_section_advance() {
  editor.section++;
  if(editor.section>=editor.song->numsections)
    editor.section=0;
}

/* Sets the timer structure according to a given BPM count */
void play_set_timer(unsigned short tempo) {
  /* Tick every 1000000/((BPM/60)*24) usecs */
  play_timer.it_interval.tv_sec=0;
  play_timer.it_interval.tv_usec=2500000/tempo;
  play_timer.it_value.tv_sec=0;
  play_timer.it_value.tv_usec=2500000/tempo;
  tick=0;
}

/* Plays a note using given instrument on a given track */
void play_note(unsigned short instrument, unsigned char note,
	       unsigned char volume, unsigned char track) {
  /* Stop currently playing note */
  if(editor.trackstatus[track*4+1]!=-1)
    midi_note_off(editor.trackstatus[track*4], editor.trackstatus[track*4+1],
		  127);

  /* Don't play a note if the instrument does not exist */
  if(instrument<editor.song->numinstruments &&
     editor.song->instruments[instrument]!=NULL) {
    /* Apply note transpose */
    note+=editor.song->instruments[instrument]->transpose;

    /* Set new track status (channel, note, volume, time) */
    editor.trackstatus[track*4]=editor.song->instruments[instrument]->midichannel;
    editor.trackstatus[track*4+1]=note;
    editor.trackstatus[track*4+2]=editor.song->instruments[instrument]->defaultvelocity*volume/127;
    if(editor.song->instruments[instrument]->hold>0)
      editor.trackstatus[track*4+3]=editor.song->instruments[instrument]->hold;
    else
      editor.trackstatus[track*4+3]=-1;

    /* Make sure the volume isn't too large */
    if(editor.trackstatus[track*4+2]<0)
      editor.trackstatus[track*4+2]=127;
    
    /* Play note */
    midi_note_on(editor.trackstatus[track*4], editor.trackstatus[track*4+1],
		 editor.trackstatus[track*4+2]);
  }
}

/* Stops notes playing at the moment */
void stop_notes() {
  int i;
  for(i=0; i<editor.song->maxtracks; i++) {
    if(editor.trackstatus[i*4+1]!=-1) {
      midi_note_off(editor.trackstatus[i*4], editor.trackstatus[i*4+1], 127);
      editor.trackstatus[i*4]=-1;
      editor.trackstatus[i*4+1]=-1;
      editor.trackstatus[i*4+2]=-1;
      editor.trackstatus[i*4+3]=-1;
    }
  }
}

/* Stops all notes playing at the moment */
void stop_all_notes() {
  int i, j;
  for(i=0; i<16; i++)
    for(j=0; j<128; j++)
      midi_note_off(i, j, 127);
}

/* Refreshes editor.playseq from editor.section and
 * editor.block from editor.playseqpos */
void refresh_playseq_and_block() {
  if(editor.section>=editor.song->numsections)
    editor.section=0;
  editor.playseq=editor.song->sections[editor.section];

  if(editor.playseqpos>=editor.song->playseqs[editor.playseq]->length) {
    editor.playseqpos=0;
    refresh_playseq_and_block();
  } else
    editor.block=editor.song->playseqs[editor.playseq]->blocknumbers[editor.playseqpos];
}
