/* keyboard.c
 *
 * Copyright 2002-2011 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <gnome.h>
#include <glade/glade.h>
#include "tutka.h"
#include "gui.h"
#include "callbacks.h"
#include "editor.h"
#include "player.h"
#include "trackerwidget.h"

static guint *keyboard_keys_down = NULL;
static guint keyboard_num_keys_down = 0;

gboolean key_press_event(GtkWidget * widget, GdkEventKey * event, gpointer user_data)
{
  return keyboard_event(widget, event, TRUE, (struct gui *)user_data);
}

gboolean key_release_event(GtkWidget * widget, GdkEventKey * event, gpointer user_data)
{
  return keyboard_event(widget, event, FALSE, (struct gui *)user_data);
}

gboolean keyboard_event(GtkWidget * widget, GdkEventKey * event, gboolean pressed, struct gui *gui)
{
  Tracker *tracker;
  int shift, ctrl, alt;
  struct editor *editor = gui->editor;
  gboolean handled = FALSE;

  if (GTK_IS_ENTRY(GTK_WINDOW(widget)->focus_widget))
    return handled;

  tracker = (Tracker *)glade_xml_get_widget(gui->xml, "tracker");
  shift = event->state & GDK_SHIFT_MASK;
  ctrl = event->state & GDK_CONTROL_MASK;
  alt = event->state & GDK_MOD1_MASK;

  if (pressed) {
    if (ctrl) {
      switch (event->keyval) {
      case 'b':
        /* CTRL-B: Selection mode on/off */
        if (tracker_is_in_selection_mode(tracker))
          tracker_mark_selection(tracker, FALSE);
        else
          tracker_mark_selection(tracker, TRUE);
        handled = TRUE;
        break;
      case 'k':
        /* CTRL-K: Clear until the end of the track */
        editor_song_block_current_clear(editor, tracker->cursor_ch, tracker->patpos, tracker->cursor_ch, editor_song_block_current_length_get(editor) - 1);
        tracker_redraw(tracker);
        handled = TRUE;
        break;
      case GDK_Left:
        /* CTRL-Left: Previous instrument */
        if (editor->instrument > 0) {
          editor->instrument--;

          /* Make sure the instrument exists */
          editor_song_instrument_current_check(editor);
          gui_instrument_refresh(gui);
        }
        handled = TRUE;
        break;
      case GDK_Right:
        /* CTRL-Right: Previous instrument */
        editor->instrument++;

        /* Make sure the instrument exists */
        editor_song_instrument_current_check(editor);
        gui_instrument_refresh(gui);

        handled = TRUE;
        break;
      case GDK_ISO_Left_Tab:
        /* CTRL-Shift-Tab: Next command page */
        editor_set_commandpage(editor, editor->cmdpage + 1);
        gui_info_refresh(gui);
        handled = TRUE;
        break;
      case GDK_Delete: {
        if (widget == gui->window_main) {
          int i, n = editor_song_block_current_commandpages_get(editor);

          /* Delete command and refresh */
          for (i = 0; i < n; i++)
            editor_song_block_current_command_full_set(editor, editor->line, tracker->cursor_ch, i, 0, 0);
          tracker_redraw_row(tracker, editor->line);

          editor_set_line(editor, editor->line + editor->space);

          handled = TRUE;
        }
        break;
      }
      case '0':
      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
      case '6':
      case '7':
      case '8':
      case '9': {
        if (alt) {
          /* CTRL+Alt+0-9: Solo tracks */
          int channel = 9;
          if (event->keyval >= '1' && event->keyval <= '9')
            channel = event->keyval - '1';
          if (channel < editor_song_maxtracks_get(editor)) {
            editor_song_track_solo_set(editor, channel, editor_song_track_solo_get(editor, channel) ? 0 : 1);
            gui_trackvolumes_refresh(gui);
          }
        } else {
          /* CTRL+0-9: Set space value */
          GtkSpinButton *spinbutton = GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "spinbutton_space"));

          editor->space = event->keyval - '0';

          g_signal_handlers_block_by_func(spinbutton, on_spinbutton_space_changed, gui);
          gtk_spin_button_set_value(spinbutton, editor->space);
          g_signal_handlers_unblock_by_func(spinbutton, on_spinbutton_space_changed, gui);

          handled = TRUE;
          break;
        }
      }
      }
    } else if (shift) {
      switch (event->keyval) {
      case GDK_Left:
        /* Shift-Left: Next position */
        if (editor->position > 0)
          editor_set_position(editor, editor->position - 1);
        handled = TRUE;
        break;
      case GDK_Right:
        /* Shift-Right: Previous position */
        if (editor->position < editor_song_playseq_current_length_get(editor) - 1)
          editor_set_position(editor, editor->position + 1);
        handled = TRUE;
        break;
      case GDK_ISO_Left_Tab:
        /* Shift-Tab: Previous track - only in main window */
        if (widget == gui->window_main) {
          if ((tracker)->cursor_item > 0) {
            (tracker)->cursor_item = 0;
            tracker_step_cursor_channel(tracker, 0);
          } else
            tracker_step_cursor_channel(tracker, -1);
          handled = TRUE;
        }
        break;
      case GDK_Delete: {
        if (widget == gui->window_main) {
          int i, n = editor_song_block_current_commandpages_get(editor);

          /* Delete note+command and refresh */
          editor_song_block_current_note_set(editor, editor->line, tracker->cursor_ch, 0, 0, 0);
          for (i = 0; i < n; i++)
            editor_song_block_current_command_full_set(editor, editor->line, tracker->cursor_ch, i, 0, 0);
          tracker_redraw_row(tracker, editor->line);

          editor_set_line(editor, editor->line + editor->space);

          handled = TRUE;
        }
        break;
      }
      case GDK_BackSpace:
        /* Shift-Backspace: Insert row */
        if (widget == gui->window_main) {
          if (editor->edit) {
            if (alt)
              editor_song_block_current_insert_line_track_all(editor);
            else
              editor_song_block_current_insert_line(editor, tracker->cursor_ch);
            tracker_redraw(tracker);
          }
          handled = TRUE;
        }
        break;
      default:
        break;
      }
    } else if (alt) {
      switch (event->keyval) {
      case GDK_Left:
        /* Alt-Left: Previous block */
        editor_set_block(editor, editor->block - 1);
        handled = TRUE;
        break;
      case GDK_Right:
        /* Alt-Right: Next block */
        editor_set_block(editor, editor->block + 1);
        handled = TRUE;
        break;
      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
      case '6':
      case '7':
      case '8':
      case '9':
      case '0': {
        /* Alt+0-9: Mute tracks */
        int channel = 9;
        if (event->keyval >= '1' && event->keyval <= '9')
          channel = event->keyval - '1';
        if (channel < editor_song_maxtracks_get(editor)) {
          editor_song_track_mute_set(editor, channel, editor_song_track_mute_get(editor, channel) ? 0 : 1);
          gui_trackvolumes_refresh(gui);
        }
        break;
      }
      case GDK_BackSpace:
        /* Backspace: delete line */
        if (widget == gui->window_main) {
          editor_song_block_current_delete_line_track_all(editor);
          tracker_redraw(tracker);
          handled = TRUE;
        }
        break;
      default:
        break;
      }
    } else {
      char instrument = -1;
      char note = -1;
      char data = -1;
      if (tracker->cursor_item == 0) {
        /* Editing notes */
        switch (event->keyval) {
        case 'z':
          data = 1;
          break;
        case 's':
          data = 2;
          break;
        case 'x':
          data = 3;
          break;
        case 'd':
          data = 4;
          break;
        case 'c':
          data = 5;
          break;
        case 'v':
          data = 6;
          break;
        case 'g':
          data = 7;
          break;
        case 'b':
          data = 8;
          break;
        case 'h':
          data = 9;
          break;
        case 'n':
          data = 10;
          break;
        case 'j':
          data = 11;
          break;
        case 'm':
          data = 12;
          break;
        case 'q':
        case ',':
          data = 13;
          break;
        case '2':
        case 'l':
          data = 14;
          break;
        case 'w':
        case '.':
          data = 15;
          break;
        case '3':
        case 0xf4: /* oumlaut */
          data = 16;
          break;
        case 'e':
        case '-':
          data = 17;
          break;
        case 'r':
          data = 18;
          break;
        case '5':
          data = 19;
          break;
        case 't':
          data = 20;
          break;
        case '6':
          data = 21;
          break;
        case 'y':
          data = 22;
          break;
        case '7':
          data = 23;
          break;
        case 'u':
          data = 24;
          break;
        case 'i':
          data = 25;
          break;
        case '9':
          data = 26;
          break;
        case 'o':
          data = 27;
          break;
        case '0':
          data = 28;
          break;
        case 'p':
          data = 29;
          break;
        case 229:
          data = 30;
          break;
        case 180:
          data = 31;
          break;
        case GDK_Delete:
          data = 0;
          break;
        }

        if (data != -1) {
          if (editor->chord) {
            gboolean found = FALSE;
            int i;

            /* Chord mode: check if the key has already been pressed (key repeat) */
            for (i = 0; i < keyboard_num_keys_down; i++)
              if (keyboard_keys_down[i] == event->keyval) {
                found = TRUE;
                break;
              }

            /* Go to next channel if the current key has not been pressed down yet */
            if (!found) {
              /* Add the key to the keys down list */
              keyboard_num_keys_down++;
              keyboard_keys_down = realloc(keyboard_keys_down, keyboard_num_keys_down * sizeof(guint));
              keyboard_keys_down[keyboard_num_keys_down - 1] = event->keyval;

              if (editor->edit) {
                /* Set note and refresh */
                editor_song_block_current_note_set(editor, editor->line, tracker->cursor_ch, editor->octave, data, editor->instrument + 1);
                tracker_redraw_row(tracker, editor->line);
              }

              tracker_step_cursor_channel(tracker, 1);
              editor->chordstatus++;
            }
          } else if (editor->edit) {
            /* Set note and refresh */
            editor_song_block_current_note_set(editor, editor->line, tracker->cursor_ch, editor->octave, data, editor->instrument + 1);
            tracker_redraw_row(tracker, editor->line);
            editor_set_line(editor, editor->line + editor->space);
          }
          handled = TRUE;
        }
      } else if (editor->edit) {
        switch (tracker->cursor_item) {
        case 1:
        case 2:
          /* Editing instrument */
          if (event->keyval >= '0' && event->keyval <= '9')
            data = event->keyval - '0';
          else if (event->keyval >= 'a' && event->keyval <= 'f')
            data = 10 + event->keyval - 'a';
          else if (event->keyval == GDK_Delete)
            data = 0;

          if (data != -1) {
            /* Set instrument and refresh */
            int ins = editor_song_block_current_instrument_get(editor, editor->line, tracker->cursor_ch);
            if (tracker->cursor_item == 1)
              ins = (ins & 0x0f) | (data << 4);
            else
              ins = (ins & 0xf0) | data;
            editor_song_block_current_instrument_set(editor, editor->line, tracker->cursor_ch, ins);
            tracker_redraw_row(tracker, editor->line);
            editor_set_line(editor, editor->line + editor->space);
            handled = TRUE;
          }
          break;
        case 3:
        case 4:
        case 5:
        case 6:
          /* Editing effects */
          if (event->keyval >= '0' && event->keyval <= '9')
            data = event->keyval - '0';
          else if (event->keyval >= 'a' && event->keyval <= 'f')
            data = 10 + event->keyval - 'a';
          else if (event->keyval == GDK_Delete)
            data = 0;

          if (data != -1) {
            /* Set effect and refresh */
            editor_song_block_current_command_set(editor, editor->line, tracker->cursor_ch, editor->cmdpage, tracker->cursor_item - 3, data);
            tracker_redraw_row(tracker, editor->line);
            editor_set_line(editor, editor->line + editor->space);
            handled = TRUE;
          }
          break;
        }
      }

      switch (event->keyval) {
      case GDK_Shift_R:
        /* Right shift: Play block */
        editor_player_start(editor, MODE_PLAY_BLOCK, 0);
        handled = TRUE;
        break;
      case GDK_Control_R:
        /* Right Control: Play song */
        editor_player_start(editor, MODE_PLAY_SONG, 0);
        handled = TRUE;
        break;
      case GDK_space:
        /* If the song is playing, stop */
        if (editor_player_mode_get(editor) != MODE_IDLE)
          editor_player_stop(editor);
        else {
          /* Otherwise toggle edit mode */
          GtkToggleButton *togglebutton = GTK_TOGGLE_BUTTON(glade_xml_get_widget(gui->xml, "checkbutton_edit"));

          editor->edit ^= 1;

          g_signal_handlers_block_by_func(togglebutton, on_checkbutton_edit_toggled, gui);
          gtk_toggle_button_set_active(togglebutton, editor->edit);
          g_signal_handlers_unblock_by_func(togglebutton, on_checkbutton_edit_toggled, gui);
        }
        handled = TRUE;
        break;
      case GDK_F1:
      case GDK_F2:
      case GDK_F3:
      case GDK_F4:
      case GDK_F5:
      case GDK_F6:
      case GDK_F7:
      case GDK_F8:
      case GDK_F9:
      case GDK_F10:
      case GDK_F11: {
        /* Set active keyboard octave */
        GtkComboBox *combobox = GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_keyboard"));

        editor->octave = event->keyval - GDK_F1;

        g_signal_handlers_block_by_func(combobox, on_combobox_keyboard_changed, gui);
        gtk_combo_box_set_active(combobox, event->keyval - GDK_F1);
        g_signal_handlers_unblock_by_func(combobox, on_combobox_keyboard_changed, gui);

        handled = TRUE;
        break;
      }
      case GDK_Num_Lock:
        instrument = 0;
        break;
      case GDK_KP_Divide:
        instrument = 1;
        break;
      case GDK_KP_Multiply:
        instrument = 2;
        break;
      case GDK_KP_Subtract:
        instrument = 3;
        break;
      case GDK_KP_7:
      case GDK_KP_Home:
        instrument = 4;
        break;
      case GDK_KP_8:
      case GDK_KP_Up:
        instrument = 5;
        break;
      case GDK_KP_9:
      case GDK_KP_Page_Up:
        instrument = 6;
        break;
      case GDK_KP_Add:
        instrument = 7;
        break;
      case GDK_KP_4:
      case GDK_KP_Left:
        instrument = 8;
        break;
      case GDK_KP_5:
      case GDK_KP_Begin:
        instrument = 9;
        break;
      case GDK_KP_6:
      case GDK_KP_Right:
        instrument = 10;
        break;
      case GDK_KP_1:
      case GDK_KP_End:
        instrument = 11;
        break;
      case GDK_KP_2:
      case GDK_KP_Down:
        instrument = 12;
        break;
      case GDK_KP_3:
      case GDK_KP_Page_Down:
        instrument = 13;
        break;
      case GDK_KP_Enter:
        instrument = 14;
        break;
      case GDK_KP_0:
      case GDK_KP_Insert:
        instrument = 15;
        break;
      case GDK_KP_Decimal:
      case GDK_KP_Delete:
        instrument = 16;
        break;
      case 'z':
        note = 0;
        break;
      case 's':
        note = 1;
        break;
      case 'x':
        note = 2;
        break;
      case 'd':
        note = 3;
        break;
      case 'c':
        note = 4;
        break;
      case 'v':
        note = 5;
        break;
      case 'g':
        note = 6;
        break;
      case 'b':
        note = 7;
        break;
      case 'h':
        note = 8;
        break;
      case 'n':
        note = 9;
        break;
      case 'j':
        note = 10;
        break;
      case 'm':
        note = 11;
        break;
      case 'q':
      case ',':
        note = 12;
        break;
      case '2':
      case 'l':
        note = 13;
        break;
      case 'w':
      case '.':
        note = 14;
        break;
      case '3':
      case 0xf4: /* oumlaut */
        note = 15;
        break;
      case 'e':
      case '-':
        note = 16;
        break;
      case 'r':
        note = 17;
        break;
      case '5':
        note = 18;
        break;
      case 't':
        note = 19;
        break;
      case '6':
        note = 20;
        break;
      case 'y':
        note = 21;
        break;
      case '7':
        note = 22;
        break;
      case 'u':
        note = 23;
        break;
      case 'i':
        note = 24;
        break;
      case '9':
        note = 25;
        break;
      case 'o':
        note = 26;
        break;
      case '0':
        note = 27;
        break;
      case 'p':
        note = 28;
        break;
      case 229:
        note = 29;
        break;
      case 180:
        note = 30;
        break;
      case GDK_Down:
        /* Down: Go down -*/
        if (widget == gui->window_main) {
          editor_set_line(editor, editor->line + 1);
          handled = TRUE;
        }
        break;
      case GDK_Up:
        /* Up: Go up */
        if (widget == gui->window_main) {
          editor_set_line(editor, editor->line - 1);
          handled = TRUE;
        }
        break;
      case GDK_Left:
        /* Left: Go left */
        if (widget == gui->window_main) {
          tracker_step_cursor_item(tracker, -1);
          handled = TRUE;
        }
        break;
      case GDK_Right:
        /* Right: Go right */
        if (widget == gui->window_main) {
          tracker_step_cursor_item(tracker, 1);
          handled = TRUE;
        }
        break;
      case GDK_Tab:
        /* Tab: Next track */
        if (widget == gui->window_main) {
          (tracker)->cursor_item = 0;
          tracker_step_cursor_channel(tracker, 1);
          handled = TRUE;
        }
        break;
      case GDK_Home:
        /* End: Go to the beginning of block */
        if (widget == gui->window_main) {
          editor_set_line(editor, 0);
          handled = TRUE;
        }
        break;
      case GDK_End:
        /* End: Go to the end of block */
        if (widget == gui->window_main) {
          editor_set_line(editor, editor_song_block_current_length_get(editor) - 1);
          handled = TRUE;
        }
        break;
      case GDK_Page_Down:
        /* Page down: Go down 8 lines */
        if (widget == gui->window_main) {
          editor_set_line(editor, editor->line + 8);
          handled = TRUE;
        }
        break;
      case GDK_Page_Up:
        /* Page up: Go up 8 lines */
        if (widget == gui->window_main) {
          editor_set_line(editor, editor->line - 8);
          handled = TRUE;
        }
        break;
      case GDK_BackSpace:
        /* Backspace: delete line */
        if (widget == gui->window_main) {
          editor_song_block_current_delete_line(editor, tracker->cursor_ch);
          tracker_redraw(tracker);
          handled = TRUE;
        }
        break;
      }

      /* Select an instrument if an instrument selection key was pressed */
      if (instrument != -1) {
        editor->instrument = instrument;

        /* Make sure the instrument exists */
        editor_song_instrument_current_check(editor);
        gui_instrument_refresh(gui);
        handled = TRUE;
      }

      /* Play note if a key was pressed but not if cursor is in cmd pos */
      if (note != -1 && tracker->cursor_item == 0) {
        editor_player_play_note(editor, note, tracker->cursor_ch);
        handled = TRUE;
      }
    }

    if (handled)
      g_signal_stop_emission_by_name(G_OBJECT(widget), "key_press_event");
  } else {
    /* Key has been released */
    if (!ctrl && !shift) {
      if (editor->chord && tracker->cursor_item == 0) {
        int i;

        switch (event->keyval) {
        case 'z':
        case 's':
        case 'x':
        case 'd':
        case 'c':
        case 'v':
        case 'g':
        case 'b':
        case 'h':
        case 'n':
        case 'j':
        case 'm':
        case 'q':
        case ',':
        case '2':
        case 'l':
        case 'w':
        case '.':
        case '3':
        case 0xf4: /* oumlaut */
        case 'e':
        case '-':
        case 'r':
        case '5':
        case 't':
        case '6':
        case 'y':
        case '7':
        case 'u':
        case 'i':
        case '9':
        case 'o':
        case '0':
        case 'p':
        case 229:
        case 180:
          /* Find the key from the keys down list and remove it */
          for (i = 0; i < keyboard_num_keys_down; i++)
            if (keyboard_keys_down[i] == event->keyval) {
              while (i < keyboard_num_keys_down - 1) {
                keyboard_keys_down[i] = keyboard_keys_down[i + 1];
                i++;
              }
              keyboard_num_keys_down--;
              keyboard_keys_down = realloc(keyboard_keys_down, keyboard_num_keys_down * sizeof(guint));
              break;
            }

          /* Go to the previous channel */
          tracker_step_cursor_channel(tracker, -1);
          editor->chordstatus--;

          /* If all chord notes have been released go to the next line */
          if (editor->chordstatus == 0 && editor->edit)
            editor_set_line(editor, editor->line + editor->space);
          handled = TRUE;
          break;
        }
      }
    }

    if (handled)
      g_signal_stop_emission_by_name(G_OBJECT(widget), "key_release_event");
  }

  return handled;
}
