/* instrument.c
 *
 * Copyright 2002-2011 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <string.h>
#include "tutka.h"
#include "instrument.h"

struct instrument *instrument_alloc()
{
  struct instrument *instrument;

  instrument = (struct instrument *)calloc(1, sizeof(struct instrument));
  instrument->name = (char *)strdup("Unnamed");
  instrument->numoutputs = 1;
  instrument->outputs = (struct instrument_output **)calloc(instrument->numoutputs, sizeof(struct instrument_output *));
  instrument->outputs[0] = (struct instrument_output *)calloc(1, sizeof(struct instrument_output));
  instrument->outputs[0]->defaultvelocity = 127;

  return instrument;
}

void instrument_free(struct instrument *instrument)
{
  if (instrument != NULL) {
    int i;

    free(instrument->name);

    if (instrument->outputs != NULL) {
      for (i = 0; i < instrument->numoutputs; i++) {
        free(instrument->outputs[i]->midiinterfacename);
        free(instrument->outputs[i]);
      }
      free(instrument->outputs);
    }

    free(instrument);
  }
}

/* Parses an instrument element in an XML file */
struct instrument *instrument_parse(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur)
{
  struct instrument *instrument = NULL;

  if ((!xmlStrcmp(cur->name, (xmlChar *)"instrument")) && (cur->ns == ns)) {
    xmlChar *prop;

    /* Allocate instrument */
    instrument = (struct instrument *)calloc(1, sizeof(struct instrument));
    prop = xmlGetProp(cur, (xmlChar *)"name");
    if (prop != NULL) {
      instrument->name = strdup((char *)prop);
      xmlFree(prop);
    } else
      instrument->name = (char *)calloc(1, sizeof(char));

    /* Get block contents */
    cur = cur->xmlChildrenNode;
    while (cur != NULL) {
      if ((!xmlStrcmp(cur->name, (xmlChar *)"output")) && (cur->ns == ns)) {
        struct instrument_output *output;

        instrument->numoutputs++;
        instrument->outputs = realloc(instrument->outputs, instrument->numoutputs * sizeof(struct instrument_output *));
        instrument->outputs[instrument->numoutputs - 1] = (struct instrument_output *)calloc(1, sizeof(struct instrument_output));
        output = instrument->outputs[instrument->numoutputs - 1];

        /* Get output properties */
        prop = xmlGetProp(cur, (xmlChar *)"midiinterface");
        if (prop != NULL) {
          output->midiinterfacename = strdup((char *)prop);
          xmlFree(prop);
        }

        prop = xmlGetProp(cur, (xmlChar *)"midipreset");
        if (prop != NULL) {
          output->midipreset = atoi((char *)prop);
          xmlFree(prop);
        }

        prop = xmlGetProp(cur, (xmlChar *)"midichannel");
        if (prop != NULL) {
          output->midichannel = atoi((char *)prop);
          xmlFree(prop);
        }

        prop = xmlGetProp(cur, (xmlChar *)"defaultvelocity");
        if (prop != NULL) {
          output->defaultvelocity = atoi((char *)prop);
          xmlFree(prop);
        }

        prop = xmlGetProp(cur, (xmlChar *)"transpose");
        if (prop != NULL) {
          output->transpose = atoi((char *)prop);
          xmlFree(prop);
        }

        prop = xmlGetProp(cur, (xmlChar *)"hold");
        if (prop != NULL) {
          output->hold = atoi((char *)prop);
          xmlFree(prop);
        }
      }
      cur = cur->next;
    }
  } else if (cur->type != XML_COMMENT_NODE)
    fprintf(stderr, "XML error: expected instrument, got %s\n", cur->name);

  return instrument;
}

/* Saves an instrument to an XML document */
void instrument_save(struct instrument *instrument, int number, xmlNodePtr parent)
{
  nullcheck_void(instrument, instrument_save);

  int i;
  xmlChar c[10];
  xmlNodePtr node, subnode;

  node = xmlNewChild(parent, NULL, (xmlChar *)"instrument", NULL);
  snprintf((char *)c, 10, "%d", number);
  xmlSetProp(node, (xmlChar *)"number", c);
  xmlSetProp(node, (xmlChar *)"name", (xmlChar *)instrument->name);

  for (i = 0; i < instrument->numoutputs; i++) {
    xmlAddChild(node, xmlNewText((xmlChar *)"\n"));
    subnode = xmlNewChild(node, NULL, (xmlChar *)"output", NULL);
    if (instrument->outputs[i]->midiinterfacename != NULL)
      xmlSetProp(subnode, (xmlChar *)"midiinterface", (xmlChar *)instrument->outputs[i]->midiinterfacename);
    snprintf((char *)c, 10, "%d", instrument->outputs[i]->midipreset);
    xmlSetProp(subnode, (xmlChar *)"midipreset", c);
    snprintf((char *)c, 10, "%d", instrument->outputs[i]->midichannel);
    xmlSetProp(subnode, (xmlChar *)"midichannel", c);
    snprintf((char *)c, 10, "%d", instrument->outputs[i]->defaultvelocity);
    xmlSetProp(subnode, (xmlChar *)"defaultvelocity", c);
    snprintf((char *)c, 10, "%d", instrument->outputs[i]->transpose);
    xmlSetProp(subnode, (xmlChar *)"transpose", c);
    snprintf((char *)c, 10, "%d", instrument->outputs[i]->hold);
    xmlSetProp(subnode, (xmlChar *)"hold", c);
    xmlAddChild(node, xmlNewText((xmlChar *)"\n"));
  }

  xmlAddChild(parent, xmlNewText((xmlChar *)"\n"));
}
