/* instrument.c
 *
 * Copyright 2002-2006 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <string.h>
#include "tutka.h"
#include "instrument.h"

struct instrument *instrument_alloc()
{
  struct instrument *instrument;

  instrument = (struct instrument *)calloc(1, sizeof(struct instrument));
  instrument->name = (char *)strdup("Unnamed");
  instrument->numoutputs = 1;
  instrument->outputs = (struct instrument_output **)calloc(instrument->numoutputs, sizeof(struct instrument_output *));
  instrument->outputs[0] = (struct instrument_output *)calloc(1, sizeof(struct instrument_output));
  instrument->outputs[0]->defaultvelocity = 127;

  return instrument;
}

void instrument_free(struct instrument *instrument)
{
  nullcheck_void(instrument, instrument_free);

  int i;

  if (instrument->name)
    free(instrument->name);

  if (instrument->outputs != NULL) {
    for (i = 0; i < instrument->numoutputs; i++) {
      if (instrument->outputs[i]->midiinterfacename)
	free(instrument->outputs[i]->midiinterfacename);
      free(instrument->outputs[i]);
    }
    free(instrument->outputs);
  }

  free(instrument);
}

/* Parses an instrument element in an XML file */
struct instrument *instrument_parse(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur)
{
  struct instrument *instrument = NULL;

  if ((!xmlStrcmp(cur->name, "instrument")) && (cur->ns == ns)) {
    char *prop;
    xmlNodePtr temp = cur;

    /* Allocate instrument */
    instrument = (struct instrument *)calloc(1, sizeof(struct instrument));
    prop = xmlGetProp(cur, "name");
    if (prop != NULL)
      instrument->name = strdup(prop);
    else
      instrument->name = (char *)calloc(1, sizeof(char));

    /* Get block contents */
    cur = cur->xmlChildrenNode;
    while (cur != NULL) {
      if ((!xmlStrcmp(cur->name, "output")) && (cur->ns == ns)) {
	struct instrument_output *output;

	instrument->numoutputs++;
	instrument->outputs = realloc(instrument->outputs, instrument->numoutputs * sizeof(struct instrument_output *));
	instrument->outputs[instrument->numoutputs - 1] = (struct instrument_output *)calloc(1, sizeof(struct instrument_output));
	output = instrument->outputs[instrument->numoutputs - 1];

	/* Get output properties */
	prop = xmlGetProp(cur, "midiinterface");
	if (prop != NULL)
	  output->midiinterfacename = strdup(prop);
	prop = xmlGetProp(cur, "midipreset");
	if (prop != NULL)
	  output->midipreset = atoi(prop);
	prop = xmlGetProp(cur, "midichannel");
	if (prop != NULL)
	  output->midichannel = atoi(prop);
	prop = xmlGetProp(cur, "defaultvelocity");
	if (prop != NULL)
	  output->defaultvelocity = atoi(prop);
	prop = xmlGetProp(cur, "transpose");
	if (prop != NULL)
	  output->transpose = atoi(prop);
	prop = xmlGetProp(cur, "hold");
	if (prop != NULL)
	  output->hold = atoi(prop);
      }
      cur = cur->next;
    }
    
    /* Backwards compatibility with Tutka 0.11.0: read stuff from the <instrument> */
    if (instrument->outputs == 0) {
      struct instrument_output *output;

      cur = temp;

      /* Get instrument name */
      temp = cur->xmlChildrenNode;
      if (temp != NULL)
	instrument->name = strdup(temp->content);
      else
	instrument->name = (char *)calloc(1, sizeof(char));

      /* Create one output */
      instrument->numoutputs++;
      instrument->outputs = calloc(instrument->numoutputs, sizeof(struct instrument_output *));
      instrument->outputs[0] = (struct instrument_output *)calloc(1, sizeof(struct instrument_output));
      output = instrument->outputs[0];

      /* Get output properties */
      prop = xmlGetProp(cur, "midichannel");
      if (prop != NULL)
	output->midichannel = atoi(prop);
      prop = xmlGetProp(cur, "defaultvelocity");
      if (prop != NULL)
	output->defaultvelocity = atoi(prop);
      prop = xmlGetProp(cur, "transpose");
      if (prop != NULL)
	output->transpose = atoi(prop);
      prop = xmlGetProp(cur, "hold");
      if (prop != NULL)
	output->hold = atoi(prop);
    }
  } else if (cur->type != XML_COMMENT_NODE)
    fprintf(stderr, "XML error: expected instrument, got %s\n", cur->name);

  return instrument;
}

/* Saves an instrument to an XML document */
void instrument_save(struct instrument *instrument, int number, xmlNodePtr parent)
{
  nullcheck_void(instrument, instrument_save);

  int i;
  char c[10];
  xmlNodePtr node, subnode;

  node = xmlNewChild(parent, NULL, "instrument", NULL);
  snprintf(c, 10, "%d", number);
  xmlSetProp(node, "number", c);
  xmlSetProp(node, "name", instrument->name);

  for (i = 0; i < instrument->numoutputs; i++) {
    xmlAddChild(node, xmlNewText("\n"));
    subnode = xmlNewChild(node, NULL, "output", NULL);
    if (instrument->outputs[i]->midiinterfacename != NULL)
      xmlSetProp(subnode, "midiinterface", instrument->outputs[i]->midiinterfacename);
    snprintf(c, 10, "%d", instrument->outputs[i]->midipreset);
    xmlSetProp(subnode, "midipreset", c);
    snprintf(c, 10, "%d", instrument->outputs[i]->midichannel);
    xmlSetProp(subnode, "midichannel", c);
    snprintf(c, 10, "%d", instrument->outputs[i]->defaultvelocity);
    xmlSetProp(subnode, "defaultvelocity", c);
    snprintf(c, 10, "%d", instrument->outputs[i]->transpose);
    xmlSetProp(subnode, "transpose", c);
    snprintf(c, 10, "%d", instrument->outputs[i]->hold);
    xmlSetProp(subnode, "hold", c);
    xmlAddChild(node, xmlNewText("\n"));
  }

  xmlAddChild(parent, xmlNewText("\n"));
}
