/* callbacks.c
 *
 * Copyright 2002-2005 Vesa Halttunen
 *
 * This file is part of Tutka. The code in this file is based on 
 * The Real SoundTracker - GTK+ Tracker widget
 *
 * Copyright (C) 1998-2001 Michael Krause
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include "trackerwidget.h"
#include "trackerwidget-marshal.h"
#include "editor.h"

const char *const notenames[128] = {
  "---",
  "C-1", "C#1", "D-1", "D#1", "E-1", "F-1", "F#1", "G-1", "G#1", "A-1", "A#1", "B-1",
  "C-2", "C#2", "D-2", "D#2", "E-2", "F-2", "F#2", "G-2", "G#2", "A-2", "A#2", "B-2",
  "C-3", "C#3", "D-3", "D#3", "E-3", "F-3", "F#3", "G-3", "G#3", "A-3", "A#3", "B-3",
  "C-4", "C#4", "D-4", "D#4", "E-4", "F-4", "F#4", "G-4", "G#4", "A-4", "A#4", "B-4",
  "C-5", "C#5", "D-5", "D#5", "E-5", "F-5", "F#5", "G-5", "G#5", "A-5", "A#5", "B-5",
  "C-6", "C#6", "D-6", "D#6", "E-6", "F-6", "F#6", "G-6", "G#6", "A-6", "A#6", "B-6",
  "C-7", "C#7", "D-7", "D#7", "E-7", "F-7", "F#7", "G-7", "G#7", "A-7", "A#7", "B-7",
  "C-8", "C#8", "D-8", "D#8", "E-8", "F-8", "F#8", "G-8", "G#8", "A-8", "A#8", "B-8",
  "C-9", "C#9", "D-9", "D#9", "E-9", "F-9", "F#9", "G-9", "G#9", "A-9", "A#9", "B-9",
  "C-A", "C#A", "D-A", "D#A", "E-A", "F-A", "F#A", "G-A", "G#A", "A-A", "A#A", "B-A",
  "C-B", "C#B", "D-B", "D#B", "E-B", "F-B", "F#B"
};

static void init_display(Tracker *t, int width, int height);

static const int default_colors[] = {
  10, 20, 30,
  100, 100, 100,
  50, 60, 70,
  230, 230, 230,
  170, 170, 200,
  230, 200, 0,
  115, 100, 0,
  255, 230, 200,
  172, 150, 0,
  250, 100, 50
};

static const int ascii_to_font[] = {
  0, -1, -1, 1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 2, -1, -1,
  3, 4, 5, 6, 7, 8, 9, 10, 11, 12, -1, -1, -1, -1, -1, -1,
  -1, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27,
  28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, -1, -1, -1, -1, -1
};

static const char font_to_ascii[] = {
  ' ', '#', '-', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
  'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M',
  'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z'
};
  
enum {
  PATPOS_SIGNAL,
  XPANNING_SIGNAL,
  BLOCKMARK_SET_SIGNAL,
  LAST_SIGNAL
};

#define CLEAR(win, x, y, w, h) \
do { gdk_draw_rectangle(win, t->bg_gc, TRUE, x, y, w, h); } while(0)

static gint tracker_signals[LAST_SIGNAL]={ 0 };

static gint tracker_idle_draw_function (Tracker *t);

static void tracker_idle_draw (Tracker *t) {
  if(t->idle_handler == 0) {
    t->idle_handler = gtk_idle_add((GtkFunction)tracker_idle_draw_function, (gpointer)t);
    g_assert(t->idle_handler != 0);
  }
}

void tracker_set_num_channels (Tracker *t, int n) {
  GtkWidget *widget = GTK_WIDGET(t);
  
  t->num_channels = n;

  /* Make sure the cursor is inside the tracker */
  if(t->cursor_ch >= t->num_channels)
    t->cursor_ch = t->num_channels - 1;

  if(GTK_WIDGET_REALIZED(widget)) {
    init_display(t, widget->allocation.width, widget->allocation.height);
    gtk_widget_queue_draw(widget);
  }
  g_signal_emit(GTK_OBJECT(t), tracker_signals[XPANNING_SIGNAL], 0, t->leftchan, t->num_channels, t->disp_numchans);
}

void tracker_set_cmdpage (Tracker *t, int cmdpage) {
  g_return_if_fail(t != NULL);

  t->cmdpage = cmdpage;

  gtk_widget_queue_draw(GTK_WIDGET(t)); 
}

void tracker_set_patpos (Tracker *t, int row) {
  g_return_if_fail(t != NULL);
  g_return_if_fail((t->curpattern == NULL && row == 0) || (row < t->curpattern->length));
  
  if(t->patpos != row) {
    t->patpos = row;
    if(t->curpattern != NULL) {
      if(t->inSelMode)
	/* Force re-draw of patterns in block selection mode */
	gtk_widget_queue_draw(GTK_WIDGET(t));
      else
	tracker_idle_draw(t);

      g_signal_emit(GTK_OBJECT(t), tracker_signals[PATPOS_SIGNAL], 0, row, t->curpattern->length, t->disp_rows);
    }
  }
}

void tracker_redraw (Tracker *t) {
  gtk_widget_queue_draw(GTK_WIDGET(t));
}

void tracker_redraw_row (Tracker *t, int row) {
  /* This is yet to be optimized :-) */
  tracker_redraw(t);
}

void tracker_redraw_current_row (Tracker *t) {
  tracker_redraw_row(t, t->patpos);
}

void tracker_set_song(Tracker *t, struct song *song) {
  g_return_if_fail(t != NULL);
  
  t->song = song;
}

void tracker_set_pattern (Tracker *t, struct block *pattern) {
  g_return_if_fail(t != NULL);
  
  if(t->curpattern != pattern) {
    t->curpattern = pattern;
    if(pattern != NULL) {
      /* Make sure the cursor is inside the tracker */
      if(t->patpos >= pattern->length)
	t->patpos = pattern->length - 1;
      if(t->cursor_ch >= pattern->tracks)
	t->cursor_ch = pattern->tracks - 1;

      g_signal_emit(GTK_OBJECT(t), tracker_signals[PATPOS_SIGNAL], 0, t->patpos, pattern->length, t->disp_rows);
    }
    gtk_widget_queue_draw(GTK_WIDGET(t));
  }
}

void tracker_set_backing_store (Tracker *t, int on) {
  GtkWidget *widget;
  
  g_return_if_fail(t != NULL);
  
  if(on == t->enable_backing_store)
    return;
  
  t->enable_backing_store = on;
  
  widget = GTK_WIDGET(t);
  if(GTK_WIDGET_REALIZED(widget)) {
    if(on) {
      t->pixmap = gdk_pixmap_new(widget->window, widget->allocation.width, widget->allocation.height, -1);
      CLEAR(t->pixmap, 0, 0, widget->allocation.width, widget->allocation.height);
    } else {
      gdk_pixmap_unref(t->pixmap);
      t->pixmap = NULL;
    }
    
    gdk_gc_set_exposures(t->bg_gc, !on);
    gtk_widget_queue_draw(GTK_WIDGET(t));
  }
}

void tracker_set_xpanning (Tracker *t, int left_channel) {
  g_return_if_fail(t != NULL);
  
  if(t->leftchan != left_channel) {
    GtkWidget *widget = GTK_WIDGET(t);

    if(GTK_WIDGET_REALIZED(widget)) {
      g_return_if_fail(left_channel + t->disp_numchans <= t->num_channels);
      
      t->leftchan = left_channel;
      gtk_widget_queue_draw(GTK_WIDGET(t));
      
      if(t->cursor_ch < t->leftchan)
	t->cursor_ch = t->leftchan;
      else if(t->cursor_ch >= t->leftchan + t->disp_numchans)
	t->cursor_ch = t->leftchan + t->disp_numchans - 1;
    }
    g_signal_emit(GTK_OBJECT(t), tracker_signals[XPANNING_SIGNAL], 0, t->leftchan, t->num_channels, t->disp_numchans);
  }
}

static void adjust_xpanning (Tracker *t) {
  if(t->leftchan + t->disp_numchans > t->num_channels)
    tracker_set_xpanning(t, t->num_channels - t->disp_numchans);
  else if(t->cursor_ch < t->leftchan)
    tracker_set_xpanning(t, t->cursor_ch);
  else if(t->cursor_ch >= t->leftchan + t->disp_numchans)
    tracker_set_xpanning(t, t->cursor_ch - t->disp_numchans + 1);
}

void tracker_step_cursor_item (Tracker *t, int direction) {
  g_return_if_fail(direction == -1 || direction == 1);
  
  t->cursor_item += direction;
  if(t->cursor_item > 6) {
    t->cursor_item %= 7;
    tracker_step_cursor_channel(t, direction);
  } else if(t->cursor_item < 0) {
    t->cursor_item += 7;
    tracker_step_cursor_channel(t, direction);
  } else {
    adjust_xpanning(t);
    gtk_widget_queue_draw(GTK_WIDGET(t));
  }
}

void tracker_step_cursor_channel (Tracker *t, int direction) {
  t->cursor_ch += direction;
  
  if(t->cursor_ch < 0)
    t->cursor_ch = t->num_channels - 1;
  else if(t->cursor_ch >= t->num_channels)
    t->cursor_ch = 0;
  
  adjust_xpanning(t);
  
  if(t->inSelMode) {
    /* Force re-draw of patterns in block selection mode */
    gtk_widget_queue_draw(GTK_WIDGET(t));
  } else {
    tracker_idle_draw(t);
  }
}

void tracker_step_cursor_row (Tracker *t, int direction) {
  int newpos = t->patpos + direction;
  
  while(newpos < 0)
    newpos += t->curpattern->length;
  newpos %= t->curpattern->length;
  
  tracker_set_patpos(t, newpos);
}

void tracker_mark_selection (Tracker *t, gboolean enable) {
  if(!enable) {
    t->sel_end_ch = t->cursor_ch;
    t->sel_end_row = t->patpos;
    t->inSelMode = FALSE;
  } else {
    t->sel_start_ch = t->sel_end_ch = t->cursor_ch;
    t->sel_start_row = t->sel_end_row = t->patpos;
    t->inSelMode = TRUE;
    tracker_redraw(t);		
  }
}

void tracker_clear_mark_selection (Tracker *t) {
  if(t->sel_start_ch != -1) {
    t->sel_start_ch = t->sel_end_ch = -1;
    t->sel_start_row = t->sel_end_row = -1;
    t->inSelMode = FALSE;
    tracker_redraw(t);
  }
}

gboolean tracker_is_in_selection_mode(Tracker *t) {
  return t->inSelMode;
}

void tracker_get_selection_rect(Tracker *t, int *chStart, int *rowStart, int *nChannel, int *nRows) {
  if(!t->inSelMode) {
    if(t->sel_start_ch <= t->sel_end_ch) {
      *nChannel = t->sel_end_ch - t->sel_start_ch + 1;
      *chStart = t->sel_start_ch;
    } else {
      *nChannel = t->sel_start_ch - t->sel_end_ch + 1;
      *chStart = t->sel_end_ch;         	
    }
    if(t->sel_start_row <= t->sel_end_row) {
      *nRows = t->sel_end_row - t->sel_start_row + 1;
      *rowStart = t->sel_start_row;
    } else {
      *nRows = t->sel_start_row - t->sel_end_row + 1;
      *rowStart = t->sel_end_row;
    }	
  } else {
    if(t->sel_start_ch <= t->cursor_ch) {
      *nChannel = t->cursor_ch - t->sel_start_ch + 1;
      *chStart = t->sel_start_ch;
    } else {
      *nChannel = t->sel_start_ch - t->cursor_ch + 1;
      *chStart = t->cursor_ch;         	
    }
    if(t->sel_start_row <= t->patpos) {
      *nRows = t->patpos - t->sel_start_row + 1;
      *rowStart = t->sel_start_row;
    } else {
      *nRows = t->sel_start_row - t->patpos + 1;
      *rowStart = t->patpos;
    }	
  }
}

gboolean tracker_is_valid_selection(Tracker *t) {
  return (t->sel_start_ch >= 0 && t->sel_start_ch < t->curpattern->tracks &&
	  t->sel_end_ch >= 0 && t->sel_end_ch < t->curpattern->tracks &&
	  t->sel_start_row >= 0 && t->sel_start_row < t->curpattern->length &&
	  t->sel_end_row >= 0 && t->sel_end_row < t->curpattern->length);
}

static void note2string (unsigned char *note, unsigned char *effect, char *buf) {
  static const char hexmap[] = {
    '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F',
    'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V',
    'W', 'X', 'Y', 'Z'
  };
  
  buf[0] = notenames[note[0]][0];
  buf[1] = notenames[note[0]][1];
  buf[2] = notenames[note[0]][2];
  buf[3] = ' ';
  buf[4] = hexmap[(note[1] & 0xf0) >> 4];
  buf[5] = hexmap[note[1] & 0x0f];
  buf[6] = ' ';
  buf[7] = hexmap[(effect[0] & 0xf0) >> 4];
  buf[8] = hexmap[effect[0] & 0x0f];
  buf[9] = ' ';
  buf[10] = hexmap[(effect[1] & 0xf0) >> 4];
  buf[11] = hexmap[effect[1] & 0x0f];
  buf[12] = ' ';
  buf[13] = 0;
}

static void tracker_clear_notes_line (GtkWidget *widget, GdkDrawable *win, int y, int pattern_row) {
  Tracker *t = TRACKER(widget);
  GdkGC *gc;
  
  gc = t->bg_gc;
  
  /* cursor line */
  if(pattern_row == t->patpos)
    gc = t->bg_cursor_gc;
  
  gdk_draw_rectangle(win, gc, TRUE, 0, y, widget->allocation.width, t->fonth);
}

static void print_notes_line (GtkWidget *widget, GdkDrawable *win, int y, int ch, int numch, int row, int cursor) {
  Tracker *t = TRACKER(widget);
  char buf[TRACKER_CHANNEL_WIDTH + 1];
  int rbs, rbe, cbs, cbe, i, j;
  unsigned char *notedata, *effectdata;

  g_return_if_fail(ch + numch <= t->curpattern->tracks);
  
  tracker_clear_notes_line(widget, win, y, row);

  /* Figure out which rows/columns should be highlighted */
  if (t->inSelMode) {
    rbs = t->sel_start_row;
    rbe = t->patpos;
    cbs = t->sel_start_ch;
    cbe = t->cursor_ch;
  } else {
    rbs = t->sel_start_row;
    rbe = t->sel_end_row;
    cbs = t->sel_start_ch;
    cbe = t->sel_end_ch;
  }
  
  /* The row number */
  sprintf(buf, "%03d", row);
  if (cursor)
    for (i = 0; i < 3; i++)
      gdk_draw_pixmap(win, t->notes_gc, t->fonts_cursor[ascii_to_font[buf[i] - ' ']], 0, 0, 1 + i * t->fontw, y, t->fontw, t->fonth);
  else
    for (i = 0; i < 3; i++)
      gdk_draw_pixmap(win, t->notes_gc, t->fonts_normal[ascii_to_font[buf[i] - ' ']], 0, 0, 1 + i * t->fontw, y, t->fontw, t->fonth);

  /* The notes */
  notedata = t->curpattern->notes;
  effectdata = t->curpattern->commands + (t->cmdpage * 2 * t->curpattern->tracks * t->curpattern->length);
  buf[TRACKER_CHANNEL_WIDTH] = 0;
  for(numch += ch, j = 0; ch < numch; ch++, j++) {
    note2string(notedata + (t->curpattern->tracks * row + ch) * 2, effectdata + (t->curpattern->tracks * row + ch) * 2, buf);

    if (cursor)
      for (i = 0; i < TRACKER_CHANNEL_WIDTH; i++)
        gdk_draw_pixmap(win, t->notes_gc, t->fonts_cursor[ascii_to_font[buf[i] - ' ']], 0, 0, t->disp_startx + (j * TRACKER_CHANNEL_WIDTH + i) * t->fontw, y, t->fontw, t->fonth);
    else if (row >= rbs && row <= rbe && ch >= cbs && ch <= cbe)
      for (i = 0; i < TRACKER_CHANNEL_WIDTH; i++)
        gdk_draw_pixmap(win, t->notes_gc, t->fonts_select[ascii_to_font[buf[i] - ' ']], 0, 0, t->disp_startx + (j * TRACKER_CHANNEL_WIDTH + i) * t->fontw, y, t->fontw, t->fonth);
    else
      for (i = 0; i < TRACKER_CHANNEL_WIDTH; i++)
        gdk_draw_pixmap(win, t->notes_gc, t->fonts_normal[ascii_to_font[buf[i] - ' ']], 0, 0, t->disp_startx + (j * TRACKER_CHANNEL_WIDTH + i) * t->fontw, y, t->fontw, t->fonth);
  }
}

static void print_notes_and_bars (GtkWidget *widget, GdkDrawable *win, int x, int y, int w, int h, int cursor_row) {
  int scry, n, n1, n2, my, i, x1, y1;
  Tracker *t = TRACKER(widget);

  /* Limit y and h to the actually used window portion */
  my = y - t->disp_starty;
  if(my < 0) {
    my = 0;
    h += my;                                                                                                           	
  }
  if(my + h > t->fonth * t->disp_rows) {
    h = t->fonth * t->disp_rows - my;
  }
  
  /* Calculate first and last line to be redrawn */
  n1 = my / t->fonth;
  n2 = (my + h - 1) / t->fonth;
  
  /* Print the notes */
  scry = t->disp_starty + n1 * t->fonth;
  for(i = n1; i <= n2; i++, scry += t->fonth) {
    n = i + cursor_row - t->disp_cursor;
    if(n >= 0 && n < t->curpattern->length) {
      print_notes_line(widget, win, scry, t->leftchan, t->disp_numchans, n, n == cursor_row ? 1 : 0);
    } else {
      CLEAR(win, 0, scry, widget->allocation.width, t->fonth);
    }
  }
  
  /* Draw the separation bars */
  gdk_gc_set_foreground(t->misc_gc, &t->colors[TRACKERCOL_BARS]);
  x1 = t->disp_startx - 3;
  y1 = t->disp_starty + n1 * t->fonth;
  h = (n2 - n1 + 1) * t->fonth;
  for(i = 0; i <= t->disp_numchans; i++, x1 += t->disp_chanwidth)
    gdk_draw_line(win, t->misc_gc, x1, 0, x1, widget->allocation.height - 1);
}

static void print_channel_headers (GtkWidget *widget, GdkDrawable *win) {
  int x, i;
  char buf[TRACKER_CHANNEL_WIDTH + 1];
  Tracker *t = TRACKER(widget);
  PangoContext *context = gdk_pango_context_get_for_screen(gdk_drawable_get_screen(win));
  PangoLayout *layout = pango_layout_new(context);
  pango_layout_set_font_description (layout, t->fontdesc);
 
  x = t->disp_startx - 3;
  
  for(i = 1; i <= t->disp_numchans; i++, x += t->disp_chanwidth) {
    char *name = t->song->tracks[i + t->leftchan - 1]->name;
    if (name == NULL)
      name = "";
    snprintf(buf, TRACKER_CHANNEL_WIDTH + 1, "%d: %s", i + t->leftchan, name);
    gdk_draw_rectangle(win, t->bg_gc, TRUE, x, 0, t->disp_chanwidth, t->fonth);

    if (t->song->tracks[i + t->leftchan - 1]->mute && !t->song->tracks[i + t->leftchan - 1]->solo)
      gdk_gc_set_foreground(t->misc_gc, &t->colors[TRACKERCOL_CHANNEL_HEADER_MUTE]);
    else if (!t->song->tracks[i + t->leftchan - 1]->mute && t->song->tracks[i + t->leftchan - 1]->solo)
      gdk_gc_set_foreground(t->misc_gc, &t->colors[TRACKERCOL_CHANNEL_HEADER_SOLO]);
    else if (t->song->tracks[i + t->leftchan - 1]->mute && t->song->tracks[i + t->leftchan - 1]->solo)
      gdk_gc_set_foreground(t->misc_gc, &t->colors[TRACKERCOL_CHANNEL_HEADER_MUTE_SOLO]);
    else
      gdk_gc_set_foreground(t->misc_gc, &t->colors[TRACKERCOL_CHANNEL_HEADER]);

    pango_layout_set_text (layout, buf, -1);
    pango_layout_context_changed (layout);
    gdk_draw_layout(win, t->misc_gc, x + 2, 0, layout);
  }
  
  g_object_unref(layout);
  g_object_unref(context);
}

static void print_cursor (GtkWidget *widget, GdkDrawable *win) {
  int x, y, width;
  Tracker *t = TRACKER(widget);

  g_return_if_fail(t->cursor_ch >= t->leftchan && t->cursor_ch < t->leftchan + t->disp_numchans);
  g_return_if_fail((unsigned)t->cursor_item <= 6);
  
  width = 1;
  x = 0;
  y = t->disp_starty + t->disp_cursor * t->fonth;
  
  switch(t->cursor_item) {
  case 0: /* note */
    width = 3;
    break;
  case 1: /* instrument 0 */
    x = 4;
    break;
  case 2: /* instrument 1 */
    x = 5;
    break;
  case 3: /* effect 0 */
    x = 7;
    break;
  case 4: /* effect 1 */
    x = 8;
    break;
  case 5: /* effect 2 */
    x = 10;
    break;
  case 6: /* effect 3 */
    x = 11;
    break;
  default:
    g_assert_not_reached();
    break;
  }
  
  x = x * t->fontw + t->disp_startx + (t->cursor_ch - t->leftchan) * t->disp_chanwidth - 1; 
  
  gdk_gc_set_foreground(t->misc_gc, &t->colors[TRACKERCOL_CURSOR]);
  gdk_draw_rectangle(win, t->misc_gc, FALSE, x, y, width * t->fontw, t->fonth - 1);
}

static void tracker_draw_clever (GtkWidget *widget, GdkRectangle *area) {
  Tracker *t;
  int dist, absdist;
  int y, redrawcnt;
  GdkWindow *win;
  int fonth;

  g_return_if_fail(widget != NULL);
  
  if(!GTK_WIDGET_VISIBLE(widget))
    return;
  
  t = TRACKER(widget);
  g_return_if_fail(t->curpattern != NULL);
  
  win = t->enable_backing_store ? (GdkWindow*)t->pixmap : widget->window;

  fonth = t->fonth;
  
  dist = t->patpos - t->oldpos;
  absdist = ABS(dist);
  t->oldpos = t->patpos;
  
  redrawcnt = t->disp_rows;
  y = t->disp_starty;
  
  if(absdist <= t->disp_cursor) {
    /* Before scrolling, redraw cursor line in the old picture;
       better than scrolling first and then redrawing the old
       cursor line (prevents flickering) */
    print_notes_and_bars(widget, win, 0, (t->disp_cursor) * fonth + t->disp_starty, widget->allocation.width, fonth, t->oldpos - dist);
  }
  
  if(absdist < t->disp_rows) {
    /* this is interesting. we don't have to redraw the whole area, we can optimize
       by scrolling around instead. */
    if(dist > 0) {
      /* go down in pattern -- scroll up */
      redrawcnt = absdist;
      gdk_window_copy_area(win, t->bg_gc, 0, y, win, 0, y + (absdist * fonth), widget->allocation.width, (t->disp_rows - absdist) * fonth);
      y += (t->disp_rows - absdist) * fonth;
    } else if(dist < 0) {
      /* go up in pattern -- scroll down */
      redrawcnt = absdist;
      gdk_window_copy_area(win, t->bg_gc, 0, y + (absdist * fonth), win, 0, y, widget->allocation.width, (t->disp_rows - absdist) * fonth);
    }
  }
  
  if(dist != 0) {
    /* It shouldn't be necessary to call this every time anymore! Figure out how to do it properly */
    print_channel_headers(widget, win);
    print_notes_and_bars(widget, win, 0, y, widget->allocation.width, redrawcnt * fonth, t->oldpos);
  }
  
  /* update the cursor */
  print_cursor(widget, win);

  if(t->enable_backing_store)
    gdk_draw_pixmap(widget->window, t->bg_gc, t->pixmap, area->x, area->y, area->x, area->y, area->width, area->height);
}

static void tracker_draw_stupid (GtkWidget *widget, GdkRectangle *area) {
  Tracker *t = TRACKER(widget);

  t->oldpos = -666;
  tracker_draw_clever(widget, area);
}

static gint tracker_expose (GtkWidget *widget, GdkEventExpose *event) {
  tracker_draw_stupid(widget, &event->area);
  return FALSE;
}

static gint tracker_idle_draw_function (Tracker *t) {
  GtkWidget *widget = GTK_WIDGET(t);
  GdkRectangle area = { 0, 0, widget->allocation.width, widget->allocation.height };

  if(GTK_WIDGET_MAPPED(GTK_WIDGET(t)))
    tracker_draw_clever(GTK_WIDGET(t), &area);
  
  if(t->idle_handler != 0)
    gtk_idle_remove(t->idle_handler);
  t->idle_handler = 0;
  return TRUE;
}

static void tracker_size_request (GtkWidget *widget, GtkRequisition *requisition) {
  Tracker *t = TRACKER(widget);

  requisition->width = TRACKER_CHANNEL_WIDTH * t->fontw + 3 * t->fontw + 10;
  requisition->height = 7 * t->fonth;
}

static void init_display (Tracker *t, int width, int height) {
  GtkWidget *widget = GTK_WIDGET(t);
  int u;
  int line_numbers_space = 3 * t->fontw;

  height -= t->fonth;
  t->disp_rows = height / t->fonth;
  if(!(t->disp_rows % 2))
    t->disp_rows--;
  t->disp_cursor = t->disp_rows / 2;
  t->disp_starty = (height - t->fonth * t->disp_rows) / 2 + t->fonth;
  
  t->disp_chanwidth = TRACKER_CHANNEL_WIDTH * t->fontw;
  u = width - line_numbers_space - 10;
  t->disp_numchans = u / t->disp_chanwidth;
  
  if(t->disp_numchans > t->num_channels)
    t->disp_numchans = t->num_channels;

  t->disp_startx = (u - t->disp_numchans * t->disp_chanwidth) / 2 + line_numbers_space + 5;
  adjust_xpanning(t);

  if(t->curpattern) {
    g_signal_emit(GTK_OBJECT(t), tracker_signals[PATPOS_SIGNAL], 0, t->patpos, t->curpattern->length, t->disp_rows);
    g_signal_emit(GTK_OBJECT(t), tracker_signals[XPANNING_SIGNAL], 0, t->leftchan, t->num_channels, t->disp_numchans);
  }
  
  if(t->enable_backing_store) {
    if(t->pixmap)
      gdk_pixmap_unref(t->pixmap);
    t->pixmap = gdk_pixmap_new(widget->window, widget->allocation.width, widget->allocation.height, -1);
    CLEAR(t->pixmap, 0, 0, widget->allocation.width, widget->allocation.height);
  }
}

static void tracker_size_allocate (GtkWidget *widget, GtkAllocation *allocation) {
  Tracker *t;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (IS_TRACKER (widget));
  g_return_if_fail (allocation != NULL);
  
  widget->allocation = *allocation;
  if (GTK_WIDGET_REALIZED (widget)) {
    t = TRACKER (widget);
    
    gdk_window_move_resize (widget->window,
			    allocation->x, allocation->y,
			    allocation->width, allocation->height);
    
    init_display(t, allocation->width, allocation->height);
  }
}

void tracker_reset(Tracker *t) {
  GtkWidget *widget;

  g_return_if_fail(t != NULL);
  
  widget=GTK_WIDGET(t);
  if(GTK_WIDGET_REALIZED(widget)) {
    t->patpos = 0;
    t->cursor_ch = 0;
    t->cursor_item = 0;
    t->leftchan = 0;
    init_display(t, widget->allocation.width, widget->allocation.height);
    adjust_xpanning(t);
    gtk_widget_queue_draw(GTK_WIDGET(t));
  }
}

static void init_colors (GtkWidget *widget) {
  int n;
  const int *p;
  GdkColor *c;
  
  for (n = 0, p = default_colors, c = TRACKER(widget)->colors; n < TRACKERCOL_LAST; n++, c++) {
    c->red = *p++ * 65535 / 255;
    c->green = *p++ * 65535 / 255;
    c->blue = *p++ * 65535 / 255;
    c->pixel = (gulong)((c->red & 0xff00) * 256 + (c->green & 0xff00) + (c->blue & 0xff00) / 256);
    gdk_color_alloc(gtk_widget_get_colormap(widget), c);
  }
}

static void calculate_font_size(GtkWidget *widget) {
  g_return_if_fail (widget != NULL);
  g_return_if_fail (IS_TRACKER (widget));
  
  if(GTK_WIDGET_REALIZED(widget)) {
    Tracker *t = TRACKER(widget);
    PangoContext *context = gdk_pango_context_get_for_screen(gdk_drawable_get_screen(widget->window));
    PangoLayout *layout = pango_layout_new(context);
    PangoRectangle ink, logical;
    pango_layout_set_font_description (layout, t->fontdesc);
    pango_layout_set_text(layout, "0", -1);
    pango_layout_context_changed(layout);
    pango_layout_get_pixel_extents(layout, &ink, &logical);
    t->fontw = logical.width;
    t->fonth = ink.y + ink.height;
    t->fontc = ink.y / 2;
    g_object_unref(layout);
    g_object_unref(context);
  }
}

static void tracker_fonts_free(GtkWidget *widget) {
  g_return_if_fail (widget != NULL);
  g_return_if_fail (IS_TRACKER (widget));

  Tracker *t = TRACKER(widget);
  int i;
  
  if (t->fonts_normal != NULL) {
    for (i = 0; i < 39; i++)
      gdk_pixmap_unref(t->fonts_normal[i]);
    free(t->fonts_normal);
  }
  if (t->fonts_cursor != NULL) {
    for (i = 0; i < 39; i++)
      gdk_pixmap_unref(t->fonts_cursor[i]);
    free(t->fonts_cursor);
  }
  if (t->fonts_select != NULL) {
    for (i = 0; i < 39; i++)
      gdk_pixmap_unref(t->fonts_select[i]);
    free(t->fonts_select);
  }
}

static void tracker_fonts_render(GtkWidget *widget) {
  g_return_if_fail (widget != NULL);
  g_return_if_fail (IS_TRACKER (widget));

  if(GTK_WIDGET_REALIZED(widget)) {
    Tracker *t = TRACKER(widget);
    PangoContext *context;
    PangoLayout *layout;
    char buf[2] = { 0, 0 };
    int i;

    tracker_fonts_free(widget);
  
    t->fonts_normal = malloc(39 * sizeof(GdkPixmap *));
    t->fonts_cursor = malloc(39 * sizeof(GdkPixmap *));
    t->fonts_select = malloc(39 * sizeof(GdkPixmap *));
    for (i = 0; i < 39; i++) {
      buf[0] = font_to_ascii[i];
      t->fonts_normal[i] = gdk_pixmap_new(widget->window, t->fontw, t->fonth, -1);
      t->fonts_cursor[i] = gdk_pixmap_new(widget->window, t->fontw, t->fonth, -1);
      t->fonts_select[i] = gdk_pixmap_new(widget->window, t->fontw, t->fonth, -1);
      gdk_gc_set_foreground(t->misc_gc, &t->colors[TRACKERCOL_BG_SELECTION]);
      gdk_draw_rectangle(t->fonts_normal[i], t->bg_gc, TRUE, 0, 0, t->fontw, t->fonth);
      gdk_draw_rectangle(t->fonts_cursor[i], t->bg_cursor_gc, TRUE, 0, 0, t->fontw, t->fonth);
      gdk_draw_rectangle(t->fonts_select[i], t->misc_gc, TRUE, 0, 0, t->fontw, t->fonth);
      context = gdk_pango_context_get_for_screen(gdk_drawable_get_screen(t->fonts_normal[i]));
      layout = pango_layout_new(context);
      pango_layout_set_font_description (layout, t->fontdesc);
      pango_layout_set_text (layout, buf, -1);
      pango_layout_context_changed (layout);
      gdk_draw_layout(t->fonts_normal[i], t->notes_gc, 0, -t->fontc, layout);
      gdk_draw_layout(t->fonts_cursor[i], t->notes_gc, 0, -t->fontc, layout);
      gdk_draw_layout(t->fonts_select[i], t->notes_gc, 0, -t->fontc, layout);
      g_object_unref(layout);
      g_object_unref(context);
    }
  }
}

static void tracker_realize(GtkWidget *widget) {
  GdkWindowAttr attributes;
  gint attributes_mask;
  Tracker *t;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (IS_TRACKER (widget));
  
  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
  t = TRACKER(widget);
  
  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.event_mask = gtk_widget_get_events (widget) |
    GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
    GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK;
  
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);
  
  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
  widget->window = gdk_window_new (widget->parent->window, &attributes, attributes_mask);  
  widget->style = gtk_style_attach (widget->style, widget->window);

  calculate_font_size(widget);
  init_colors(widget);
  
  t->bg_gc = gdk_gc_new(widget->window);
  t->bg_cursor_gc = gdk_gc_new(widget->window);
  t->notes_gc = gdk_gc_new(widget->window);
  t->misc_gc = gdk_gc_new(widget->window);
  gdk_gc_set_foreground(t->bg_gc, &t->colors[TRACKERCOL_BG]);
  gdk_gc_set_foreground(t->bg_cursor_gc, &t->colors[TRACKERCOL_BG_CURSOR]);
  gdk_gc_set_foreground(t->notes_gc, &t->colors[TRACKERCOL_NOTES]);
  
  if(!t->enable_backing_store)
    gdk_gc_set_exposures (t->bg_gc, 1); /* man XCopyArea, grep exposures */
  
  init_display(t, attributes.width, attributes.height);
  
  gdk_window_set_user_data (widget->window, widget);
  gdk_window_set_background(widget->window, &t->colors[TRACKERCOL_BG]);
  
  tracker_fonts_render(widget);
}

gboolean tracker_set_font (Tracker *t, const gchar *fontname) {
  if (t->fontdesc != NULL)
    pango_font_description_free (t->fontdesc);
  
  t->fontdesc = pango_font_description_from_string(fontname);
  
  calculate_font_size(GTK_WIDGET(t));
  tracker_fonts_render(GTK_WIDGET(t));
  tracker_reset(t);
  
  return TRUE;
}

/* If selecting, mouse is used to select in pattern */
static void tracker_mouse_to_cursor_pos (Tracker *t, int x, int y, int *cursor_ch, int *cursor_item, int *patpos) {
  int HPatHalf;
  
  /* Calc the column (channel and pos in channel) */
  if(x < t->disp_startx) {
    if(t->leftchan)
      *cursor_ch = t->leftchan - 1;
    else
      *cursor_ch = t->leftchan;
    *cursor_item = 0;
  } else if(x > t->disp_startx + t->disp_numchans * t->disp_chanwidth) {
    if(t->leftchan + t->disp_numchans < t->num_channels) {
      *cursor_ch = t->leftchan + t->disp_numchans;
      *cursor_item = 0;
    } else {
      *cursor_ch = t->num_channels - 1;
      *cursor_item = 6;
    }
  } else {
    /* WTF */
    *cursor_ch = t->leftchan + ((x - t->disp_startx) / t->disp_chanwidth);
    *cursor_item = (x - (t->disp_startx + (*cursor_ch - t->leftchan) * t->disp_chanwidth)) / t->fontw;
    if(*cursor_item < 4)
      *cursor_item = 0;
    else if(*cursor_item == 4)
      *cursor_item = 1;
    else if(*cursor_item == 5 || *cursor_item == 6)
      *cursor_item = 2;
    else if(*cursor_item == 7)
      *cursor_item = 3;
    else if(*cursor_item == 8 || *cursor_item == 9)
      *cursor_item = 4;
    else if(*cursor_item == 10)
      *cursor_item = 5;
    else if(*cursor_item >= 11)
      *cursor_item = 6;
  }
  
  /* Calc the row */
  HPatHalf = t->disp_rows / 2;
  if(y < t->disp_starty)
    *patpos = t->patpos - HPatHalf - 1;
  else if(y > t->disp_rows * t->fonth)
    *patpos = t->patpos + HPatHalf + 1;
  else {
    *patpos = (y - t->disp_starty) / t->fonth;
    if(t->patpos <= *patpos)
      *patpos = t->patpos + *patpos - HPatHalf;
    else
      *patpos = t->patpos - (HPatHalf - *patpos);
  }
  if(*patpos < 0)
    *patpos = 0;
  else if(*patpos >= t->curpattern->length)
    *patpos = t->curpattern->length-1;
}

static gint tracker_button_press (GtkWidget *widget, GdkEventButton *event) {
  Tracker *t;
  int x, y, cursor_ch, cursor_item, patpos;
  GdkModifierType state;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (IS_TRACKER(widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);
  
  t = TRACKER(widget);
  gdk_window_get_pointer (event->window, &x, &y, &state);
  
  if(t->mouse_selecting && event->button != 1) {
    t->mouse_selecting = 0;
  } else if(!t->mouse_selecting) {
    t->button = event->button;		
    gdk_window_get_pointer (event->window, &x, &y, &state);
    if(t->button == 1) {
      /* Start selecting block */
      g_signal_emit(GTK_OBJECT(t), tracker_signals[BLOCKMARK_SET_SIGNAL], 0, 1);
      t->inSelMode = FALSE;
      tracker_mouse_to_cursor_pos(t, x, y, &t->sel_start_ch, &cursor_item, &t->sel_start_row);
      t->sel_end_row = t->sel_start_row;
      t->sel_end_ch = t->sel_start_ch;
      t->mouse_selecting = 1;
      tracker_redraw(t);
    } else if(t->button == 2) {
      /* Tracker cursor posititioning and clear block mark if any */
      if(t->inSelMode || t->sel_start_ch != -1) {
	g_signal_emit(GTK_OBJECT(t), tracker_signals[BLOCKMARK_SET_SIGNAL], 0, 0);
	t->sel_start_ch = t->sel_end_ch = -1;
	t->sel_start_row = t->sel_end_row = -1;
	t->inSelMode = FALSE;			
	tracker_redraw(t);
      }
      tracker_mouse_to_cursor_pos(t, x, y, &cursor_ch, &cursor_item, &patpos);
      if(cursor_ch != t->cursor_ch || cursor_item != t->cursor_item) {
	t->cursor_ch = cursor_ch;
	t->cursor_item = cursor_item;
	adjust_xpanning(t);
      }
      if(patpos != t->patpos)
	tracker_set_patpos(t, patpos);
      gtk_widget_queue_draw(GTK_WIDGET(t));
    }
  }
  return TRUE;
}

static gint tracker_motion_notify (GtkWidget *widget, GdkEventMotion *event) {
  Tracker *t;
  int x, y, cursor_item;
  GdkModifierType state;
  
  t = TRACKER(widget);
  
  if(!t->mouse_selecting)
    return TRUE;
  
  if (event->is_hint) {
    gdk_window_get_pointer (event->window, &x, &y, &state);
  } else {
    x = event->x;
    y = event->y;
    state = event->state;
  }
  
  if((state & GDK_BUTTON1_MASK) && t->mouse_selecting) {
    tracker_mouse_to_cursor_pos(t, x, y, &t->sel_end_ch, &cursor_item, &t->sel_end_row);
    
    if(x > t->disp_startx + t->disp_numchans*t->disp_chanwidth && t->leftchan + t->disp_numchans < t->num_channels)	{
      t->sel_end_ch++;
      tracker_set_xpanning(t, t->leftchan + 1);
    } else if(x < t->disp_startx && t->leftchan > 0) {
      t->sel_end_ch--;
      tracker_set_xpanning(t, t->leftchan - 1);
    }	
    if((t->sel_end_row > t->patpos + (t->disp_rows/2)) || (y > t->widget.allocation.height && t->patpos < t->sel_end_row))
      tracker_set_patpos(t, t->patpos+1);
    else if((t->sel_end_row < t->patpos - (t->disp_rows/2)) || (y <= 0 && t->patpos > t->sel_end_row))
      tracker_set_patpos(t, t->patpos-1);
    tracker_redraw(t);
  }
  
  return TRUE;
}

static gint tracker_button_release (GtkWidget *widget, GdkEventButton *event) {
  Tracker *t;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (IS_TRACKER (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);
  
  t = TRACKER(widget);
  
  if(t->mouse_selecting && event->button == 1) {
    t->mouse_selecting = 0;
    g_signal_emit(GTK_OBJECT(t), tracker_signals[BLOCKMARK_SET_SIGNAL], 0, 0);
  }
  
  return TRUE;
}

static void tracker_class_init (TrackerClass *class) {
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;

  object_class = (GtkObjectClass*)class;
  widget_class = (GtkWidgetClass*)class;

  widget_class->realize = tracker_realize;
  widget_class->expose_event = tracker_expose;
  widget_class->size_request = tracker_size_request;
  widget_class->size_allocate = tracker_size_allocate;
  widget_class->button_press_event = tracker_button_press;
  widget_class->button_release_event = tracker_button_release;
  widget_class->motion_notify_event = tracker_motion_notify;

  tracker_signals[PATPOS_SIGNAL] = g_signal_new("patpos",
					      G_TYPE_FROM_CLASS(object_class),
					      G_SIGNAL_RUN_FIRST|G_SIGNAL_DETAILED,
					      G_STRUCT_OFFSET(TrackerClass, patpos),
					      NULL, NULL,
					      trackerwidget_marshal_VOID__INT_INT_INT,
					      G_TYPE_NONE, 3,
					      G_TYPE_INT,
					      G_TYPE_INT,
					      G_TYPE_INT);

  tracker_signals[XPANNING_SIGNAL] = g_signal_new("xpanning",
						G_TYPE_FROM_CLASS(object_class),
						G_SIGNAL_RUN_FIRST|G_SIGNAL_DETAILED,
						G_STRUCT_OFFSET(TrackerClass, xpanning),
						NULL, NULL,
						trackerwidget_marshal_VOID__INT_INT_INT,
						G_TYPE_NONE, 3,
						G_TYPE_INT,
						G_TYPE_INT,
						G_TYPE_INT);

  tracker_signals[BLOCKMARK_SET_SIGNAL] = g_signal_new("blockmark_set",
						     G_TYPE_FROM_CLASS(object_class),
						     G_SIGNAL_RUN_FIRST|G_SIGNAL_DETAILED,
						     G_STRUCT_OFFSET(TrackerClass, blockmark_set),
						     NULL, NULL,
						     g_cclosure_marshal_VOID__INT,
						     G_TYPE_NONE, 1,
						     G_TYPE_INT);
  
  class->patpos = NULL;
  class->xpanning = NULL;
  class->blockmark_set = NULL;
}

static void tracker_init (Tracker *t) {
  t->fontdesc = pango_font_description_from_string ("Monospace 9");
  t->oldpos = -1;
  t->curpattern = NULL;
  t->enable_backing_store = 0;
  t->pixmap = NULL;
  t->patpos = 0;
  t->cursor_ch = 0;
  t->cursor_item = 0;
  t->idle_handler = 0;

  t->inSelMode = FALSE;
  t->sel_end_row = -1;
  t->sel_end_ch = -1;
  t->sel_start_row = -1;
  t->sel_start_ch = -1;
  t->mouse_selecting = 0;
  t->button = -1;
}

GType tracker_get_type() {
  static GType tracker_type = 0;

  if(!tracker_type) {
    static const GTypeInfo tracker_info = {
      sizeof(TrackerClass),
      NULL,
      NULL,
      (GClassInitFunc)tracker_class_init,
      NULL,
      NULL,
      sizeof(Tracker),
      0,
      (GInstanceInitFunc)tracker_init,
    };
    
    tracker_type = g_type_register_static(GTK_TYPE_WIDGET, "Tracker", &tracker_info, 0);
  }
  
  return tracker_type;
}

GtkWidget* tracker_new() {
  return GTK_WIDGET(g_object_new(tracker_get_type(), NULL));
}
