/* editor.c
 *
 * Copyright 2002-2005 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <string.h>
#include "tutka.h"
#include "conversion.h"
#include "editor.h"
#include "midi.h"
#include "player.h"
#include "callbacks.h"
#ifdef HAVE_JACK
#include "jack.h"
#endif

/* Initializes the editor */
struct editor *editor_open()
{
  struct editor *editor = (struct editor *)calloc(1, sizeof(struct editor));

  /* Use GConf */
  editor->gconfclient = gconf_client_get_default();
  gconf_client_add_dir(editor->gconfclient, "/apps/tutka", GCONF_CLIENT_PRELOAD_NONE, NULL);

  /* Set defaults */
  editor->space = 1;

  /* Get a pointer to the main thread and lower priority to give time to the player */
  editor->thread = g_thread_self();
  g_thread_set_priority(editor->thread, G_THREAD_PRIORITY_LOW);

  /* Create a mutex for locking the song */
  editor->songmutex = g_mutex_new();

#ifdef HAVE_JACK
  /* Open the Jack subsystem */
  editor->jack = jack_open(editor);
#endif

  /* Open the GUI subsystem */
  editor->gui = gui_open(editor);
  if (editor->gui == NULL) {
    editor_close(editor);
    return NULL;
  }

  /* Open the MIDI subsystem */
  editor->midi = midi_open(editor, editor_midi_interface_get(editor, DIRECTION_IN, MIDIINTERFACES_AVAILABLE), editor_midi_interface_get(editor, DIRECTION_OUT, MIDIINTERFACES_AVAILABLE), editor_midi_interface_get(editor, DIRECTION_IN, MIDIINTERFACES_UNAVAILABLE), editor_midi_interface_get(editor, DIRECTION_OUT, MIDIINTERFACES_UNAVAILABLE));

  /* Some interfaces may have been removed so refresh preferences */
  editor_midi_interface_set_from_current(editor);

  /* Refresh the preferences window */
  gui_preferences_refresh(editor->gui);

  editor->gconfnotifyid = gconf_client_notify_add(editor->gconfclient, "/apps/tutka", gconf_notify_function, editor->gui, NULL, NULL);

  return editor;
}

/* Opens an existing song or creates new */
void editor_song_open(struct editor *editor, char *filename)
{
  nullcheck_void(editor, editor_song_open);

  /* Close the old player */
  if (editor->player != NULL)
    player_close(editor->player);

  /* Free the old song */
  if (editor->song != NULL)
    song_free(editor->song);
  if (editor->filename != NULL)
    free(editor->filename);

  /* If a filename was supplied open the file */
  if (filename != NULL) {
    gchar *path;
    FILE *file = fopen(filename, "r");
    unsigned char data[4] = { 0, 0, 0, 0 };

    editor->filename = strdup(filename);

    /* Read first four bytes for checking the file type */
    if (file != NULL) {
      fread(data, sizeof(unsigned char), 4, file);
      fclose(file);
    }

    /* Is it an MMD2 file? */
    if (data[0] == (ID_MMD2 >> 24) && data[1] == ((ID_MMD2 >> 16) & 0xff) && data[2] == ((ID_MMD2 >> 8) & 0xff) && data[3] == (ID_MMD2 & 0xff)) {
      struct MMD2 *mmd;
      mmd = MMD2_load(filename);
      editor->song = MMD2_convert_song(mmd);
      MMD2_free(mmd);
    } else
      editor->song = song_load(filename);

    /* Just allocate a new song if the song couldn't be opened */
    if (editor->song == NULL) {
      /* Figure out default MIDI interface number for new instruments */
      int i = 0;
      if (editor->midi->numoutputs > 1)
	i = 1;

      editor->song = song_alloc();
      song_check_instrument(editor->song, 0, i);
    } else
      editor_song_midi_interface_numbers_set(editor);

    /* Store the path in GConf for further use */
    path = g_path_get_dirname(filename);
    path = g_realloc(path, strlen(path) + 2);
    path[strlen(path) + 1] = 0;
    path[strlen(path)] = '/';
    gconf_client_set_string(editor->gconfclient, "/apps/tutka/song_path", path, NULL);
    g_free(path);
  } else {
    /* Figure out default MIDI interface number for new instruments */
    int i = 0;
    if (editor->midi->numoutputs > 1)
      i = 1;

    editor->filename = NULL;
    editor->song = song_alloc();
    song_check_instrument(editor->song, 0, i);
  }

  /* Reset editor location */
  editor->section = 0;
  editor->playseq = 0;
  editor->position = 0;
  editor->block = 0;
  editor->line = 0;
  editor->time = 0;
  editor->cmdpage = 0;

  /* Open the player */
  editor->player = player_open(editor->song, editor, editor->midi);
  player_set_scheduler(editor->player, editor_scheduler_get(editor));

  /* Refresh the GUI */
  gui_refresh_all(editor->gui);
  gui_set_saveas_filename(editor->gui, filename);
}

/* Deinitializes the editor */
void editor_close(struct editor *editor)
{
  nullcheck_void(editor, editor_close);

  /* Close the player */
  if (editor->player != NULL)
    player_close(editor->player);

#ifdef HAVE_JACK
  /* Close Jack */
  if (editor->jack != NULL)
    jack_close(editor->jack);
#endif

  /* Close MIDI */
  if (editor->midi != NULL)
    midi_close(editor->midi);

  /* Remove the GConf notification */
  if (editor->gconfclient != NULL)
    gconf_client_notify_remove(editor->gconfclient, editor->gconfnotifyid);

  /* Close the GUI */
  if (editor->gui != NULL)
    gui_close(editor->gui);

  /* Free the mutex */
  if (editor->songmutex != NULL)
    g_mutex_free(editor->songmutex);

  /* Free the song */
  if (editor->song != NULL)
    song_free(editor->song);

  /* Free the filename */
  if (editor->filename != NULL)
    free(editor->filename);

  /* Free the copy & paste buffers */
  if (editor->copyblock != NULL)
    block_free(editor->copyblock);
  if (editor->copytrack != NULL)
    block_free(editor->copytrack);
  if (editor->copyarea != NULL)
    block_free(editor->copyarea);

  /* Unref GConf client */
  if (editor->gconfclient != NULL)
    g_object_unref(editor->gconfclient);

  free(editor);
}

/* Remaps the instruments to current set of MIDI interfaces */
void editor_song_midi_interface_numbers_set(struct editor *editor)
{
  nullcheck_void(editor, editor_song_midi_interface_numbers_set);

  int i, j;

  if (editor->song == NULL)
    return;

  for (i = 0; i < editor->song->numinstruments; i++) {
    struct instrument *instrument = editor->song->instruments[i];
    for (j = 0; j < instrument->numoutputs; j++) {
      struct instrument_output *output = instrument->outputs[j];
      output->midiinterface = midi_interface_number_get_by_name(editor->midi, output->midiinterfacename, DIRECTION_OUT);
    }
  }
}

/* Remaps the instruments to current set of MIDI interfaces */
void editor_song_midi_interface_names_set(struct editor *editor)
{
  nullcheck_void(editor, editor_song_midi_interface_names_set);

  int i, j;

  if (editor->song == NULL)
    return;

  for (i = 0; i < editor->song->numinstruments; i++) {
    struct instrument *instrument = editor->song->instruments[i];
    for (j = 0; j < instrument->numoutputs; j++) {
      struct instrument_output *output = instrument->outputs[j];
      if (output->midiinterfacename != NULL)
	free(output->midiinterfacename);
      output->midiinterfacename = midi_interface_name_get_by_number(editor->midi, output->midiinterface, DIRECTION_OUT);
    }
  }
}

/* Checks whether there have been changes in MIDI interfaces */
void editor_midi_status_check(struct editor *editor)
{
  nullcheck_void(editor, editor_midi_status_check);

  if (!editor->midi->changed)
    return;

  /* Store the old MIDI interface names */
  editor_song_midi_interface_names_set(editor);

  /* Make sure the player doesn't try to access the MIDI interfaces until they're refreshed */
  player_lock(editor->player);

  /* Refresh interfaces list */
  midi_interfaces_refresh(editor->midi);

  /* Make sure the MIDI interface numbers are still valid */
  editor_song_midi_interface_numbers_set(editor);

  /* Let the player know about the changes */
  player_midi_changed(editor->player);

  /* The player is free to go */
  player_unlock(editor->player);

  /* Refresh preferences and instrument windows */
  gui_instrument_refresh(editor->gui);
  gui_preferences_refresh(editor->gui);

  /* Update the preferences */
  editor_midi_interface_set_from_current(editor);

  editor->midi->changed = 0;
}

/* Changes the current section and refreshes the GUI accordingly */
void editor_set_section(struct editor *editor, int section)
{
  nullcheck_void(editor, editor_set_section);

  if (section != editor->section) {
    /* Bounds checking */
    if (section < 0)
      section = 0;
    if (section >= editor->song->numsections)
      section = editor->song->numsections - 1;

    editor->section = section;

    /* If called from GUI set player position, always refresh GUI */
    if (g_thread_self() == editor->thread)
      player_set_section(editor->player, section);
    gui_set_refresh(editor->gui, GUI_REFRESH_SECTIONLIST | GUI_REFRESH_INFO);
  }
}

/* Changes the position of the editor and refreshes the GUI accordingly */
void editor_set_playseq(struct editor *editor, int playseq)
{
  nullcheck_void(editor, editor_set_playseq);

  if (playseq != editor->playseq) {
    /* Bounds checking */
    if (playseq < 0)
      playseq = 0;
    if (playseq >= editor->song->numplayseqs)
      playseq = editor->song->numplayseqs - 1;

    editor->playseq = playseq;

    /* If called from GUI set player position, always refresh GUI */
    if (g_thread_self() == editor->thread)
      player_set_playseq(editor->player, playseq);
    gui_set_refresh(editor->gui, GUI_REFRESH_PLAYSEQLIST | GUI_REFRESH_PLAYSEQ_RECREATE | GUI_REFRESH_INFO);
  }
}

/* Changes the position of the editor and refreshes the GUI accordingly */
void editor_set_position(struct editor *editor, int position)
{
  nullcheck_void(editor, editor_set_position);

  if (position != editor->position) {
    struct playseq *curplayseq = editor->song->playseqs[editor->playseq];

    /* Bounds checking */
    if (position < 0)
      position = 0;
    if (position >= curplayseq->length)
      position = curplayseq->length - 1;

    editor->position = position;

    /* If called from GUI set player position, always refresh GUI */
    if (g_thread_self() == editor->thread)
      player_set_position(editor->player, position);
    gui_set_refresh(editor->gui, GUI_REFRESH_PLAYSEQ | GUI_REFRESH_INFO);
  }
}

/* Changes the position of the editor and refreshes the GUI accordingly */
void editor_set_block(struct editor *editor, int block)
{
  nullcheck_void(editor, editor_set_block);

  if (block != editor->block) {
    /* Bounds checking */
    if (block < 0)
      block = 0;
    if (block >= editor->song->numblocks)
      block = editor->song->numblocks - 1;
    while (editor->cmdpage >= editor->song->blocks[block]->commandpages)
      editor->cmdpage--;

    editor->block = block;

    /* If called from GUI set player position, always refresh GUI */
    if (g_thread_self() == editor->thread)
      player_set_block(editor->player, block);
    gui_set_refresh(editor->gui, GUI_REFRESH_BLOCKLIST | GUI_REFRESH_INFO | GUI_REFRESH_TRACKER);
  }
}

/* Changes the position of the editor and refreshes the GUI accordingly */
void editor_set_line(struct editor *editor, int line)
{
  nullcheck_void(editor, editor_set_line);

  if (line != editor->line) {
    /* Bounds checking */
    while (line < 0)
      line += editor->song->blocks[editor->block]->length;
    while (line >= editor->song->blocks[editor->block]->length)
      line -= editor->song->blocks[editor->block]->length;

    editor->line = line;

    /* If called from GUI set player position, always refresh GUI */
    if (g_thread_self() == editor->thread) {
      player_set_line(editor->player, line);
      gui_tracker_refresh(editor->gui);
    } else
      gui_set_refresh(editor->gui, GUI_REFRESH_TRACKER);
  }
}

/* Seeks to a certain number of ticks in a song and refreshes the GUI accordingly */
void editor_seek(struct editor *editor, int ticks)
{
  nullcheck_void(editor, editor_seek);

  unsigned int section, playseq, position, block, line, tick, tpl, i, j;

  tpl = editor->song->ticksperline;
  section = 0;
  playseq = editor->song->sections[section];
  position = 0;
  block = editor->song->playseqs[playseq]->blocknumbers[position];
  line = 0;
  tick = 0;

  while (ticks > 0) {
    struct block *b = editor->song->blocks[block];
    unsigned char postcommand = 0, postvalue = 0, changeblock = 0;

    /* Look for commands that alter the playing position */
    for (i = 0; i < b->tracks; i++) {
      for (j = 0; j < b->commandpages; j++) {
	unsigned char command = b->commands[j * b->tracks * b->length * 2 + (b->tracks * line + i) * 2];
	unsigned char value = b->commands[j * b->tracks * b->length * 2 + (b->tracks * line + i) * 2 + 1];

	if (tick == tpl - 1 && (command == COMMAND_END_BLOCK || command == COMMAND_PLAYSEQ_POSITION || command == COMMAND_TPL)) {
	  postcommand = command;
	  postvalue = value;
	}
      }
    }

    tick++;
    tick %= tpl;

    /* Ticks per line ticks have passed, move on */
    if (tick == 0) {
      line++;

      /* Check if a playing position altering command was used */
      switch (postcommand) {
      case COMMAND_END_BLOCK:
	line = postvalue;
	position++;
	if (position >= editor->song->playseqs[playseq]->length) {
	  position = 0;
	  section++;
	  if (section >= editor->song->numsections)
	    section = 0;
	}
	changeblock = 1;
	break;
      case COMMAND_PLAYSEQ_POSITION:
	line = 0;
	position = postvalue;
	if (position >= editor->song->playseqs[playseq]->length) {
	  position = 0;
	  section++;
	  if (section >= editor->song->numsections)
	    section = 0;
	}
	changeblock = 1;
	break;
      case COMMAND_TPL:
	tpl = postvalue;
	/* Fall through to default: */
      default:
	if (line >= b->length) {
	  line = 0;
	  position++;
	  if (position >= editor->song->playseqs[playseq]->length) {
	    position = 0;
	    section++;
	    if (section >= editor->song->numsections)
	      section = 0;
	  }
	  changeblock = 1;
	}
	break;
      }

      if (changeblock == 1) {
	if (section >= editor->song->numsections)
	  section = 0;
	playseq = editor->song->sections[section];

	if (position >= editor->song->playseqs[playseq]->length)
	  position = 0;
          
        block = editor->song->playseqs[playseq]->blocknumbers[position];
      }
    }

    ticks--;
  }
  
  /* Set the position in editor, player and gui */
  editor->section = section;
  editor->playseq = playseq;
  editor->position = position;
  editor->block = block;
  editor->line = line;
  player_set_section(editor->player, section);
  player_set_playseq(editor->player, playseq);
  player_set_position(editor->player, position);
  player_set_block(editor->player, block);
  player_set_line(editor->player, line);
  player_set_tick(editor->player, tick);
  editor_refresh_playseq_and_block(editor);
  gui_set_refresh(editor->gui, GUI_REFRESH_SECTIONLIST | GUI_REFRESH_PLAYSEQLIST | GUI_REFRESH_PLAYSEQ_RECREATE | GUI_REFRESH_PLAYSEQ | GUI_REFRESH_BLOCKLIST | GUI_REFRESH_TRACKER | GUI_REFRESH_INFO);
}

/* Changes the position of the editor and refreshes the GUI accordingly */
void editor_set_commandpage(struct editor *editor, int cmdpage)
{
  nullcheck_void(editor, editor_set_commandpage);

  if (cmdpage != editor->cmdpage) {
    struct block *curblock = editor->song->blocks[editor->block];
    /* Bounds checking */
    while (cmdpage < 0)
      cmdpage += curblock->commandpages;
    while (cmdpage >= curblock->commandpages)
      cmdpage -= curblock->commandpages;

    editor->cmdpage = cmdpage;

    gui_set_refresh(editor->gui, GUI_REFRESH_INFO | GUI_REFRESH_TRACKER);
  }
}

/* Changes the position of the editor and refreshes the GUI accordingly */
void editor_set_time(struct editor *editor, int time)
{
  nullcheck_void(editor, editor_set_time);

  editor->time = time;
  gui_set_refresh(editor->gui, GUI_REFRESH_TIMER);
}

/* Changes the position of the editor and refreshes the GUI accordingly */
void editor_set_tempo(struct editor *editor)
{
  nullcheck_void(editor, editor_set_tempo);

  gui_set_refresh(editor->gui, GUI_REFRESH_TEMPO);
}

/* Refreshes playseq from section and block from position */
void editor_refresh_playseq_and_block(struct editor *editor)
{
  nullcheck_void(editor, editor_refresh_playseq_and_block);

  struct song *song = editor->song;

  /* Bounds checking */
  if (editor->section >= song->numsections)
    editor->section = 0;

  /* Get playseq number */
  editor->playseq = song->sections[editor->section];

  /* Bounds checking */
  if (editor->position >= song->playseqs[editor->playseq]->length)
    editor->position = 0;

  editor->block = song->playseqs[editor->playseq]->blocknumbers[editor->position];

  /* If called from GUI set player position, always refresh GUI */
  if (g_thread_self() == editor->thread)
    player_set_playseq(editor->player, editor->playseq);
  gui_set_refresh(editor->gui, GUI_REFRESH_PLAYSEQLIST | GUI_REFRESH_PLAYSEQ | GUI_REFRESH_BLOCKLIST | GUI_REFRESH_INFO);
}

/* Reads MIDI input */
void editor_midi_input(struct editor *editor, unsigned char *data)
{
  nullcheck_void(editor, editor_midi_input);

  /* FIX: gui->tracker should not be used here directly (or at all) */
  Tracker *tracker = (Tracker *) glade_xml_get_widget(editor->gui->xml, "tracker");
  struct song *song = editor->song;

  switch (data[0] & 0xf0) {
  case 0x80:
    /* Note off */
    if (editor->edit) {
      if (editor->chord) {
	/* Do not allow negative chord status: it should never happen anyway though */
	if (editor->chordstatus > 0)
	  editor->chordstatus--;

	/* Move the cursor to the previous channel */
	tracker_step_cursor_channel(tracker, -1);

	/* If all notes of the chord have been released move to the next line */
	if (editor->chordstatus == 0)
	  editor_set_line(editor, editor->line + editor->space);
      }
    }
    break;
  case 0x90:
    /* Note on */
    if (editor->edit) {
      block_set_note(song->blocks[editor->block], editor->line, tracker->cursor_ch, data[1] / 12, data[1] % 12 + 1, editor->instrument + 1);
      block_set_command_full(song->blocks[editor->block], editor->line, tracker->cursor_ch, editor->cmdpage, COMMAND_VELOCITY, data[2]);
      tracker_redraw_row(tracker, editor->line);
      if (editor->chord) {
	editor->chordstatus++;

	/* Move the cursor to the next channel */
	tracker_step_cursor_channel(tracker, 1);
      } else
	editor_set_line(editor, editor->line + editor->space);
    }
    break;
  case 0xb0:
    /* MIDI controller */
    if (editor->edit && editor->recordcontrollers) {
      block_set_command_full(song->blocks[editor->block], editor->line, tracker->cursor_ch, editor->cmdpage, COMMAND_MIDI_CONTROLLERS + data[1], data[2]);
      tracker_redraw_row(tracker, editor->line);
    }
    break;
  case 0xe0:
    /* Pitch wheel */
    if (editor->edit) {
      block_set_command_full(song->blocks[editor->block], editor->line, tracker->cursor_ch, editor->cmdpage, COMMAND_PITCH_WHEEL, data[2]);
      tracker_redraw_row(tracker, editor->line);
    }
    break;
  default:
    break;
  }
}

/* Refreshes the GUI when the player has stopped */
void editor_stop(struct editor *editor)
{
  nullcheck_void(editor, editor_stop);

  gui_set_refresh(editor->gui, GUI_REFRESH_STATUSBAR);
}

/* Resets the timer to 00:00 */
void editor_reset_time(struct editor *editor)
{
  nullcheck_void(editor, editor_reset_timer);

  /* If called from GUI reset player timer */
  if (g_thread_self() == editor->thread)
    player_reset_time(editor->player, TRUE);

  editor->time = 0;
  gui_set_refresh(editor->gui, GUI_REFRESH_TIMER);
}

/* Locks the song so the player and the editor won't access it simultaneously */
void editor_song_lock(struct editor *editor)
{
  nullcheck_void(editor, editor_song_lock);

  g_mutex_lock(editor->songmutex);
}

/* Unlocks the song after it has been accessed */
void editor_song_unlock(struct editor *editor)
{
  nullcheck_void(editor, editor_song_unlock)

    g_mutex_unlock(editor->songmutex);
}

/* Checks if the maximum number of tracks in the song has changed */
int editor_song_maxtracks_check(struct editor *editor)
{
  nullcheck_int(editor, editor_song_maxtracks_check);

  int oldmax, v;

  g_mutex_lock(editor->songmutex);
  oldmax = editor->song->maxtracks;
  v = song_check_maxtracks(editor->song);

  if (v)
    player_trackstatus_create(editor->player, oldmax);
  g_mutex_unlock(editor->songmutex);

  return v;
}

/* Makes sure the current instrument is allocated in the song */
void editor_song_instrument_current_check(struct editor *editor)
{
  nullcheck_void(editor, editor_song_instrument_current_check);

  /* This routine COULD set the interface to the first subscribed interface... */
  g_mutex_lock(editor->songmutex);
  song_check_instrument(editor->song, editor->instrument, 0);
  g_mutex_unlock(editor->songmutex);
}

/* Sets the MIDI interface number of the current instrument */
void editor_song_instrument_current_midiinterface_set(struct editor *editor, unsigned short midiinterface)
{
  nullcheck_void(editor, editor_song_instrument_current_midiinterface_set);

  g_mutex_lock(editor->songmutex);
  editor->song->instruments[editor->instrument]->outputs[0]->midiinterface = midiinterface;
  g_mutex_unlock(editor->songmutex);
}

/* Returns the MIDI interface number of the current instrument */
unsigned short editor_song_instrument_current_midiinterface_get(struct editor *editor)
{
  nullcheck_int(editor, editor_song_instrument_current_midiinterface_get);

  return editor->song->instruments[editor->instrument]->outputs[0]->midiinterface;
}


/* Sets the default velocity of the current instrument */
void editor_song_instrument_current_defaultvelocity_set(struct editor *editor, unsigned char defaultvelocity)
{
  nullcheck_void(editor, editor_song_instrument_current_defaultvelocity_set);

  g_mutex_lock(editor->songmutex);
  editor->song->instruments[editor->instrument]->outputs[0]->defaultvelocity = defaultvelocity;
  g_mutex_unlock(editor->songmutex);
}

/* Returns the default velocity of the current instrument */
unsigned char editor_song_instrument_current_defaultvelocity_get(struct editor *editor)
{
  nullcheck_int(editor, editor_song_instrument_current_defaultvelocity_get);

  return editor->song->instruments[editor->instrument]->outputs[0]->defaultvelocity;
}

/* Sets the transposing of the current instrument */
void editor_song_instrument_current_transpose_set(struct editor *editor, char transpose)
{
  nullcheck_void(editor, editor_song_instrument_current_transpose_set);

  g_mutex_lock(editor->songmutex);
  editor->song->instruments[editor->instrument]->outputs[0]->transpose = transpose;
  g_mutex_unlock(editor->songmutex);
}

/* Returns the transposing of the current instrument */
char editor_song_instrument_current_transpose_get(struct editor *editor)
{
  nullcheck_int(editor, editor_song_instrument_current_transpose_get);

  return editor->song->instruments[editor->instrument]->outputs[0]->transpose;
}

/* Sets the hold time of the current instrument */
void editor_song_instrument_current_hold_set(struct editor *editor, unsigned char hold)
{
  nullcheck_void(editor, editor_song_instrument_current_hold_set);

  g_mutex_lock(editor->songmutex);
  editor->song->instruments[editor->instrument]->outputs[0]->hold = hold;
  g_mutex_unlock(editor->songmutex);
}

/* Returns the hold time of the current instrument */
unsigned char editor_song_instrument_current_hold_get(struct editor *editor)
{
  nullcheck_int(editor, editor_song_instrument_current_hold_get);

  return editor->song->instruments[editor->instrument]->outputs[0]->hold;
}

/* Sets the MIDI channel of the current instrument */
void editor_song_instrument_current_midichannel_set(struct editor *editor, unsigned char midichannel)
{
  nullcheck_void(editor, editor_song_instrument_current_midichannel_set);

  g_mutex_lock(editor->songmutex);
  editor->song->instruments[editor->instrument]->outputs[0]->midichannel = midichannel;
  g_mutex_unlock(editor->songmutex);
}

/* Returns the MIDI channel of the current instrument */
unsigned char editor_song_instrument_current_midichannel_get(struct editor *editor)
{
  nullcheck_int(editor, editor_song_instrument_current_midichannel_get);

  return editor->song->instruments[editor->instrument]->outputs[0]->midichannel;
}

/* Clears a part of the current block */
void editor_song_block_current_clear(struct editor *editor, int starttrack, int startline, int endtrack, int endline)
{
  nullcheck_void(editor, editor_song_block_current_clear);

  g_mutex_lock(editor->songmutex);
  block_clear(editor->song->blocks[editor->block], starttrack, startline, endtrack, endline);
  g_mutex_unlock(editor->songmutex);
}

/* Returns the length of the current block */
unsigned int editor_song_block_current_length_get(struct editor *editor)
{
  nullcheck_int(editor, editor_song_block_current_length_get);

  return editor->song->blocks[editor->block]->length;
};

/* Returns the number of tracks in the current block */
unsigned int editor_song_block_current_tracks_get(struct editor *editor)
{
  nullcheck_int(editor, editor_song_block_current_tracks_get);

  return editor->song->blocks[editor->block]->tracks;
};

/* Returns the number of command pages in the current block */
unsigned int editor_song_block_current_commandpages_get(struct editor *editor)
{
  nullcheck_int(editor, editor_song_block_current_commandpages_get);

  return editor->song->blocks[editor->block]->commandpages;
};

/* Sets a note in the current block */
void editor_song_block_current_note_set(struct editor *editor, unsigned int line, unsigned int track, unsigned char octave, unsigned char note, unsigned char instrument)
{
  nullcheck_void(editor, editor_song_block_current_note_set);

  g_mutex_lock(editor->songmutex);
  block_set_note(editor->song->blocks[editor->block], line, track, octave, note, instrument);
  g_mutex_unlock(editor->songmutex);
}

/* Gets an instrument from the current block */
unsigned char editor_song_block_current_instrument_get(struct editor *editor, unsigned int line, unsigned int track)
{
  nullcheck_int(editor, editor_song_block_current_instrument_get);

  return block_get_instrument(editor->song->blocks[editor->block], line, track);
}

/* Sets an instrument in the current block */
void editor_song_block_current_instrument_set(struct editor *editor, unsigned int line, unsigned int track, unsigned char instrument)
{
  nullcheck_void(editor, editor_song_block_current_instrument_set);

  g_mutex_lock(editor->songmutex);
  block_set_instrument(editor->song->blocks[editor->block], line, track, instrument);
  g_mutex_unlock(editor->songmutex);
}

/* Sets a command column value in the current block */
void editor_song_block_current_command_set(struct editor *editor, unsigned int line, unsigned int track, unsigned int commandpage, unsigned char slot, unsigned char data)
{
  nullcheck_void(editor, editor_song_block_current_command_set);

  g_mutex_lock(editor->songmutex);
  block_set_command(editor->song->blocks[editor->block], line, track, commandpage, slot, data);
  g_mutex_unlock(editor->songmutex);
}

/* Sets a command in the current block */
void editor_song_block_current_command_full_set(struct editor *editor, unsigned int line, unsigned int track, unsigned int commandpage, unsigned char command, unsigned char data)
{
  nullcheck_void(editor, editor_song_block_current_command_full_set);

  g_mutex_lock(editor->songmutex);
  block_set_command_full(editor->song->blocks[editor->block], line, track, commandpage, command, data);
  g_mutex_unlock(editor->songmutex);
}

/* Returns the length of the current playing sequence */
unsigned int editor_song_playseq_current_length_get(struct editor *editor)
{
  nullcheck_int(editor, editor_song_playseq_current_length_get);

  return editor->song->playseqs[editor->playseq]->length;
}

/* Returns the maximum number of tracks in the current song */
unsigned int editor_song_maxtracks_get(struct editor *editor)
{
  nullcheck_int(editor, editor_song_maxtracks_get);

  return editor->song->maxtracks;
}

/* Inserts a line to the current track in the current block */
void editor_song_block_current_insert_line(struct editor *editor, unsigned int track)
{
  nullcheck_void(editor, editor_song_block_current_insert_line);

  int i;
  struct song *song;
  struct block *b;

  g_mutex_lock(editor->songmutex);
  song = editor->song;

  /* Move lines downwards */
  b = block_copy(song->blocks[editor->block], track, editor->line, track, song->blocks[editor->block]->length - 1);
  block_paste(song->blocks[editor->block], b, track, editor->line + 1);

  /* Clear the inserted line */
  block_set_note(song->blocks[editor->block], editor->line, track, 0, 0, 0);
  for (i = 0; i < song->blocks[editor->block]->commandpages; i++)
    block_set_command_full(song->blocks[editor->block], editor->line, track, i, 0, 0);
  block_free(b);
  g_mutex_unlock(editor->songmutex);
}

/* Inserts a line to all tracks in the current block */
void editor_song_block_current_insert_line_track_all(struct editor *editor)
{
  nullcheck_void(editor, editor_song_block_current_insert_line_track_all);

  int i, j;
  struct song *song;
  struct block *b;

  g_mutex_lock(editor->songmutex);
  song = editor->song;

  /* Move lines downwards */
  b = block_copy(song->blocks[editor->block], 0, editor->line, song->blocks[editor->block]->tracks - 1, song->blocks[editor->block]->length - 1);
  block_paste(song->blocks[editor->block], b, 0, editor->line + 1);

  /* Clear the inserted line */
  for (j = 0; j < song->blocks[editor->block]->tracks; j++) {
    block_set_note(song->blocks[editor->block], editor->line, j, 0, 0, 0);
    for (i = 0; i < song->blocks[editor->block]->commandpages; i++)
      block_set_command_full(song->blocks[editor->block], editor->line, j, i, 0, 0);
  }
  block_free(b);
  g_mutex_unlock(editor->songmutex);
}

/* Deletes a line from the current track in the current block */
void editor_song_block_current_delete_line(struct editor *editor, unsigned int track)
{
  nullcheck_void(editor, editor_song_block_current_delete_line);

  int i;
  struct song *song;
  struct block *b;

  g_mutex_lock(editor->songmutex);
  song = editor->song;

  /* Move lines upwards */
  b = block_copy(song->blocks[editor->block], track, editor->line + 1, track, song->blocks[editor->block]->length - 1);
  block_paste(song->blocks[editor->block], b, track, editor->line);
  block_free(b);

  /* Clear the last line */
  block_set_note(song->blocks[editor->block], song->blocks[editor->block]->length - 1, track, 0, 0, 0);
  for (i = 0; i < song->blocks[editor->block]->commandpages; i++)
    block_set_command_full(song->blocks[editor->block], song->blocks[editor->block]->length - 1, track, i, 0, 0);
  g_mutex_unlock(editor->songmutex);
}

/* Deletes a line from all tracks in the current block */
void editor_song_block_current_delete_line_track_all(struct editor *editor)
{
  nullcheck_void(editor, editor_song_block_current_delete_line_track_all);

  int i, j;
  struct song *song;
  struct block *b;

  g_mutex_lock(editor->songmutex);
  song = editor->song;

  /* Move lines upwards */
  b = block_copy(song->blocks[editor->block], 0, editor->line + 1, song->blocks[editor->block]->tracks - 1, song->blocks[editor->block]->length - 1);
  block_paste(song->blocks[editor->block], b, 0, editor->line);
  block_free(b);

  /* Clear the last line */
  for (i = 0; i < song->blocks[editor->block]->tracks; i++) {
    block_set_note(song->blocks[editor->block], song->blocks[editor->block]->length - 1, i, 0, 0, 0);
    for (j = 0; j < song->blocks[editor->block]->commandpages; j++)
      block_set_command_full(song->blocks[editor->block], song->blocks[editor->block]->length - 1, i, j, 0, 0);
  }
  g_mutex_unlock(editor->songmutex);
}

/* Sets the volume of a track */
void editor_song_track_volume_set(struct editor *editor, unsigned int track, unsigned int volume)
{
  nullcheck_void(editor, editor_song_track_volume_set);

  g_mutex_lock(editor->songmutex);
  editor->song->tracks[track]->volume = volume;
  g_mutex_unlock(editor->songmutex);
}

/* Returns the volume of a track */
unsigned int editor_song_track_volume_get(struct editor *editor, unsigned int track)
{
  nullcheck_int(editor, editor_song_track_volume_get);

  return editor->song->tracks[track]->volume;
}

/* Sets the muting of a track */
void editor_song_track_mute_set(struct editor *editor, unsigned int track, unsigned int mute)
{
  nullcheck_void(editor, editor_song_track_mute_set);

  g_mutex_lock(editor->songmutex);
  editor->song->tracks[track]->mute = mute;
  g_mutex_unlock(editor->songmutex);

  player_lock(editor->player);
  player_stop_muted(editor->player);
  player_unlock(editor->player);
}

/* Returns the muting of a track */
unsigned int editor_song_track_mute_get(struct editor *editor, unsigned int track)
{
  nullcheck_int(editor, editor_song_track_mute_get);

  return editor->song->tracks[track]->mute;
}

/* Sets the soloing of a track */
void editor_song_track_solo_set(struct editor *editor, unsigned int track, unsigned int solo)
{
  nullcheck_void(editor, editor_song_track_solo_set);

  g_mutex_lock(editor->songmutex);
  editor->song->tracks[track]->solo = solo;
  /* The player needs to be informed whether some tracks are soloed or not */
  if (solo)
    player_set_solo(editor->player, 1);
  else
    player_check_solo(editor->player);
  g_mutex_unlock(editor->songmutex);

  player_lock(editor->player);
  player_stop_muted(editor->player);
  player_unlock(editor->player);
}

/* Returns the soloing of a track */
unsigned int editor_song_track_solo_get(struct editor *editor, unsigned int track)
{
  nullcheck_int(editor, editor_song_track_solo_get);

  return editor->song->tracks[track]->solo;
}

/* Sets the name of a track */
void editor_song_track_name_set(struct editor *editor, unsigned int track, char *name)
{
  nullcheck_void(editor, editor_song_track_name_set);

  g_mutex_lock(editor->songmutex);
  if (name != NULL)
    editor->song->tracks[track]->name = strdup(name);
  else
    editor->song->tracks[track]->name = NULL;
  g_mutex_unlock(editor->songmutex);

  /* Refresh the tracker */
  gui_tracker_refresh(editor->gui);
}

/* Returns the name of a track */
char *editor_song_track_name_get(struct editor *editor, unsigned int track)
{
  nullcheck_pointer(editor, editor_song_track_name_get);

  return editor->song->tracks[track]->name;
}

/* Copys an area of the current block to the copy buffer */
void editor_song_block_current_copyarea(struct editor *editor, int starttrack, int startline, int endtrack, int endline)
{
  nullcheck_void(editor, editor_song_block_current_copyarea);

  g_mutex_lock(editor->songmutex);
  /* Free the previous copy buffer if one exists */
  if (editor->copyarea)
    block_free(editor->copyarea);

  editor->copyarea = block_copy(editor->song->blocks[editor->block], starttrack, startline, endtrack, endline);
  g_mutex_unlock(editor->songmutex);
}

/* Pastes a copied area into the current block */
void editor_song_block_current_copyarea_paste(struct editor *editor, int track, int line)
{
  nullcheck_void(editor, editor_song_block_current_copyarea_paste);

  g_mutex_lock(editor->songmutex);
  if (editor->copyarea)
    block_paste(editor->song->blocks[editor->block], editor->copyarea, track, line);
  g_mutex_unlock(editor->songmutex);
}

/* Copys the current track of the current block to the copy buffer */
void editor_song_block_current_copytrack(struct editor *editor, int track)
{
  nullcheck_void(editor, editor_song_block_current_copytrack);

  g_mutex_lock(editor->songmutex);
  /* Free the previous copy buffer if one exists */
  if (editor->copytrack)
    block_free(editor->copytrack);

  editor->copytrack = block_copy(editor->song->blocks[editor->block], track, 0, track, editor->song->blocks[editor->block]->length - 1);
  g_mutex_unlock(editor->songmutex);
}

/* Pastes a copied track into the current block */
void editor_song_block_current_copytrack_paste(struct editor *editor, int track)
{
  nullcheck_void(editor, editor_song_block_current_copytrack_paste);

  g_mutex_lock(editor->songmutex);
  if (editor->copytrack)
    block_paste(editor->song->blocks[editor->block], editor->copytrack, track, 0);
  g_mutex_unlock(editor->songmutex);
}

/* Copys the current block to the copy buffer */
void editor_song_block_current_copyblock(struct editor *editor)
{
  nullcheck_void(editor, editor_song_block_current_copyblock);

  g_mutex_lock(editor->songmutex);
  if (editor->copyblock)
    block_free(editor->copyblock);
  editor->copyblock = block_copy(editor->song->blocks[editor->block], 0, 0, editor->song->blocks[editor->block]->tracks - 1, editor->song->blocks[editor->block]->length - 1);
  g_mutex_unlock(editor->songmutex);
}

/* Pastes a copied block into the current block */
void editor_song_block_current_copyblock_paste(struct editor *editor)
{
  nullcheck_void(editor, editor_song_block_current_copyblock_paste);

  g_mutex_lock(editor->songmutex);
  if (editor->copyblock)
    block_paste(editor->song->blocks[editor->block], editor->copyblock, 0, 0);
  g_mutex_unlock(editor->songmutex);
}

/* Returns the number of messages in the current song */
unsigned int editor_song_nummessages_get(struct editor *editor)
{
  nullcheck_int(editor, editor_song_nummessages_get);

  return editor->song->nummessages;
}

/* Inserts a message in the given position */
void editor_song_message_insert(struct editor *editor, unsigned int position)
{
  nullcheck_void(editor, editor_song_message_insert);

  g_mutex_lock(editor->songmutex);
  song_insert_message(editor->song, position);
  g_mutex_unlock(editor->songmutex);
}

/* Deletes a message from the given position */
void editor_song_message_delete(struct editor *editor, unsigned int position)
{
  nullcheck_void(editor, editor_song_message_delete);

  g_mutex_lock(editor->songmutex);
  song_delete_message(editor->song, position);
  g_mutex_unlock(editor->songmutex);
}

/* Sends a message in the given position */
void editor_song_message_send(struct editor *editor, unsigned int position)
{
  nullcheck_void(editor, editor_song_message_send);

  int i;

  /* Should midi_write_raw be used here directly? */
  g_mutex_lock(editor->songmutex);
  for (i = 1; i < editor->midi->numoutputs; i++)
    midi_write_raw(editor->midi->outputs[i], editor->song->messages[position]->data, editor->song->messages[position]->length);
  g_mutex_unlock(editor->songmutex);
}

/* Receives a message to the given position */
unsigned int editor_song_message_receive(struct editor *editor)
{
  nullcheck_int(editor, editor_song_message_receive);

  if (editor->receiveactive) {
    struct message *message = editor->song->messages[editor->receivemessage];

    /* Read as many SysEx bytes as available */
    int i = midi_read_system_exclusive(editor->midi, message, editor->receivedbytes, editor->receiveautostop);

    /* If the amount of bytes is negative last bytes have been received */
    if (i > 0)
      editor->receivedbytes += i;
    else if (i < 0) {
      /* -i is the actual number of bytes received */
      editor->receivedbytes -= i;

      on_button_receive_stop_clicked(NULL, editor->gui);
      return 1;
    } else {
      /* No bytes received: if autostop is not used and as many bytes received as requested, quit */
      if (!editor->receiveautostop && editor->receivedbytes == message->length) {
	on_button_receive_stop_clicked(NULL, editor->gui);
	return 1;
      }
    }
  }

  return 0;
}

/* Deletes the current section */
void editor_song_section_current_delete(struct editor *editor)
{
  nullcheck_void(editor, editor_song_section_current_delete);

  g_mutex_lock(editor->songmutex);
  song_delete_section(editor->song, editor->section);

  /* Make sure the current section is not beyond limits */
  if (editor->section >= editor->song->numsections)
    editor->section = editor->song->numsections - 1;
  g_mutex_unlock(editor->songmutex);
}

/* Returns the number of sections in the current song */
unsigned int editor_song_numsections_get(struct editor *editor)
{
  nullcheck_int(editor, editor_song_numsections_get);

  return editor->song->numsections;
}

/* Inserts a section to the given position */
void editor_song_section_insert(struct editor *editor, unsigned int position)
{
  nullcheck_void(editor, editor_song_section_insert);

  g_mutex_lock(editor->songmutex);
  song_insert_section(editor->song, position);
  g_mutex_unlock(editor->songmutex);
}

/* Gets the playing sequence number in the given position */
unsigned int editor_song_section_get(struct editor *editor, unsigned int position)
{
  nullcheck_int(editor, editor_song_section_get);

  return editor->song->sections[position];
}

/* Deletes the current position of the playing sequence */
void editor_song_playseq_current_delete_current(struct editor *editor)
{
  nullcheck_void(editor, editor_song_playseq_current_delete_current);

  g_mutex_lock(editor->songmutex);
  playseq_delete(editor->song->playseqs[editor->playseq], editor->position);
  editor_refresh_playseq_and_block(editor);

  /* Make sure the current playing sequence position is not beyond limits */
  if (editor->position >= editor->song->playseqs[editor->playseq]->length)
    editor->position = editor->song->playseqs[editor->playseq]->length - 1;
  g_mutex_unlock(editor->songmutex);
}

/* Inserts a block into the current playing sequence at given position */
void editor_song_playseq_current_insert(struct editor *editor, unsigned int position)
{
  nullcheck_void(editor, editor_song_playseq_current_insert);

  g_mutex_lock(editor->songmutex);
  playseq_insert(editor->song->playseqs[editor->playseq], position);
  g_mutex_unlock(editor->songmutex);
}

/* Inserts a playing sequence into the given position */
void editor_song_playseq_insert(struct editor *editor, unsigned int position)
{
  nullcheck_void(editor, editor_song_playseq_insert);

  g_mutex_lock(editor->songmutex);
  song_insert_playseq(editor->song, position);
  g_mutex_unlock(editor->songmutex);
}

/* Returns the number of playing sequences in the current song */
unsigned int editor_song_numplayseqs_get(struct editor *editor)
{
  nullcheck_int(editor, editor_song_numplayseqs_get);

  return editor->song->numplayseqs;
}

/* Deletes the current playing sequence */
void editor_song_playseq_current_delete(struct editor *editor)
{
  nullcheck_void(editor, editor_song_playseq_current_delete);

  g_mutex_lock(editor->songmutex);
  song_delete_playseq(editor->song, editor->playseq);
  editor_refresh_playseq_and_block(editor);
  g_mutex_unlock(editor->songmutex);
}

/* Returns the block number of the given position in the current playing sequence */
unsigned int editor_song_playseq_current_block_get(struct editor *editor, unsigned int position)
{
  nullcheck_int(editor, editor_song_playseq_current_block_get);

  return editor->song->playseqs[editor->playseq]->blocknumbers[position];
}

/* Inserts a block in the given position of the current song */
void editor_song_block_insert(struct editor *editor, unsigned int position, unsigned int block)
{
  nullcheck_void(editor, editor_song_block_insert);

  g_mutex_lock(editor->songmutex);
  song_insert_block(editor->song, position, block);
  g_mutex_unlock(editor->songmutex);
}

/* Returns the number of blocks in the current song */
unsigned int editor_song_numblocks_get(struct editor *editor)
{
  nullcheck_int(editor, editor_song_numblocks_get);

  return editor->song->numblocks;
}

/* Deletes the current block from the current song */
void editor_song_block_current_delete(struct editor *editor)
{
  nullcheck_void(editor, editor_song_block_current_delete);

  g_mutex_lock(editor->songmutex);
  song_delete_block(editor->song, editor->block);

  /* Make sure the current block is not beyond limits */
  if (editor->block >= editor->song->numblocks)
    editor_set_block(editor, editor->song->numblocks - 1);
  g_mutex_unlock(editor->songmutex);
}

/* Set the name of the current instrument */
void editor_song_instrument_current_name_set(struct editor *editor, char *name)
{
  nullcheck_void(editor, editor_song_instrument_current_name_set);

  g_mutex_lock(editor->songmutex);
  /* Free the old name if there is one */
  if (editor->song->instruments[editor->instrument]->name != NULL)
    free(editor->song->instruments[editor->instrument]->name);

  editor->song->instruments[editor->instrument]->name = strdup(name);
  g_mutex_unlock(editor->songmutex);
}

/* Returns the name of the current instrument */
char *editor_song_instrument_current_name_get(struct editor *editor)
{
  nullcheck_pointer(editor, editor_song_instrument_current_name_get);

  return editor->song->instruments[editor->instrument]->name;
}

/* Saves the current song */
void editor_song_save(struct editor *editor, char *filename)
{
  nullcheck_void(editor, editor_song_save);

  gchar *path;
  int l;

  /* If a filename was specified store it */
  if (filename != NULL) {
    /* Store the filename */
    if (editor->filename != NULL)
      free(editor->filename);
    editor->filename = strdup(filename);
  } else if (editor->filename == NULL)
    /* Unable to save if there was no name and no name was specified */
    return;

  /* Save in the format determined by the filename extension */
  l = strlen(editor->filename);
  g_mutex_lock(editor->songmutex);
  if (l > 4 && (strcasecmp(editor->filename + l - 3, "med") == 0 || strcasecmp(editor->filename + l - 3, "mmd") == 0 || strcasecmp(editor->filename + l - 4, "mmd2") == 0)) {
    struct MMD2 *mmd = song_convert_MMD2(editor->song);
    MMD2_save(mmd, editor->filename);
    MMD2_free(mmd);
  } else if (l > 4 && (strcasecmp(editor->filename + l - 3, "mid") == 0)) {
    struct smf *smf = song_convert_smf(editor->song);
    smf_save(smf, editor->filename);
    smf_free(smf);
  } else {
    editor_song_midi_interface_names_set(editor);
    song_save(editor->song, editor->filename);
  }
  g_mutex_unlock(editor->songmutex);

  /* Store the path in GConf for further use */
  path = g_path_get_dirname(editor->filename);
  path = g_realloc(path, strlen(path) + 2);
  path[strlen(path) + 1] = 0;
  path[strlen(path)] = '/';
  gconf_client_set_string(editor->gconfclient, "/apps/tutka/song_path", path, NULL);
  g_free(path);
}

/* Sets the name of the current song */
void editor_song_name_set(struct editor *editor, char *name)
{
  nullcheck_void(editor, editor_song_name_set);

  g_mutex_lock(editor->songmutex);
  /* Free the old name if one exists */
  if (editor->song->name != NULL)
    free(editor->song->name);

  editor->song->name = strdup(name);
  g_mutex_unlock(editor->songmutex);
}

/* Transposes the entire song */
void editor_song_transpose(struct editor *editor, int instrument, int halfnotes)
{
  nullcheck_void(editor, editor_song_transpose);

  g_mutex_lock(editor->songmutex);
  song_transpose(editor->song, instrument, halfnotes);
  g_mutex_unlock(editor->songmutex);
}

/* Transposes the current block */
void editor_song_block_current_transpose(struct editor *editor, int instrument, int halfnotes, int starttrack, int startline, int endtrack, int endline)
{
  nullcheck_void(editor, editor_block_current_transpose);

  g_mutex_lock(editor->songmutex);
  block_transpose(editor->song->blocks[editor->block], instrument, halfnotes, starttrack, startline, endtrack, endline);
  g_mutex_unlock(editor->songmutex);
}

/* Expands/shrinks the entire song */
void editor_song_expandshrink(struct editor *editor, int factor)
{
  nullcheck_void(editor, editor_block_song_expandshrink);

  g_mutex_lock(editor->songmutex);
  song_expandshrink(editor->song, factor);
  g_mutex_unlock(editor->songmutex);
}

/* Expands/shrinks the current block */
void editor_song_block_current_expandshrink(struct editor *editor, int factor, int starttrack, int startline, int endtrack, int endline)
{
  nullcheck_void(editor, editor_block_current_expandshrink);

  g_mutex_lock(editor->songmutex);
  block_expandshrink(editor->song->blocks[editor->block], factor, starttrack, startline, endtrack, endline);
  g_mutex_unlock(editor->songmutex);
}

/* Changes an instrument in the entire song */
void editor_song_changeinstrument(struct editor *editor, int from, int to, int swap)
{
  nullcheck_void(editor, editor_block_current_changeinstrument);

  g_mutex_lock(editor->songmutex);
  song_changeinstrument(editor->song, from, to, swap);
  g_mutex_unlock(editor->songmutex);
}

/* Changes an instrument in the current block */
void editor_song_block_current_changeinstrument(struct editor *editor, int from, int to, int swap, int starttrack, int startline, int endtrack, int endline)
{
  nullcheck_void(editor, editor_block_current_changeinstrument);

  g_mutex_lock(editor->songmutex);
  block_changeinstrument(editor->song->blocks[editor->block], from, to, swap, starttrack, startline, endtrack, endline);
  g_mutex_unlock(editor->songmutex);
}

/* Checks that an instrument has been allocated in the current song */
void editor_song_instrument_check(struct editor *editor, unsigned int instrument)
{
  nullcheck_void(editor, editor_song_instrument_check);

  /* This routine COULD check which one is the first subscribed MIDI interface... */
  g_mutex_lock(editor->songmutex);
  song_check_instrument(editor->song, instrument, 0);
  g_mutex_unlock(editor->songmutex);
}

/* Sets the send sync flag in the current song */
void editor_song_sendsync_set(struct editor *editor, unsigned int sendsync)
{
  nullcheck_void(editor, editor_song_sendsync_set);

  g_mutex_lock(editor->songmutex);
  editor->song->sendsync = sendsync;
  g_mutex_unlock(editor->songmutex);
}

/* Sets the autosend flag in a MIDI message */
void editor_song_message_autosend_set(struct editor *editor, unsigned int message, unsigned int autosend)
{
  nullcheck_void(editor, editor_song_message_autosend_set);

  g_mutex_lock(editor->songmutex);
  message_set_autosend(editor->song->messages[message], autosend);
  g_mutex_unlock(editor->songmutex);
}

/* Saves a MIDI message */
void editor_song_message_save(struct editor *editor, unsigned int message, char *filename)
{
  nullcheck_void(editor, editor_song_message_save);
  gchar *path;

  /* Save the message */
  g_mutex_lock(editor->songmutex);
  message_save_binary(editor->song->messages[message], filename);
  g_mutex_unlock(editor->songmutex);

  /* Store the path in GConf for further use */
  path = g_path_get_dirname(filename);
  path = g_realloc(path, strlen(path) + 2);
  path[strlen(path) + 1] = 0;
  path[strlen(path)] = '/';
  gconf_client_set_string(editor->gconfclient, "/apps/tutka/message_path", path, NULL);
  g_free(path);
}

/* Loads a MIDI message */
void editor_song_message_load(struct editor *editor, unsigned int message, char *filename)
{
  nullcheck_void(editor, editor_song_message_load);
  gchar *path;

  /* Load the message */
  g_mutex_lock(editor->songmutex);
  message_load_binary(editor->song->messages[message], filename);
  g_mutex_unlock(editor->songmutex);

  /* Store the path in GConf for further use */
  path = g_path_get_dirname(filename);
  path = g_realloc(path, strlen(path) + 2);
  path[strlen(path) + 1] = 0;
  path[strlen(path)] = '/';
  gconf_client_set_string(editor->gconfclient, "/apps/tutka/message_path", path, NULL);
  g_free(path);
}

/* Mutes the current song */
void editor_song_mute(struct editor *editor)
{
  nullcheck_void(editor, editor_song_mute);
  int i, unmute = 1;

  /* Check if all tracks are already muted */
  g_mutex_lock(editor->songmutex);
  for (i = 0; i < editor->song->maxtracks; i++)
    if (!editor->song->tracks[i]->mute) {
      unmute = 0;
      break;
    }

  if (unmute)
    for (i = 0; i < editor->song->maxtracks; i++)
      editor->song->tracks[i]->mute = 0;
  else
    for (i = 0; i < editor->song->maxtracks; i++)
      editor->song->tracks[i]->mute = 1;
  g_mutex_unlock(editor->songmutex);

  player_lock(editor->player);
  player_stop_muted(editor->player);
  player_unlock(editor->player);
}

/* Sets the playing sequence of the current section */
void editor_song_section_current_set(struct editor *editor, unsigned int playseq)
{
  nullcheck_void(editor, editor_song_section_current_set);

  g_mutex_lock(editor->songmutex);
  editor->song->sections[editor->section] = playseq;
  g_mutex_unlock(editor->songmutex);
}

/* Returns the name of a playing sequence */
char *editor_song_playseq_name_get(struct editor *editor, unsigned int playseq)
{
  nullcheck_pointer(editor, editor_song_playseq_name_get);

  return editor->song->playseqs[playseq]->name;
}

/* Sets the name of the current playing sequence */
void editor_song_playseq_current_name_set(struct editor *editor, char *name)
{
  nullcheck_void(editor, editor_song_playseq_current_name_set);

  g_mutex_lock(editor->songmutex);
  /* Free the old name if one exists */
  if (editor->song->playseqs[editor->playseq]->name)
    free(editor->song->playseqs[editor->playseq]->name);

  editor->song->playseqs[editor->playseq]->name = (char *)strdup(name);
  g_mutex_unlock(editor->songmutex);
}

/* Returns the name of the current playing sequence */
char *editor_song_playseq_current_name_get(struct editor *editor)
{
  nullcheck_pointer(editor, editor_song_playseq_current_name_get);

  return editor->song->playseqs[editor->playseq]->name;
}

/* Sets the block of the current playing sequence position */
void editor_song_playseq_current_set(struct editor *editor, unsigned int block)
{
  nullcheck_void(editor, editor_song_playseq_current_set);

  g_mutex_lock(editor->songmutex);
  playseq_set(editor->song->playseqs[editor->playseq], editor->position, block);
  g_mutex_unlock(editor->songmutex);
}

/* Sets the name of the current block */
void editor_song_block_current_name_set(struct editor *editor, char *name)
{
  nullcheck_void(editor, editor_song_block_current_name_set);

  g_mutex_lock(editor->songmutex);
  block_set_name(editor->song->blocks[editor->block], name);
  g_mutex_unlock(editor->songmutex);
}

/* Returns the name of the current block */
char *editor_song_block_current_name_get(struct editor *editor)
{
  nullcheck_pointer(editor, editor_song_block_current_name_get);

  return editor->song->blocks[editor->block]->name;
}

/* Returns the name of a block */
char *editor_song_block_name_get(struct editor *editor, unsigned int block)
{
  nullcheck_pointer(editor, editor_song_block_name_get);

  return editor->song->blocks[block]->name;
}

/* Sets the number of tracks in the current block */
void editor_song_block_current_tracks_set(struct editor *editor, unsigned int tracks)
{
  nullcheck_void(editor, editor_song_block_current_tracks_set);

  g_mutex_lock(editor->songmutex);
  block_set_tracks(editor->song->blocks[editor->block], tracks);
  g_mutex_unlock(editor->songmutex);
}

/* Returns the number of tracks in a block */
unsigned int editor_song_block_tracks_get(struct editor *editor, unsigned int block)
{
  nullcheck_int(editor, editor_song_block_tracks_get);

  return editor->song->blocks[block]->tracks;
}

/* Sets the length of the current block */
void editor_song_block_current_length_set(struct editor *editor, unsigned int length)
{
  int i = editor->line;

  nullcheck_void(editor, editor_song_block_current_length_set);

  g_mutex_lock(editor->songmutex);
  block_set_length(editor->song->blocks[editor->block], length);
  editor->line = -1;
  editor_set_line(editor, i);
  g_mutex_unlock(editor->songmutex);
}

/* Returns the length of a block */
unsigned int editor_song_block_length_get(struct editor *editor, unsigned int block)
{
  nullcheck_int(editor, editor_song_block_length_get);

  return editor->song->blocks[block]->length;
}

/* Sets the number of command pages in the current block */
void editor_song_block_current_commandpages_set(struct editor *editor, unsigned int commandpages)
{
  int i = editor->cmdpage;

  nullcheck_void(editor, editor_song_block_current_commandpages_set);

  g_mutex_lock(editor->songmutex);
  block_set_commandpages(editor->song->blocks[editor->block], commandpages);
  editor->cmdpage = -1;
  editor_set_commandpage(editor, i);
  g_mutex_unlock(editor->songmutex);
}

/* Returns the number of command pages in a block */
unsigned int editor_song_block_commandpages_get(struct editor *editor, unsigned int block)
{
  nullcheck_int(editor, editor_song_block_commandpages_get);

  return editor->song->blocks[block]->commandpages;
}

/* Sets the name of a MIDI message */
void editor_song_message_name_set(struct editor *editor, unsigned int message, char *name)
{
  nullcheck_void(editor, editor_song_message_name_set);

  g_mutex_lock(editor->songmutex);
  if (editor->song->messages[message]->name)
    free(editor->song->messages[message]->name);

  editor->song->messages[message]->name = (char *)strdup(name);
  g_mutex_unlock(editor->songmutex);
}

/* Returns the name of a MIDI message */
char *editor_song_message_name_get(struct editor *editor, unsigned int message)
{
  nullcheck_pointer(editor, editor_song_message_name_get);

  return editor->song->messages[message]->name;
}

/* Sets the length of a MIDI message */
void editor_song_message_length_set(struct editor *editor, unsigned int message, unsigned int length)
{
  nullcheck_void(editor, editor_song_message_length_set);

  g_mutex_lock(editor->songmutex);
  message_set_length(editor->song->messages[message], length);
  g_mutex_unlock(editor->songmutex);
}

/* Returns the length of a MIDI message */
unsigned int editor_song_message_length_get(struct editor *editor, unsigned int message)
{
  nullcheck_int(editor, editor_song_message_length_get);

  return editor->song->messages[message]->length;
}

/* Returns the autosend flag of a MIDI message */
unsigned int editor_song_message_autosend_get(struct editor *editor, unsigned int message)
{
  nullcheck_int(editor, editor_song_message_autosend_get);

  return editor->song->messages[message]->autosend;
}

/* Prints the song */
void editor_song_print(struct editor *editor)
{
  nullcheck_void(editor, editor_song_print);

  g_mutex_lock(editor->songmutex);
  song_print(editor->song, NULL);
  g_mutex_unlock(editor->songmutex);
}

/* Sets the tempo of the song */
void editor_song_tempo_set(struct editor *editor, unsigned int tempo)
{
  nullcheck_void(editor, editor_song_tempo_set);

  g_mutex_lock(editor->songmutex);
  editor->song->tempo = tempo;
  g_mutex_unlock(editor->songmutex);
}

/* Returns the tempo of the song */
unsigned int editor_song_tempo_get(struct editor *editor)
{
  nullcheck_int(editor, editor_song_tempo_get);

  return editor->song->tempo;
}

/* Sets the number of ticks per line in the song */
void editor_song_ticksperline_set(struct editor *editor, unsigned int ticksperline)
{
  nullcheck_void(editor, editor_song_ticksperline_set);

  g_mutex_lock(editor->songmutex);
  editor->song->ticksperline = ticksperline;
  g_mutex_unlock(editor->songmutex);
}

/* Returns the number of ticks per line in the song */
unsigned int editor_song_ticksperline_get(struct editor *editor)
{
  nullcheck_int(editor, editor_song_ticksperline_get);

  return editor->song->ticksperline;
}

/* Returns the current block (for the tracker widget only) */
struct block *editor_song_block_current_get(struct editor *editor)
{
  nullcheck_pointer(editor, editor_song_block_current_get);

  return editor->song->blocks[editor->block];
}

/* Returns the name of the song */
char *editor_song_name_get(struct editor *editor)
{
  nullcheck_pointer(editor, editor_song_name_get);

  return editor->song->name;
}

/* Returns the send sync flag of the song */
unsigned int editor_song_sendsync_get(struct editor *editor)
{
  nullcheck_int(editor, editor_song_sendsync_get);

  return editor->song->sendsync;
}

/* Sets the scheduling mode */
void editor_scheduler_set(struct editor *editor, unsigned int mode)
{
  nullcheck_void(editor, editor_scheduler_set);

  gconf_client_set_int(editor->gconfclient, "/apps/tutka/scheduling_mode", mode, NULL);
  editor_player_set_scheduler(editor);
}

/* Returns the scheduling mode */
unsigned int editor_scheduler_get(struct editor *editor)
{
  nullcheck_int(editor, editor_scheduler_get);

  int scheduler = gconf_client_get_int(editor->gconfclient, "/apps/tutka/scheduling_mode", NULL);

  /* For backwards compatibility: don't allow the scheduler to be 0 */
  if (scheduler == SCHED_NONE) {
    scheduler = SCHED_RTC;
    editor_scheduler_set(editor, scheduler);
  }

  return scheduler;
}

/* Returns the tracker font name */
char *editor_tracker_font_get(struct editor *editor)
{
  nullcheck_pointer(editor, editor_tracker_font_get);

  return gconf_client_get_string(editor->gconfclient, "/apps/tutka/tracker_font", NULL);
}

/* Sets the names of the MIDI interfaces to enable separated by , */
void editor_midi_interface_set(struct editor *editor, char *midi_interface, unsigned int direction, unsigned int mode)
{
  nullcheck_void(editor, editor_midi_interface_set);

  if (mode == MIDIINTERFACES_AVAILABLE) {
    if (direction == DIRECTION_IN)
      gconf_client_set_string(editor->gconfclient, "/apps/tutka/midi_interfaces/in", midi_interface, NULL);
    else
      gconf_client_set_string(editor->gconfclient, "/apps/tutka/midi_interfaces/out", midi_interface, NULL);
  } else {
    if (direction == DIRECTION_IN)
      gconf_client_set_string(editor->gconfclient, "/apps/tutka/midi_interfaces/unavailable_in", midi_interface, NULL);
    else
      gconf_client_set_string(editor->gconfclient, "/apps/tutka/midi_interfaces/unavailable_out", midi_interface, NULL);
  }
}

/* Returns the names of the MIDI interfaces to enable separated by , */
char *editor_midi_interface_get(struct editor *editor, unsigned int direction, unsigned int mode)
{
  nullcheck_pointer(editor, editor_midi_interface_get);

  if (mode == MIDIINTERFACES_AVAILABLE) {
    if (direction == DIRECTION_IN)
      return gconf_client_get_string(editor->gconfclient, "/apps/tutka/midi_interfaces/in", NULL);
    else
      return gconf_client_get_string(editor->gconfclient, "/apps/tutka/midi_interfaces/out", NULL);
  } else {
    if (direction == DIRECTION_IN)
      return gconf_client_get_string(editor->gconfclient, "/apps/tutka/midi_interfaces/unavailable_in", NULL);
    else
      return gconf_client_get_string(editor->gconfclient, "/apps/tutka/midi_interfaces/unavailable_out", NULL);
  }
}

/* Removes the last character from a string if the string is not null */
static char *string_shorten(char *in)
{
  if (in == NULL)
    return (char *)calloc(1, sizeof(char));

  int l = strlen(in);
  if (l == 0)
    return (char *)calloc(1, sizeof(char));

  char *out = (char *)calloc(l, sizeof(char));
  memcpy(out, in, l - 1);
  return out;
}

/* Marks the enabled interfaces as enabled in the preferences */
void editor_midi_interface_set_from_current(struct editor *editor)
{
  nullcheck_void(editor, editor_midi_interface_preferences_refresh);

  /* Set the MIDI interfaces */
  char *outputs_ok = string_shorten(editor->midi->outputs_ok);
  char *inputs_ok = string_shorten(editor->midi->inputs_ok);
  char *outputs_error = string_shorten(editor->midi->outputs_error);
  char *inputs_error = string_shorten(editor->midi->inputs_error);
  editor_midi_interface_set(editor, inputs_ok, DIRECTION_IN, MIDIINTERFACES_AVAILABLE);
  editor_midi_interface_set(editor, outputs_ok, DIRECTION_OUT, MIDIINTERFACES_AVAILABLE);
  editor_midi_interface_set(editor, inputs_error, DIRECTION_IN, MIDIINTERFACES_UNAVAILABLE);
  editor_midi_interface_set(editor, outputs_error, DIRECTION_OUT, MIDIINTERFACES_UNAVAILABLE);
}

/* Enables or disables a MIDI interface by client and port */
unsigned int editor_midi_interface_status_set(struct editor *editor, unsigned int client, unsigned int port, unsigned int direction, unsigned int enabled)
{
  nullcheck_int(editor, editor_midi_interface_status_set);

  unsigned int result;

  if (enabled)
    /* Subscribe */
    result = midi_subscribe(editor->midi, client, port, direction);
  else {
    /* Store the old MIDI interface names */
    editor_song_midi_interface_names_set(editor);

    /* Unsubscribe */
    result = 1 - midi_unsubscribe(editor->midi, client, port, direction);

    /* Make sure the MIDI interface numbers are still valid */
    editor_song_midi_interface_numbers_set(editor);
  }

  /* Refresh the instrument window */
  gui_instrument_refresh(editor->gui);

  /* Update the preferences */
  editor_midi_interface_set_from_current(editor);

  return result;
}

/* Returns the song path from the preferences */
char *editor_song_path_get(struct editor *editor)
{
  nullcheck_pointer(editor, editor_song_path_get);

  return gconf_client_get_string(editor->gconfclient, "/apps/tutka/song_path", NULL);
}

/* Returns the message path from the preferences */
char *editor_message_path_get(struct editor *editor)
{
  nullcheck_pointer(editor, editor_message_path_get);

  return gconf_client_get_string(editor->gconfclient, "/apps/tutka/message_path", NULL);
}

/* Starts the player */
void editor_player_start(struct editor *editor, unsigned int mode, int cont)
{
  nullcheck_void(editor, editor_player_start);

  player_start(editor->player, mode, editor->section, editor->position, editor->block, cont);
  gui_statusbar_refresh(editor->gui);
}

/* Stops the player */
void editor_player_stop(struct editor *editor)
{
  nullcheck_void(editor, editor_player_stop);

  if (editor->externalsync)
    player_external_sync(editor->player, 0);
  player_stop(editor->player);
  gui_info_refresh(editor->gui);
}

/* Stops all notes playing */
void editor_player_all_notes_stop(struct editor *editor)
{
  nullcheck_void(editor, editor_player_all_notes_stop);

  player_stop_all_notes(editor->player);
}

/* Resets the pitch of each MIDI channel */
void editor_player_pitch_reset(struct editor *editor)
{
  nullcheck_void(editor, editor_player_pitch_reset);

  player_reset_pitch(editor->player);
}

/* Recreates the player's track status array */
void editor_player_trackstatus_create(struct editor *editor, unsigned int tracks)
{
  nullcheck_void(editor, editor_player_trackstatus_create);

  player_trackstatus_create(editor->player, tracks);
}

/* Returns the current playing mode */
unsigned int editor_player_mode_get(struct editor *editor)
{
  nullcheck_int(editor, editor_player_mode_get);

  return editor->player->mode;
}

/* Plays a note */
void editor_player_play_note(struct editor *editor, unsigned char note, unsigned char track)
{
  nullcheck_void(editor, editor_player_play_note);

  player_lock(editor->player);
  player_play_note(editor->player, editor->instrument, editor->octave * 12 + note, 127, track);
  player_unlock(editor->player);
}

void editor_player_set_scheduler(struct editor *editor)
{
  nullcheck_void(editor, editor_player_set_scheduler);

  /* It's possible there is no player yet */
  if (editor->player == NULL)
    return;

  if (editor->externalsync != EXTERNAL_SYNC_OFF)
    player_set_scheduler(editor->player, SCHED_EXTERNAL_SYNC);
  else {
    player_set_scheduler(editor->player, editor_scheduler_get(editor));
    /* Sync once so the sync wait is broken */
    player_external_sync(editor->player, 0);
  }
}


unsigned int editor_player_get_external_sync(struct editor *editor)
{
  nullcheck_int(editor, editor_player_get_external_sync);

  return editor->externalsync;
}

void editor_player_set_external_sync(struct editor *editor, unsigned int sync)
{
  nullcheck_void(editor, editor_player_set_external_sync);

  editor->externalsync = sync;
  editor_player_set_scheduler(editor);
}

void editor_player_external_sync(struct editor *editor, unsigned int ticks)
{
  nullcheck_void(editor, editor_player_external_sync);

  player_external_sync(editor->player, ticks);
}

unsigned int editor_player_available(struct editor *editor)
{
  nullcheck_int(editor, editor_player_available);

  return editor->player != NULL ? 1 : 0;
}
