/* midi.c
 *
 * Copyright 2002-2005 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include "tutka.h"
#include "midi.h"

/* Gets the number of available ALSA MIDI devices */
unsigned int midi_get_num_alsa_devices()
{
  int devices = 0, card = -1, err;

  if ((err = snd_card_next(&card)) < 0) {
    fprintf(stderr, "cannot determine card number: %s", snd_strerror(err));
    return devices;
  }
  if (card < 0) {
    fprintf(stderr, "no sound card found");
    return devices;
  }
  do {
    int device = -1;
    char devname[32];
    snd_ctl_t *ctl;

    snprintf(devname, 32, "hw:%d", card);
    if ((err = snd_ctl_open(&ctl, devname, 0)) < 0) {
      fprintf(stderr, "cannot open control for card %d: %s", card, snd_strerror(err));
      return devices;
    }
    for (;;) {
      if ((err = snd_ctl_rawmidi_next_device(ctl, &device)) < 0) {
	fprintf(stderr, "cannot determine device number: %s", snd_strerror(err));
	break;
      }
      if (device < 0)
	break;
      devices++;
    }
    snd_ctl_close(ctl);

    if ((err = snd_card_next(&card)) < 0) {
      fprintf(stderr, "cannot determine card number: %s", snd_strerror(err));
      break;
    }
  } while (card >= 0);

  return devices;
}

/* Returns MIDI data structures for available ALSA MIDI interfaces */
struct midi **midi_get_alsa_devices()
{
  struct midi **midiinterfaces = (struct midi **)calloc(midi_get_num_alsa_devices(), sizeof(struct midi *));
  int card = -1, err, current = 0;

  if ((err = snd_card_next(&card)) < 0) {
    fprintf(stderr, "cannot determine card number: %s", snd_strerror(err));
    return midiinterfaces;
  }
  if (card < 0) {
    fprintf(stderr, "no sound card found");
    return midiinterfaces;
  }
  do {
    snd_ctl_t *ctl;
    char devname[32];
    int device = -1;

    snprintf(devname, 32, "hw:%d", card);
    if ((err = snd_ctl_open(&ctl, devname, 0)) < 0) {
      fprintf(stderr, "cannot open control for card %d: %s", card, snd_strerror(err));
      return midiinterfaces;
    }
    for (;;) {
      if ((err = snd_ctl_rawmidi_next_device(ctl, &device)) < 0) {
	fprintf(stderr, "cannot determine device number: %s", snd_strerror(err));
	break;
      }
      if (device < 0)
	break;

      snd_rawmidi_info_t *info;
      char *name = NULL;

      snd_rawmidi_info_alloca(&info);
      snd_rawmidi_info_set_device(info, device);
      snd_rawmidi_info_set_subdevice(info, 0);
      snd_rawmidi_info_set_stream(info, SND_RAWMIDI_STREAM_OUTPUT);
      if ((err = snd_ctl_rawmidi_info(ctl, info)) < 0 && err != -ENOENT)
	fprintf(stderr, "cannot get rawmidi information %d:%d: %s", card, device, snd_strerror(err));
      if (err == -ENOENT) {
	snd_rawmidi_info_set_stream(info, SND_RAWMIDI_STREAM_INPUT);
	if ((err = snd_ctl_rawmidi_info(ctl, info)) < 0 && err != -ENOENT)
	  fprintf(stderr, "cannot get rawmidi information %d:%d: %s", card, device, snd_strerror(err));
      }
      if (err != -ENOENT)
	name = (char *)snd_rawmidi_info_get_name(info);
      snprintf(devname, 32, "hw:%d,%d", card, device);
      midiinterfaces[current++] = midi_alloc(MIDI_ALSA, devname, name);
    }
    snd_ctl_close(ctl);

    if ((err = snd_card_next(&card)) < 0) {
      fprintf(stderr, "cannot determine card number: %s", snd_strerror(err));
      break;
    }
  } while (card >= 0);

  return midiinterfaces;
}

/* Allocates a null MIDI interface */
struct midi *midi_alloc(unsigned int type, char *devicename, char *name)
{
  struct midi *midi = (struct midi *)calloc(1, sizeof(struct midi));

  if (devicename != NULL)
    midi->devicename = strdup(devicename);
  if (name != NULL)
    midi->name = strdup(name);
  midi->type = type;

  return midi;
}

/* Frees a MIDI data structure */
void midi_free(struct midi *midi)
{
  nullcheck_void(midi, midi_free);

  midi_close(midi, DIRECTION_BOTH);

  if (midi->devicename != NULL)
    free(midi->devicename);
  if (midi->name != NULL)
    free(midi->name);
  free(midi);
}

/* Opens the MIDI interfaces defined in a string */
void midi_open_defined(struct midi **midiinterfaces, unsigned int nummidiinterfaces, char *in, char *out)
{
  nullcheck_void(midiinterfaces, midi_open_defined);
  nullcheck_void(in, midi_open_defined);
  nullcheck_void(out, midi_open_defined);

  int i, j, last;
  unsigned int direction;
  char *current, *next;

  /* Close all devices in the MIDI interfaces list */
  for (i = 0; i < nummidiinterfaces; i++)
    midi_close(midiinterfaces[i], DIRECTION_BOTH);

  /* Handle both input and output */
  for (j = 0; j < 2; j++) {
    last = 0;
    if (j == 0) {
      current = in;
      next = in;
      direction = DIRECTION_IN;
    } else {
      current = out;
      next = out;
      direction = DIRECTION_OUT;
    }

    while (!last) {
      /* Find the end of the device name */
      while (*next != ';' && *next != 0)
	next++;
      if (*next == 0)
	last = 1;
      else
	*next = 0;

      /* Find the device from the MIDI interfaces list */
      for (i = 0; i < nummidiinterfaces; i++) {
	if (midiinterfaces[i]->devicename != NULL && strcmp(midiinterfaces[i]->devicename, current) == 0) {
	  /* Open the interface */
	  midi_open(midiinterfaces[i], direction);
	  break;
	}
      }

      current = ++next;
    }
  }
}

/* Opens a MIDI device in specified direction */
unsigned int midi_open(struct midi *midi, unsigned int direction)
{
  nullcheck_int(midi, midi_open);

  int retval = 1;

  /* If no direction specified or device is already open do nothing */
  if (direction == DIRECTION_NONE || midi->direction & direction)
    return 1;

  switch (midi->type) {
  case MIDI_NULL:
    {
      break;
    }
  case MIDI_ALSA:
    {
      struct midi_alsa *alsa;
      int inerr, outerr;

      /* Allocate data structure if not allocated already */
      if (midi->data == NULL)
	midi->data = calloc(1, sizeof(struct midi_alsa));

      alsa = midi->data;

      /* Open input MIDI device */
      if (direction & DIRECTION_IN) {
	inerr = snd_rawmidi_open(&alsa->in, NULL, midi->devicename, SND_RAWMIDI_NONBLOCK);
	if (inerr) {
	  fprintf(stderr, "Could not open ALSA MIDI device %s for input: %d\n", midi->devicename, inerr);
	  direction &= (~DIRECTION_IN);
	  retval = 0;
	}
      }

      /* Open output MIDI device */
      if (direction & DIRECTION_OUT) {
	outerr = snd_rawmidi_open(NULL, &alsa->out, midi->devicename, SND_RAWMIDI_NONBLOCK);
	if (outerr) {
	  fprintf(stderr, "Could not open ALSA MIDI device %s for output: %d\n", midi->devicename, outerr);
	  direction &= (~DIRECTION_OUT);
	  retval = 0;
	}
      }
      break;
    }
  case MIDI_OSS:
    {
      struct midi_oss *oss;

      /* Allocate data structure if not allocated already */
      if (midi->data == NULL)
	midi->data = calloc(1, sizeof(struct midi_oss));

      oss = midi->data;

      /* Open MIDI device:read/write, do not block (for input), auto flush */
      if (direction & DIRECTION_IN) {
	oss->in = open(midi->devicename, O_RDONLY | O_NONBLOCK);
	if (oss->in == -1) {
	  fprintf(stderr, "Could not open OSS MIDI device %s for input.\n", midi->devicename);
	  direction &= (~DIRECTION_IN);
	  retval = 0;
	}
      }
      if (direction & DIRECTION_OUT) {
	oss->out = open(midi->devicename, O_WRONLY | O_NONBLOCK);
	if (oss->out == -1) {
	  fprintf(stderr, "Could not open OSS MIDI device %s for output.\n", midi->devicename);
	  direction &= (~DIRECTION_OUT);
	  retval = 0;
	}
      }
      break;
    }
  case MIDI_BUFFER:
    {
      struct midi_buffer *buffer;

      if (direction != DIRECTION_BOTH) {
	fprintf(stderr, "midi_open(): Only bidirectional mode supported for buffers.\n");
	direction = DIRECTION_NONE;
	retval = 0;
      } else {
	midi->data = calloc(1, sizeof(struct midi_buffer));
	buffer = midi->data;
	buffer->allocated = MIDI_BUFFER_BLOCK_SIZE;
	buffer->data = malloc(buffer->allocated);
	buffer->ptr = buffer->data;
      }
      break;
    }
  }

  midi->direction |= direction;

  /* If both input and output are closed free the data structure */
  if (midi->direction == DIRECTION_NONE) {
    free(midi->data);
    midi->data = NULL;
  }

  return retval;
}

/* Closes a MIDI interface */
void midi_close(struct midi *midi, unsigned int direction)
{
  nullcheck_void(midi, midi_close);

  /* If no direction is specified do nothing */
  if (direction == DIRECTION_NONE)
    return;

  /* If the device is not open do nothing */
  if (midi->data == NULL) {
    midi->direction = DIRECTION_NONE;
    return;
  }
  
  switch (midi->type) {
  case MIDI_OSS:
    {
      struct midi_oss *oss = midi->data;

      /* Close input and output file descriptors if requested */
      if (direction & DIRECTION_IN && oss->in != -1) {
	fsync(oss->in);
	close(oss->in);
      }

      if (direction & DIRECTION_OUT && oss->out != -1) {
	fsync(oss->out);
	close(oss->out);
      }
      break;
    }
  case MIDI_ALSA:
    {
      struct midi_alsa *alsa = midi->data;

      /* Close input and output devices if requested */
      if (direction & DIRECTION_IN && alsa->in != NULL) {
	snd_rawmidi_drain(alsa->in);
	snd_rawmidi_close(alsa->in);
      }

      if (direction & DIRECTION_OUT && alsa->out != NULL) {
	snd_rawmidi_drain(alsa->out);
	snd_rawmidi_close(alsa->out);
      }
      break;
    }
  case MIDI_BUFFER:
    {
      struct midi_buffer *buffer = midi->data;

      free(buffer->data);
      break;
    }
  default:
    break;
  }

  midi->direction &= (~direction);

  /* If both input and output are closed free the data structure */
  if (midi->direction == DIRECTION_NONE) {
    free(midi->data);
    midi->data = NULL;
  }
}

/* Sets the current tick */
void midi_set_tick(struct midi *midi, unsigned int tick)
{
  nullcheck_void(midi, midi_set_tick);

  midi->tick = tick;
}

/* Reads data from a MIDI interface */
static inline int midi_read(struct midi *midi, unsigned char *data, unsigned int length)
{
  if ((midi->direction & DIRECTION_IN) != DIRECTION_IN)
    return 0;

  /* No NULL check here since it has already been checked */
  switch (midi->type) {
  case MIDI_OSS:
    {
      struct midi_oss *oss = midi->data;

      if (oss->in != -1)
	return read(oss->in, data, length);
      break;
    }
  case MIDI_ALSA:
    {
      struct midi_alsa *alsa = midi->data;

      if (alsa->in != NULL) {
	int n = snd_rawmidi_read(alsa->in, data, length);
	if (n < 0)
	  n = 0;
	return n;
      }
      break;
    }
  default:
    break;
  }

  return 0;
}

/* Writes data to the MIDI interface */
static inline void midi_write(struct midi *midi, unsigned char *data, unsigned int length)
{
  if ((midi->direction & DIRECTION_OUT) != DIRECTION_OUT)
    return;

  unsigned int delta = midi->tick - midi->oldtick;
  midi->oldtick = midi->tick;

  /* No NULL check here since it has already been checked */
  switch (midi->type) {
  case MIDI_OSS:
    {
      struct midi_oss *oss = midi->data;

      if (oss->out != -1) {
	write(oss->out, data, length);
	fsync(oss->out);
      }
      break;
    }
  case MIDI_ALSA:
    {
      struct midi_alsa *alsa = midi->data;

      if (alsa->out != NULL)
	snd_rawmidi_write(alsa->out, data, length);
      break;
    }
  case MIDI_BUFFER:
    {
      struct midi_buffer *buffer = midi->data;
      unsigned int curpos = buffer->ptr - buffer->data;
      unsigned long value = delta;
      unsigned long varlen = value & 0x7F;
      int i;

      /* Create a variable length version of the tick delta */
      while ((value >>= 7)) {
	varlen <<= 8;
	varlen |= ((value & 0x7F) | 0x80);
      }

      /* Check how many bytes the varlen version requires */
      value = varlen;
      for (i = 1; i <= 4; i++) {
	if (value & 0x80)
	  value >>= 8;
	else
	  break;
      }

      /* Allocate more space if required */
      if (curpos + length + i > buffer->allocated) {
	buffer->allocated += MIDI_BUFFER_BLOCK_SIZE;
	buffer->data = realloc(buffer->data, buffer->allocated);
	buffer->ptr = buffer->data + curpos;
      }

      /* Write the varlen tick delta value */
      for (i = 1; i <= 4; i++) {
	*buffer->ptr++ = (unsigned char)(varlen & 0xff);
	if (varlen & 0x80)
	  varlen >>= 8;
	else
	  break;
      }

      /* Write the actual payload */
      memcpy(buffer->ptr, data, length);
      buffer->ptr += length;
      break;
    }
  default:
    break;
  }
}

/* Stops a note playing on a MIDI channel using requested velocity */
void midi_note_off(struct midi *midi, unsigned char channel, unsigned char note, unsigned char velocity)
{
  nullcheck_void(midi, midi_note_off);

  unsigned char data[3];

  data[0] = 0x80 | channel;
  data[1] = note & 0x7f;
  data[2] = velocity & 0x7f;

  midi_write(midi, data, 3);
}

/* Plays a note on a MIDI channel using requested velocity */
void midi_note_on(struct midi *midi, unsigned char channel, unsigned char note, unsigned char velocity)
{
  nullcheck_void(midi, midi_note_on);

  unsigned char data[3];

  data[0] = 0x90 | channel;
  data[1] = note & 0x7f;
  data[2] = velocity & 0x7f;

  midi_write(midi, data, 3);
}

/* Sets the aftertouch pressure of a note playing on a MIDI channel */
void midi_aftertouch(struct midi *midi, unsigned char channel, unsigned char note, unsigned char pressure)
{
  nullcheck_void(midi, midi_aftertouch);

  unsigned char data[3];

  data[0] = 0xa0 | channel;
  data[1] = note & 0x7f;
  data[2] = pressure & 0x7f;

  midi_write(midi, data, 3);
}

/* Sets the MIDI controller value of a MIDI channel */
void midi_controller(struct midi *midi, unsigned char channel, unsigned char controller, unsigned char value)
{
  nullcheck_void(midi, midi_controller);

  unsigned char data[3];

  data[0] = 0xb0 | channel;
  data[1] = controller & 0x7f;
  data[2] = value & 0x7f;

  midi_write(midi, data, 3);
}

/* Send a program change on a MIDI channel */
void midi_program_change(struct midi *midi, unsigned char channel, unsigned char program)
{
  nullcheck_void(midi, midi_program_change);

  unsigned char data[2];

  data[0] = 0xc0 | channel;
  data[1] = program & 0x7f;

  midi_write(midi, data, 2);
}

/* Sets the channel pressure of a MIDI channel */
void midi_channel_pressure(struct midi *midi, unsigned char channel, unsigned char pressure)
{
  nullcheck_void(midi, midi_channel_pressure);

  unsigned char data[2];

  data[0] = 0xd0 | channel;
  data[1] = pressure & 0x7f;

  midi_write(midi, data, 2);
}

/* Sets the pitch wheel value of a MIDI channel */
void midi_pitch_wheel(struct midi *midi, unsigned char channel, unsigned short value)
{
  nullcheck_void(midi, midi_pitch_wheel);

  unsigned char data[3];

  data[0] = 0xe0 | channel;
  data[1] = (value >> 7) & 0x7f;
  data[2] = value & 0x7f;

  midi_write(midi, data, 3);
}

/* Sends an arbitrary MIDI message */
void midi_write_raw(struct midi *midi, unsigned char *message, unsigned short length)
{
  nullcheck_void(midi, midi_write_raw);

  if (length == 0)
    return;

  /* SysEx messages need special treatment if written to a file */
  if (midi->type == MIDI_BUFFER && message[0] == 0xf0) {
    unsigned long value = length - 1;
    unsigned long varlen = value & 0x7F;
    unsigned char *newmessage;
    int i, l;

    /* Create a variable length version of the length */
    while ((value >>= 7)) {
      varlen <<= 8;
      varlen |= ((value & 0x7F) | 0x80);
    }

    /* Check how many bytes the varlen version requires */
    value = varlen;
    for (l = 1; l <= 4; l++) {
      if (value & 0x80)
	value >>= 8;
      else
	break;
    }
    
    /* Allocate buffer */
    newmessage = (unsigned char *)malloc(length + l);
    newmessage[0] = 0xf0;

    /* Write the varlen length */
    for (i = 1; i <= 4; i++) {
      newmessage[i] = (unsigned char)(varlen & 0xff);
      if (varlen & 0x80)
	varlen >>= 8;
      else
	break;
    }

    /* Copy the rest of the data */
    memcpy(newmessage + i + 1, message + 1, length - 1);
    midi_write(midi, newmessage, length + l);
    free(newmessage);
  } else
    midi_write(midi, message, length);
}

/* Send a clock message */
void midi_clock(struct midi *midi)
{
  nullcheck_void(midi, midi_clock);

  unsigned char data[] = { 0xf8 };

  midi_write(midi, data, 1);
}

/* Set the tempo (used when exporting) */
void midi_tempo(struct midi *midi, unsigned int tempo)
{
  nullcheck_void(midi, midi_tempo);
  
  unsigned char data[] = { 0xff, 0x51, 0x03, 0x00, 0x00, 0x00 };
  unsigned int ms = 60000000 / tempo;
  
  /* Tempo is only relevant when exporting */
  if (midi->type != MIDI_BUFFER)
    return;
  
  data[3] = (ms >> 16) & 0xff;
  data[4] = (ms >> 8) & 0xff;
  data[5] = ms & 0xff;
  
  midi_write(midi, data, 6);
}

/* Receives a MIDI message */
unsigned char *midi_read_raw(struct midi *midi)
{
  nullcheck_int(midi, midi_read_raw);

  int status = 1;
  unsigned char c = 0, *d = NULL;

  while ((status = midi_read(midi, &c, 1)) == 1 && c < 0x80);

  if (status != 1)
    return NULL;

  switch (c & 0xf0) {
  case 0x80:
  case 0x90:
  case 0xa0:
  case 0xb0:
  case 0xe0:
    {
      /* Number of bytes read */
      int r = 0;

      d = (unsigned char *)calloc(3, sizeof(unsigned char));
      d[0] = c;

      /* Read until 2 bytes have been read. If for some reason these two bytes will never come Tutka hangs here! */
      while (r < 2) {
	status = midi_read(midi, d + 1 + r, 2 - r);
	if (status >= 0)
	  r += status;
      }

      return d;
    }
  default:
    break;
  }
  return NULL;
}

/* Receives a MIDI message */
void midi_read_system_exclusive(struct midi *midi, struct message *message, int autostop)
{
  nullcheck_void(midi, midi_read_system_exclusive);
  nullcheck_void(message, midi_read_system_exclusive);

  int status = 0, i = 0;
  unsigned char c = 0;

  if (autostop == 1) {
    /* Read until 0xf7 (end of SysEx) */
    c = 0;
    message->length = 0;

    while (c != 0xf7) {
      status = midi_read(midi, &c, 1);

      if (status == 1) {
	if (i >= message->length) {
	  message->length += 65536;
	  message->data = (unsigned char *)realloc(message->data, message->length);
	}

	if (c != 0xf1 && c != 0xf8 && c != 0xf9 && c != 0xfe)
	  message->data[i++] = c;
      }
    }
  } else {
    /* Read until the buffer is full */
    while (i < message->length) {
      status = midi_read(midi, &c, 1);

      if (status == 1) {
	if (c != 0xf1 && c != 0xf8 && c != 0xf9 && c != 0xfe)
	  message->data[i++] = c;
      }
    }
  }

  /* Remove unnecessary bytes if necessary */
  if (i < message->length) {
    message->length = i;
    message->data = (unsigned char *)realloc(message->data, message->length);
  }
}

/* Returns the data written to the MIDI buffer */
unsigned char *midi_get_buffer(struct midi *midi)
{
  nullcheck_pointer(midi, midi_get_buffer);

  unsigned char *data;
  struct midi_buffer *buffer = midi->data;

  if (midi->type != MIDI_BUFFER) {
    fprintf(stderr, "midi_get_buffer() called but MIDI interface not a buffer\n");
    return NULL;
  }

  data = malloc(buffer->ptr - buffer->data);
  memcpy(data, buffer->data, buffer->ptr - buffer->data);

  return data;
}

/* Returns the data written to the MIDI buffer */
int midi_get_buffer_length(struct midi *midi)
{
  nullcheck_int(midi, midi_get_buffer_length);

  struct midi_buffer *buffer = midi->data;

  if (midi->type != MIDI_BUFFER) {
    fprintf(stderr, "midi_get_buffer_length() called but MIDI interface not a buffer\n");
    return 0;
  }

  return buffer->ptr - buffer->data;
}

/* Sets the length of a MIDI message */
void message_set_length(struct message *message, unsigned int length)
{
  nullcheck_void(message, message_set_length);

  message->length = length;
  message->data = (unsigned char *)realloc(message->data, message->length);
}

/* Sets the autosend flag of a MIDI message */
void message_set_autosend(struct message *message, unsigned int autosend)
{
  nullcheck_void(message, message_set_autosend);

  message->autosend = autosend;
}

/* Parses a message element in an XML file */
struct message *message_parse(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur)
{
  struct message *message = NULL;

  if ((!xmlStrcmp(cur->name, "message")) && (cur->ns == ns)) {
    xmlNodePtr temp = cur->xmlChildrenNode;
    char c[3], *prop;
    int i;
    unsigned int d;

    /* Temporary string for hexadecimal parsing */
    c[2] = 0;

    /* Allocate message */
    message = (struct message *)calloc(1, sizeof(struct message));
    message->name = xmlGetProp(cur, "name");
    if (temp != NULL)
      message->length = strlen(temp->content) / 2;
    prop = xmlGetProp(cur, "autosend");
    if (prop != NULL)
      message->autosend = atoi(prop);
    message->data = (unsigned char *)calloc(message->length, sizeof(unsigned char));
    for (i = 0; i < message->length; i++) {
      c[0] = temp->content[i * 2];
      c[1] = temp->content[i * 2 + 1];
      sscanf(c, "%X", &d);
      message->data[i] = d;
    }
  } else if (cur->type != XML_COMMENT_NODE)
    fprintf(stderr, "XML error: expected message, got %s\n", cur->name);

  return message;
}

/* Saves a message to an XML file */
void message_save(struct message *message, int number, xmlNodePtr parent)
{
  nullcheck_void(message, message_save);

  char *c, t[10];
  xmlNodePtr node;
  int i;

  c = (char *)calloc(2 * message->length + 1, sizeof(char));
  for (i = 0; i < message->length; i++)
    sprintf(c + (i * 2), "%.2X", message->data[i]);

  node = xmlNewChild(parent, NULL, "message", c);
  snprintf(t, 10, "%d", number);
  xmlSetProp(node, "number", t);
  xmlSetProp(node, "name", message->name);
  snprintf(t, 10, "%d", message->autosend);
  xmlSetProp(node, "autosend", t);

  xmlAddChild(parent, xmlNewText("\n"));

  free(c);
}

/* Loads a message from a file */
void message_load_binary(struct message *message, char *filename)
{
  nullcheck_void(message, message_load_binary);
  nullcheck_void(filename, message_load_binary);

  unsigned char *data;
  FILE *file;

  file = fopen(filename, "r");

  if (file != NULL) {
    int length;

    fseek(file, 0, SEEK_END);
    length = ftell(file);
    data = (unsigned char *)calloc(length, sizeof(unsigned char));
    fseek(file, 0, SEEK_SET);
    fread(data, sizeof(unsigned char), length, file);
    fclose(file);

    if (message->data != NULL)
      free(message->data);

    message->data = data;
    message->length = length;
  }
}

/* Saves a message to a file */
void message_save_binary(struct message *message, char *filename)
{
  nullcheck_void(message, message_save_binary);
  nullcheck_void(filename, message_save_binary);

  FILE *file;

  file = fopen(filename, "w");

  if (file != NULL) {
    fwrite(message->data, sizeof(unsigned char), message->length, file);
    fclose(file);
  }
}
